<?php

namespace App\Classes;

use App\Helpers\Files;
use App\Models\Currency;
use App\Models\Lang;
use App\Models\Settings;
use App\Models\SubscriptionPlan;
use App\Models\StaffMember;
use App\Scopes\CompanyScope;
use Carbon\Carbon;
use Carbon\CarbonTimeZone;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Nwidart\Modules\Facades\Module;
use Vinkla\Hashids\Facades\Hashids;

class Common
{
    public static function getFolderPath($type = null)
    {
        $paths = [
            'companyLogoPath' => 'companies',
            'userImagePath' => 'user',
            'langImagePath' => 'langs',
            'pdfFontFilePath' => 'fonts',
            'test-uploads' => 'test-uploads',
            'doorImageFilePath' => 'doors',
            'laundryWebsiteBuilderPath' => 'laundry/website-builder',
        ];

        return ($type == null) ? $paths : $paths[$type];
    }

    public static function uploadFile($request)
    {
        $folder = $request->folder;
        $folderString = "";

        if ($folder == "user") {
            $folderString = "userImagePath";
        } else if ($folder == "company") {
            $folderString = "companyLogoPath";
        } else if ($folder == "langs") {
            $folderString = "langImagePath";
        } else if ($folder == "fonts") {
            $folderString = "pdfFontFilePath";
        } else if ($folder == "test-uploads") {
            $folderString = "test-uploads";
        } else if ($folder == "doors") {
            $folderString = "doorImageFilePath";
        }
        $folderPath = self::getFolderPath($folderString);

        if ($request->hasFile('image') || $request->hasFile('file')) {
            $uploadedFile  = $request->hasFile('image') ? $request->file('image') : $request->file('file');

            // Storage is already configured by StorageSettingsProvider
            // Use the default disk (either 's3' or 'local' based on settings)
            try {
                $fileName = self::uploadFileUsingDriver($uploadedFile, $folderPath);
            } catch (\Exception $e) {
                throw new \Exception('Failed to upload file: ' . $e->getMessage());
            }
        }

        return [
            'file' => $fileName,
            'file_url' => self::getFileUrl($folderPath, $fileName),
        ];
    }

    public static function uploadFileUsingDriver($uploadedFile, $folderPath, $fileName = null)
    {
        // Through exception if malicious file uploaded
        Files::validateUploadedFile($uploadedFile);

        // Generate secrue filename if not supplied
        if (empty($fileName)) {
            $fileName = Files::generateNewFileName(
                $uploadedFile->getClientOriginalName()
            );
        }

        $defaultDisk = config('filesystems.default');

        if ($defaultDisk == 's3') {
            // For S3, use storeAs without public visibility (no ACL)
            // Public access is controlled by bucket policy or temporary URLs
            $uploadedFile->storeAs($folderPath, $fileName, 's3');
        } else {
            // For local storage, store in storage/app/public (not public folder)
            // This keeps files secure and accessible only through Laravel
            $uploadedFile->storeAs($folderPath, $fileName, 'public');
        }

        return $fileName;
    }

    public static function deleteFileUsingDriver($fileName, $folderPath)
    {
        if (empty($fileName)) {
            return false;
        }

        $filePath = $folderPath . '/' . $fileName;
        $defaultDisk = config('filesystems.default');

        if ($defaultDisk == 's3') {
            // For S3, delete from S3 bucket
            return Storage::disk('s3')->delete($filePath);
        } else {
            // For local storage, delete from storage/app/public
            return Storage::disk('public')->delete($filePath);
        }
    }

    public static function checkFileExists($folderString, $fileName)
    {
        $folderPath = self::getFolderPath($folderString);
        $fullPath = $folderPath . '/' . $fileName;

        // Storage is already configured by StorageSettingsProvider
        // Use the default disk
        return Storage::exists($fullPath);
    }

    public static function getFileUrl($folderPath, $fileName, $pathType = 'asset')
    {
        // Storage is already configured by StorageSettingsProvider
        $defaultDisk = config('filesystems.default');
        $path = $folderPath . '/' . $fileName;

        if ($defaultDisk == 's3') {
            // For S3, use temporary URLs with caching
            $cacheKey = 's3-' . $path;
            $tempUrlExpiry = 60; // Minutes

            // Check if the URL is already cached
            if (Cache::has($cacheKey)) {
                return Cache::get($cacheKey);
            }

            // Generate a new temporary URL and cache it
            $temporaryUrl = Storage::disk('s3')->temporaryUrl($path, now()->addMinutes($tempUrlExpiry));
            Cache::put($cacheKey, $temporaryUrl, $tempUrlExpiry * 60);

            return $temporaryUrl;
        } else {
            // For local storage, use storage URL (points to storage/app/public via symlink)
            // This works after running: php artisan storage:link
            if ($pathType == 'public') {
                return Storage::disk('public')->path($path);
            }

            // Generate URL using current request URL instead of APP_URL
            // This ensures correct URL whether using localhost, Herd, or production
            return url('storage/' . $path);
        }
    }

    public static function moduleInformations()
    {
        $allModules = Module::all();
        $allEnabledModules = Module::allEnabled();
        $installedModules = [];
        $enabledModules = [];

        foreach ($allModules as $key => $allModule) {
            $modulePath = $allModule->getPath();
            $versionFileName = app_type() == 'saas' ? 'superadmin_version.txt' : 'version.txt';
            $version = File::get($modulePath . '/' . $versionFileName);

            $installedModules[] = [
                'verified_name' => $key,
                'name' => $allModule->getName(),
                'current_version' => preg_replace("/\r|\n/", "", $version)
            ];
        }

        foreach ($allEnabledModules as $allEnabledModuleKey => $allEnabledModule) {
            $enabledModules[] = $allEnabledModuleKey;
        }

        return [
            'installed_modules' => $installedModules,
            'enabled_modules' => $enabledModules,
        ];
    }

    public static function getIdFromHash($hash)
    {
        if ($hash != "") {
            $convertedId = Hashids::decode($hash);
            $id = $convertedId[0];

            return $id;
        }

        return $hash;
    }

    public static function getHashFromId($id)
    {
        $id = Hashids::encode($id);

        return $id;
    }

    public static function formatSizeUnits($bytes)
    {
        if ($bytes >= 1073741824) {
            $bytes = number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            $bytes = number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            $bytes = number_format($bytes / 1024, 2) . ' KB';
        } elseif ($bytes > 1) {
            $bytes = $bytes . ' bytes';
        } elseif ($bytes == 1) {
            $bytes = $bytes . ' byte';
        } else {
            $bytes = '0 bytes';
        }

        return $bytes;
    }

    public static function formatDate($date = null, $company)
    {
        $timezone = new CarbonTimeZone($company->timezone);
        $format = $company->date_format;

        return $date
            ? Carbon::parse($date)->setTimezone($timezone)->format('d-m-Y')
            : Carbon::now($timezone)->format('d-m-Y');
    }

    public static function formatAmountCurrency($currency, $amount)
    {
        if (!isset($currency['symbol']) || !isset($currency['position'])) {
            return $amount;
        }

        $symbol = $currency['symbol'];
        $position = $currency['position'];

        $formattedAmount = number_format(abs($amount), 2, '.', ',');

        $formattedAmount = $position === 'front' ? $symbol . $formattedAmount : $formattedAmount . $symbol;

        return $amount < 0 ? "-{$formattedAmount}" : $formattedAmount;
    }

    public static function calculateTotalUsers($companyId, $update = false)
    {
        $totalUsers =  StaffMember::withoutGlobalScope(CompanyScope::class)
            ->where('company_id', $companyId)
            ->count('id');

        if ($update) {
            DB::table('companies')
                ->where('id', $companyId)
                ->update([
                    'total_users' => $totalUsers
                ]);
        }

        return $totalUsers;
    }

    public static function convertToCollection($data)
    {
        $data = collect($data)->map(function ($item) {
            return (object) $item;
        });

        return $data;
    }

    public static function addCurrencies($company)
    {
        $newCurrency = new Currency();
        $newCurrency->company_id = $company->id;
        $newCurrency->name = 'Dollar';
        $newCurrency->code = 'USD';
        $newCurrency->symbol = '$';
        $newCurrency->position = 'front';
        $newCurrency->is_deletable = false;
        $newCurrency->save();

        $rupeeCurrency = new Currency();
        $rupeeCurrency->company_id = $company->id;
        $rupeeCurrency->name = 'Rupee';
        $rupeeCurrency->code = 'INR';
        $rupeeCurrency->symbol = '₹';
        $rupeeCurrency->position = 'front';
        $rupeeCurrency->is_deletable = false;
        $rupeeCurrency->save();

        $enLang = Lang::where('key', 'en')->first();

        $company->currency_id = $newCurrency->id;
        $company->lang_id = $enLang->id;
        $company->save();

        return $company;
    }

    public static function checkSubscriptionModuleVisibility($itemType)
    {
        $visible = true;

        if (app_type() == 'saas') {
            if ($itemType == 'user') {
                $userCounts = StaffMember::count();
                $company = company();

                $visible = $company && $company->subscriptionPlan && $userCounts < $company->subscriptionPlan->max_users ? true : false;
            }
        }

        return $visible;
    }

    public static function allVisibleSubscriptionModules()
    {
        $visibleSubscriptionModules = [];

        if (self::checkSubscriptionModuleVisibility('user')) {
            $visibleSubscriptionModules[] = 'user';
        }

        return $visibleSubscriptionModules;
    }

    public static function insertInitSettings($company)
    {
        if ((app_type() == 'saas' && $company->is_global == 1) || (app_type() == 'non-saas' && $company->is_global == 0)) {
            $storageLocalSettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'storage')
                ->where('name_key', 'local')
                ->where('is_global', $company->is_global)
                ->where('company_id', $company->id)
                ->count();
            if ($storageLocalSettingCount == 0) {
                $local = new Settings();
                $local->company_id = $company->id;
                $local->setting_type = 'storage';
                $local->name = 'Local';
                $local->name_key = 'local';
                $local->status = true;
                $local->is_global = $company->is_global;
                $local->save();
            }

            $storageAwsSettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'storage')
                ->where('name_key', 'aws')
                ->where('is_global', $company->is_global)
                ->where('company_id', $company->id)
                ->count();
            if ($storageAwsSettingCount == 0) {
                $aws = new Settings();
                $aws->company_id = $company->id;
                $aws->setting_type = 'storage';
                $aws->name = 'AWS';
                $aws->name_key = 'aws';
                $aws->credentials = [
                    'driver' => 's3',
                    'key' => '',
                    'secret' => '',
                    'region' => '',
                    'bucket' => '',

                ];
                $aws->is_global = $company->is_global;
                $aws->save();
            }

            $smtpEmailSettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'email')
                ->where('name_key', 'smtp')
                ->where('is_global', $company->is_global)
                ->where('company_id', $company->id)
                ->count();
            if ($smtpEmailSettingCount == 0) {
                $smtp = new Settings();
                $smtp->company_id = $company->id;
                $smtp->setting_type = 'email';
                $smtp->name = 'SMTP';
                $smtp->name_key = 'smtp';
                $smtp->credentials = [
                    'from_name' => '',
                    'from_email' => '',
                    'host' => '',
                    'port' => '',
                    'encryption' => '',
                    'username' => '',
                    'password' => '',

                ];
                $smtp->is_global = $company->is_global;
                $smtp->save();
            }
        }

        if ($company->is_global == 0) {

            $sendMailSettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'send_mail_settings')
                ->where('name_key', 'company')
                ->where('is_global', 0)
                ->where('company_id', $company->id)
                ->count();
            if ($sendMailSettingCount == 0) {
                $sendMailSettings = new Settings();
                $sendMailSettings->company_id = $company->id;
                $sendMailSettings->setting_type = 'send_mail_settings';
                $sendMailSettings->name = 'Send mail to company';
                $sendMailSettings->name_key = 'company';
                $sendMailSettings->credentials = [];
                $sendMailSettings->save();
            }

            $shortcutMenuSettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'shortcut_menus')
                ->where('name_key', 'shortcut_menus')
                ->where('is_global', 0)
                ->where('company_id', $company->id)
                ->count();
            if ($shortcutMenuSettingCount == 0) {
                // Create Menu Setting
                $setting = new Settings();
                $setting->company_id = $company->id;
                $setting->setting_type = 'shortcut_menus';
                $setting->name = 'Add Menu';
                $setting->name_key = 'shortcut_menus';
                $setting->credentials = [
                    'language',
                    'role',
                ];
                $setting->status = 1;
                $setting->save();
            }

            // Seed for email templates
            NotificationSeed::seedMainNotifications($company->id);

            // Initialize AI settings
            $aiSettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'ai')
                ->where('name_key', 'openai')
                ->where('is_global', 0)
                ->where('company_id', $company->id)
                ->count();
            if ($aiSettingCount == 0) {
                $aiSetting = new Settings();
                $aiSetting->company_id = $company->id;
                $aiSetting->setting_type = 'ai';
                $aiSetting->name = 'OpenAI';
                $aiSetting->name_key = 'openai';
                $aiSetting->credentials = [
                    'api_key' => '',
                    'model' => 'gpt-4o-mini',
                    'max_tokens' => 2000,
                ];
                $aiSetting->save();
            }

            $claudeSettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'ai')
                ->where('name_key', 'claude')
                ->where('is_global', 0)
                ->where('company_id', $company->id)
                ->count();
            if ($claudeSettingCount == 0) {
                $claudeSetting = new Settings();
                $claudeSetting->company_id = $company->id;
                $claudeSetting->setting_type = 'ai';
                $claudeSetting->name = 'Claude';
                $claudeSetting->name_key = 'claude';
                $claudeSetting->credentials = [
                    'api_key' => '',
                    'model' => 'claude-sonnet-4-20250514',
                    'max_tokens' => 2000,
                ];
                $claudeSetting->save();
            }

            // Initialize Perplexity AI settings
            $perplexitySettingCount = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', 'ai')
                ->where('name_key', 'perplexity')
                ->where('is_global', 0)
                ->where('company_id', $company->id)
                ->count();
            if ($perplexitySettingCount == 0) {
                $perplexitySetting = new Settings();
                $perplexitySetting->company_id = $company->id;
                $perplexitySetting->setting_type = 'ai';
                $perplexitySetting->name = 'Perplexity';
                $perplexitySetting->name_key = 'perplexity';
                $perplexitySetting->credentials = [
                    'api_key' => '',
                    'model' => 'sonar',
                    'max_tokens' => 2000,
                ];
                $perplexitySetting->save();
            }
        }
    }

    public static function assignCompanyForNonSaas($company)
    {
        $adminUser = StaffMember::first();
        $company->admin_id = $adminUser->id;
        // Setting Trial Plan
        if (app_type() == 'saas') {
            $trialPlan = SubscriptionPlan::where('default', 'trial')->first();
            if ($trialPlan) {
                $company->subscription_plan_id = $trialPlan->id;
                // set company license expire date
                $company->licence_expire_on = Carbon::now()->addDays($trialPlan->duration)->format('Y-m-d');
            }
        }
        $company->save();

        // Insert records in settings table
        // For inital settings like email, storage
        Common::insertInitSettings($company);
    }

    public static function lightenHexColor($hex, $percent)
    {
        // Remove the hash symbol if it exists
        $hex = ltrim($hex, '#');

        // Convert the hex code to RGB
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        // Calculate the new RGB values
        $r = min(255, round($r + (255 - $r) * ($percent / 100)));
        $g = min(255, round($g + (255 - $g) * ($percent / 100)));
        $b = min(255, round($b + (255 - $b) * ($percent / 100)));

        // Convert the RGB back to hex
        $lightenedHex = sprintf("#%02x%02x%02x", $r, $g, $b);

        return $lightenedHex;
    }

    public static function getIdArrayFromHash($values)
    {
        $convertedArray = [];

        foreach ($values as $value) {
            $convertedArray[] = Common::getIdFromHash($value);
        }

        return $convertedArray;
    }
}
