<?php

namespace App\Classes;

use Illuminate\Support\Facades\File;

class ModuleAssets
{
    /**
     * Get all Vite entry assets including enabled module assets
     *
     * @return array
     */
    public static function getViteAssets(): array
    {
        $paths = [
            'resources/js/app.js',
            'resources/less/custom.less',
        ];

        // Check if Vite dev server is running by looking for .hot file
        $hotFilePath = public_path('hot');
        if (File::exists($hotFilePath)) {
            $modulesPath = base_path('Modules');
            $moduleStatusesPath = base_path('modules_statuses.json');

            // Check if modules_statuses.json exists
            if (!File::exists($moduleStatusesPath)) {
                return $paths;
            }

            // Read module statuses
            $moduleStatuses = json_decode(File::get($moduleStatusesPath), true);

            if (!is_array($moduleStatuses)) {
                return $paths;
            }

            // Check if Modules directory exists
            if (!File::isDirectory($modulesPath)) {
                return $paths;
            }

            // Get all module directories
            $moduleDirectories = File::directories($modulesPath);

            foreach ($moduleDirectories as $moduleDir) {
                $moduleName = basename($moduleDir);

                // Skip if module is not enabled
                if (!isset($moduleStatuses[$moduleName]) || $moduleStatuses[$moduleName] !== true) {
                    continue;
                }

                $viteConfigPath = $moduleDir . '/vite.config.js';

                // Check if vite.config.js exists for this module
                if (!File::exists($viteConfigPath)) {
                    continue;
                }

                $paths[] = "Modules/{$moduleName}/resources/assets/js/app.js";

                // // Read vite config file
                // $viteConfigContent = File::get($viteConfigPath);

                // // Extract paths array from the config using regex
                // if (preg_match('/export\s+const\s+paths\s*=\s*\[([\s\S]*?)\];/m', $viteConfigContent, $matches)) {
                //     // Parse the paths array
                //     $pathsString = $matches[1];

                //     // Extract all quoted strings (both single and double quotes)
                //     preg_match_all('/[\'"]([^\'"]+)[\'"]/m', $pathsString, $pathMatches);

                //     if (!empty($pathMatches[1])) {
                //         foreach ($pathMatches[1] as $path) {
                //             $paths[] = trim($path);
                //         }
                //     }
                // }
            }
        }

        return $paths;
    }

    /**
     * Get module asset paths for a specific module
     *
     * @param string $moduleName
     * @return array
     */
    public static function getModuleAssets(string $moduleName): array
    {
        $paths = [];
        $modulePath = base_path("Modules/{$moduleName}");
        $viteConfigPath = $modulePath . '/vite.config.js';

        if (!File::exists($viteConfigPath)) {
            return $paths;
        }

        $viteConfigContent = File::get($viteConfigPath);

        // Extract paths array from the config using regex
        if (preg_match('/export\s+const\s+paths\s*=\s*\[([\s\S]*?)\];/m', $viteConfigContent, $matches)) {
            // Parse the paths array
            $pathsString = $matches[1];

            // Extract all quoted strings (both single and double quotes)
            preg_match_all('/[\'"]([^\'"]+)[\'"]/m', $pathsString, $pathMatches);

            if (!empty($pathMatches[1])) {
                foreach ($pathMatches[1] as $path) {
                    $trimmedPath = trim($path);

                    // Remove "Modules/{ModuleName}/" prefix if present
                    $prefix = "Modules/{$moduleName}/";
                    if (strpos($trimmedPath, $prefix) === 0) {
                        $trimmedPath = substr($trimmedPath, strlen($prefix));
                    }

                    $paths[] = $trimmedPath;
                }
            }
        }

        return $paths;
    }

    /**
     * Check if a module is enabled
     *
     * @param string $moduleName
     * @return bool
     */
    public static function isModuleEnabled(string $moduleName): bool
    {
        $moduleStatusesPath = base_path('modules_statuses.json');

        if (!File::exists($moduleStatusesPath)) {
            return false;
        }

        $moduleStatuses = json_decode(File::get($moduleStatusesPath), true);

        return isset($moduleStatuses[$moduleName]) && $moduleStatuses[$moduleName] === true;
    }

    /**
     * Get all active module folder names from filesystem
     *
     * @return array
     */
    public static function getActiveModuleFolders(): array
    {
        $modulesPath = base_path('Modules');
        $statusFilePath = base_path('modules_statuses.json');

        if (!File::isDirectory($modulesPath)) {
            return [];
        }

        // Read modules_statuses.json
        $moduleStatuses = [];
        if (File::exists($statusFilePath)) {
            $moduleStatuses = json_decode(File::get($statusFilePath), true) ?? [];
        }

        $activeModules = [];

        // Scan Modules directory
        $folders = File::directories($modulesPath);

        foreach ($folders as $folder) {
            $folderName = basename($folder);
            $moduleJsonPath = $folder . '/module.json';

            // Check if module.json exists
            if (!File::exists($moduleJsonPath)) {
                continue;
            }

            // Read module.json to get the module name
            $moduleJson = json_decode(File::get($moduleJsonPath), true);
            $moduleName = $moduleJson['name'] ?? $folderName;

            // Check if module is enabled in modules_statuses.json
            if (isset($moduleStatuses[$moduleName]) && $moduleStatuses[$moduleName] === true) {
                $activeModules[] = $folderName;
            }
        }

        return $activeModules;
    }

    /**
     * Get build directory for a module from its vite.config.js
     *
     * @param string $moduleName
     * @return string|null
     */
    public static function getModuleBuildDirectory(string $moduleName): ?string
    {
        $modulePath = base_path("Modules/{$moduleName}");
        $viteConfigPath = $modulePath . '/vite.config.js';

        if (!File::exists($viteConfigPath)) {
            return null;
        }

        $viteConfigContent = File::get($viteConfigPath);

        // Extract buildDirectory from laravel plugin config
        if (preg_match('/buildDirectory:\s*[\'"]([^\'"]+)[\'"]/m', $viteConfigContent, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    /**
     * Get all enabled modules with their metadata
     *
     * @return array Array of modules with name, buildDirectory, and assets
     */
    public static function getEnabledModules(): array
    {
        $modules = [];
        $hotFilePath = public_path('hot');

        if (!File::exists($hotFilePath)) {
            $modulesPath = base_path('Modules');
            $moduleStatusesPath = base_path('modules_statuses.json');

            if (!File::exists($moduleStatusesPath) || !File::isDirectory($modulesPath)) {
                return $modules;
            }

            $moduleStatuses = json_decode(File::get($moduleStatusesPath), true);

            if (!is_array($moduleStatuses)) {
                return $modules;
            }

            $moduleDirectories = File::directories($modulesPath);

            foreach ($moduleDirectories as $moduleDir) {
                $moduleName = basename($moduleDir);

                if (!isset($moduleStatuses[$moduleName]) || $moduleStatuses[$moduleName] !== true) {
                    continue;
                }

                $buildDirectory = self::getModuleBuildDirectory($moduleName);
                $assets = self::getModuleAssets($moduleName);

                if ($buildDirectory && !empty($assets)) {
                    $modules[] = [
                        'name' => $moduleName,
                        'buildDirectory' => $buildDirectory,
                        'assets' => $assets,
                    ];
                }
            }
        }

        return $modules;
    }
}
