<?php

namespace App\Classes;

use App\Models\Settings;
use App\Scopes\CompanyScope;
use Nwidart\Modules\Facades\Module;

class NotificationSeed
{
    public static $emailNotificaiotnArray = [

        // Users
        'user_welcome_mail' => [
            'type' => "email_templates",
            'title' => "Welcome to the Company!",
            'content' => <<<HTML
<p>Hello <strong>##EMPLOYEE_NAME##</strong>,</p>
<p>We are excited to welcome you to our team at <strong>##COMPANY_NAME##!</strong> We believe your skills and experience will be a valuable addition to our company.</p>

<p>Below is the login details to company account:</p>
<ul>
    <li><strong>Email:</strong> ##EMPLOYEE_EMAIL##</li>
    <li><strong>Password:</strong> ##EMPLOYEE_PASSWORD##</li>
</ul>

<p>To get started, please log in to your employee portal using the button below:</p>
<p><a href="##LOGIN_URL##" rel="noopener noreferrer" target="_blank">Login to Portal</a> If you have any questions, feel free to reach out to HR.</p>

<p>Best regards,<br>
HR Team</p>
HTML,
            'other_data' => []
        ],
    ];

    /**
     * Common function to insert notifications into settings table
     *
     * @param array $notifications Array of notifications to seed
     * @param int $companyId Company ID
     * @param string|null $moduleName Module name (null for main app)
     * @return void
     */
    protected static function insertNotifications(array $notifications, $companyId, $moduleName = null)
    {
        foreach ($notifications as $key => $notification) {
            $query = Settings::withoutGlobalScope(CompanyScope::class)
                ->where('setting_type', $notification['type'])
                ->where('name_key', $key)
                ->where('company_id', $companyId);

            if ($moduleName) {
                $query->where('module_name', $moduleName);
            } else {
                $query->whereNull('module_name');
            }

            $notificationCount = $query->count();

            if ($notificationCount == 0) {
                $newNotification = new Settings();
                $newNotification->company_id = $companyId;
                $newNotification->setting_type = $notification['type'];
                $newNotification->name = $notification['title'];
                $newNotification->name_key = $key;
                $newNotification->module_name = $moduleName;
                $newNotification->credentials = [
                    'title' => $notification['title'],
                    'content' => $notification['content'],
                ];
                $newNotification->save();
            }
        }
    }

    /**
     * Seed notifications for main application (without module)
     *
     * @param int $companyId Company ID
     * @return void
     */
    public static function seedMainNotifications($companyId)
    {
        self::insertNotifications(self::$emailNotificaiotnArray, $companyId, null);
    }

    /**
     * Seed notifications for a specific module
     *
     * @param int $companyId Company ID
     * @param string $moduleName Module name
     * @return void
     */
    public static function seedModuleNotifications($companyId, $moduleName)
    {
        $notifications = [];

        // Fetch $emailNotificaiotnArray from module NotificationSeed
        // file if exists from path base_path("Modules/{$moduleName}/app/Classes")
        $moduleNotificationSeedFile = base_path("Modules/{$moduleName}/app/Classes/NotificationSeed.php");

        if (file_exists($moduleNotificationSeedFile)) {
            $moduleNotificationSeedClass = "Modules\\{$moduleName}\\app\\Classes\\NotificationSeed";

            if (class_exists($moduleNotificationSeedClass) && property_exists($moduleNotificationSeedClass, 'emailNotificaiotnArray')) {
                $notifications = $moduleNotificationSeedClass::$emailNotificaiotnArray;
            }
        }

        self::insertNotifications($notifications, $companyId, $moduleName);
    }

    /**
     * Seed notifications for main application and all active modules
     *
     * @param int $companyId Company ID
     * @return void
     */
    public static function seedAllNotifications($companyId)
    {
        // Seed main application notifications
        self::seedMainNotifications($companyId);

        // Seed all active modules notifications
        $activeModules = Module::allEnabled();

        foreach ($activeModules as $module) {
            self::seedModuleNotifications($companyId, $module->getName());
        }
    }
}
