<?php

namespace App\Http\Controllers\Api\Common;

use App\Http\Controllers\ApiBaseController;
use App\Http\Requests\Api\AiSettings\UpdateRequest;
use App\Http\Requests\Api\AiSettings\TestConnectionRequest;
use App\Models\Settings;
use App\Models\Website;
use App\Models\WebsitePage;
use App\Models\WebsitePageTranslation;
use App\Models\WebsiteLanguage;
use App\Models\WebsiteLegalPage;
use App\Models\WebsiteBuilder;
use App\Models\ThemeSection;
use Examyou\RestAPI\ApiResponse;
use Examyou\RestAPI\Exceptions\ApiException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class AiSettingsController extends ApiBaseController
{
    /**
     * Check if user has AI features access permission
     */
    protected function checkAiFeaturesAccess()
    {
        $user = auth('api')->user();
        if (!$user || !$user->ability('admin', 'ai_features_access')) {
            throw new ApiException('Unauthorized: AI features access required.', null, 403, 403);
        }
    }

    /**
     * OpenAI models available for selection
     */
    public static $openAiModels = [
        'gpt-5.2' => 'GPT-5.2 (Latest, Best for Coding & Agents)',
        'gpt-5-mini' => 'GPT-5 Mini (Fast, Cost-Effective)',
        'gpt-5.2-pro' => 'GPT-5.2 Pro (Precise Responses)',
        'gpt-4.1' => 'GPT-4.1 (Smart Non-Reasoning)',
        'gpt-5-nano' => 'GPT-5 Nano (Fastest, Budget)',
    ];

    /**
     * Claude models available for selection
     */
    public static $claudeModels = [
        'claude-opus-4-5-20251101' => 'Claude Opus 4.5 (Most Capable)',
        'claude-sonnet-4-5-20250929' => 'Claude Sonnet 4.5 (Recommended)',
        'claude-haiku-4-5-20251001' => 'Claude Haiku 4.5 (Fastest)',
        'claude-3-5-sonnet-20241022' => 'Claude 3.5 Sonnet',
        'claude-3-5-haiku-20241022' => 'Claude 3.5 Haiku',
    ];

    /**
     * Perplexity models available for selection
     */
    public static $perplexityModels = [
        'sonar-pro' => 'Sonar Pro (Best Quality)',
        'sonar' => 'Sonar (Balanced)',
        'sonar-reasoning-pro' => 'Sonar Reasoning Pro (Complex Tasks)',
        'sonar-reasoning' => 'Sonar Reasoning (Analysis)',
    ];

    /**
     * Ollama models (common ones - user can enter custom model names)
     */
    public static $ollamaModels = [
        'qwen2.5:7b' => 'Qwen 2.5',
        'llama3.2' => 'Llama 3.2 (Latest)',
        'llama3.1' => 'Llama 3.1',
        'mistral' => 'Mistral',
        'codellama' => 'Code Llama',
        'phi3' => 'Phi-3',
        'gemma2' => 'Gemma 2',
        'qwen2.5' => 'Qwen 2.5',
    ];

    /**
     * OpenAI-compatible providers models (Groq, Together AI, OpenRouter, etc.)
     * Note: Model availability depends on the provider you're using
     */
    public static $openaiCompatibleModels = [
        'gpt-4o' => 'GPT-4o (OpenRouter)',
        'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
        'llama-3.1-70b-versatile' => 'Llama 3.1 70B (Groq)',
        'mixtral-8x7b-32768' => 'Mixtral 8x7B (Groq)',
        'meta-llama/Meta-Llama-3.1-70B-Instruct-Turbo' => 'Llama 3.1 70B (Together)',
    ];

    /**
     * Check authorization for AI settings
     *
     * @throws ApiException
     */
    protected function authorizeAiAccess()
    {
        $user = auth('api')->user();

        if (app_type() == 'saas') {
            if (!$user || !$user->is_superadmin) {
                throw new ApiException('UNAUTHORIZED EXCEPTION', null, 401, 401);
            }
        } else {
            if (!$user || !$user->ability('admin', 'ai_settings_edit')) {
                throw new ApiException('UNAUTHORIZED EXCEPTION', null, 401, 401);
            }
        }
    }

    /**
     * Check if AI setting is properly configured based on provider type
     *
     * @param Settings|null $aiSetting
     * @return bool
     */
    protected function isAiConfigured($aiSetting)
    {
        if (!$aiSetting) {
            return false;
        }

        $provider = $aiSetting->name_key;

        if ($provider === 'ollama') {
            // Ollama needs base_url and model (API key is optional)
            return !empty($aiSetting->credentials['base_url']) && !empty($aiSetting->credentials['model']);
        } elseif ($provider === 'openai_compatible') {
            // OpenAI-compatible needs api_key and base_url
            return !empty($aiSetting->credentials['api_key']) && !empty($aiSetting->credentials['base_url']);
        } else {
            // Other providers need api_key
            return !empty($aiSetting->credentials['api_key']);
        }
    }

    /**
     * Get AI settings
     */
    public function getAiSettings()
    {
        $this->authorizeAiAccess();

        $aiSettings = Settings::where('setting_type', 'ai')->get();

        $settingsData = [];
        $activeProvider = null;

        foreach ($aiSettings as $setting) {
            $settingsData[$setting->name_key] = [
                'name' => $setting->name,
                'api_key' => $setting->credentials['api_key'] ?? '',
                'model' => $setting->credentials['model'] ?? '',
                'max_tokens' => $setting->credentials['max_tokens'] ?? 2000,
                'base_url' => $setting->credentials['base_url'] ?? '',
                'status' => $setting->status,
            ];

            if ($setting->status) {
                $activeProvider = $setting->name_key;
            }
        }

        return ApiResponse::make(
            'Success',
            [
                'settings' => $settingsData,
                'active_provider' => $activeProvider,
                'openai_models' => self::$openAiModels,
                'claude_models' => self::$claudeModels,
                'perplexity_models' => self::$perplexityModels,
                'ollama_models' => self::$ollamaModels,
                'openai_compatible_models' => self::$openaiCompatibleModels,
                'is_local_env' => env('APP_ENV') === 'local',
            ]
        );
    }

    /**
     * Update AI settings
     */
    public function updateAiSettings(UpdateRequest $request)
    {
        // Not Allowed in Demo Mode
        if (env('APP_ENV') == 'production') {
            throw new ApiException('Not Allowed In Demo Mode');
        }

        // Request is already validated by UpdateRequest
        $provider = $request->provider;

        // Disable all AI settings first
        Settings::where('setting_type', 'ai')->update(['status' => 0]);

        // If provider is 'none', just disable all settings and return
        if ($provider === 'none') {
            return ApiResponse::make('AI Settings disabled successfully', []);
        }

        $apiKey = $request->api_key;
        $model = $request->model;
        $maxTokens = $request->max_tokens ?? 2000;
        $baseUrl = $request->base_url ?? '';

        // Check if setting exists, create if not
        $setting = Settings::where('setting_type', 'ai')
            ->where('name_key', $provider)
            ->first();

        if (!$setting) {
            // Create new setting for this provider
            $setting = new Settings();
            $setting->company_id = company()->id;
            $setting->setting_type = 'ai';
            $setting->name = ucfirst(str_replace('_', ' ', $provider));
            $setting->name_key = $provider;
        }

        $setting->credentials = [
            'api_key' => $apiKey,
            'model' => $model,
            'max_tokens' => $maxTokens,
            'base_url' => $baseUrl,
        ];

        // For providers like Ollama that don't need API key, check model or base_url
        if (in_array($provider, ['ollama'])) {
            $setting->status = !empty($model) && !empty($baseUrl) ? 1 : 0;
        } elseif (in_array($provider, ['openai_compatible'])) {
            $setting->status = !empty($apiKey) && !empty($baseUrl) ? 1 : 0;
        } else {
            $setting->status = !empty($apiKey) ? 1 : 0;
        }

        $setting->save();

        return ApiResponse::make('AI Settings updated successfully', []);
    }

    /**
     * Test AI connection
     */
    public function testConnection(TestConnectionRequest $request)
    {
        // Request is already validated by TestConnectionRequest
        $provider = $request->provider;
        $apiKey = $request->api_key;
        $model = $request->model;
        $baseUrl = $request->base_url ?? '';

        try {
            if ($provider === 'openai') {
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ])->post('https://api.openai.com/v1/chat/completions', [
                    'model' => $model,
                    'messages' => [
                        ['role' => 'user', 'content' => 'Say "Connection successful" in exactly those words.']
                    ],
                    'max_tokens' => 20,
                ]);

                if ($response->successful()) {
                    return ApiResponse::make('Success', [
                        'status' => 'success',
                        'message' => 'Connection successful! OpenAI is configured correctly.',
                    ]);
                } else {
                    $error = $response->json()['error']['message'] ?? 'Unknown error';
                    return ApiResponse::make('Error', [
                        'status' => 'error',
                        'message' => 'Connection failed: ' . $error,
                    ]);
                }
            } elseif ($provider === 'claude') {
                $response = Http::withHeaders([
                    'x-api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                    'anthropic-version' => '2023-06-01',
                ])->post('https://api.anthropic.com/v1/messages', [
                    'model' => $model,
                    'max_tokens' => 20,
                    'messages' => [
                        ['role' => 'user', 'content' => 'Say "Connection successful" in exactly those words.']
                    ],
                ]);

                if ($response->successful()) {
                    return ApiResponse::make('Success', [
                        'status' => 'success',
                        'message' => 'Connection successful! Claude is configured correctly.',
                    ]);
                } else {
                    $error = $response->json()['error']['message'] ?? 'Unknown error';
                    return ApiResponse::make('Error', [
                        'status' => 'error',
                        'message' => 'Connection failed: ' . $error,
                    ]);
                }
            } elseif ($provider === 'perplexity') {
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ])->post('https://api.perplexity.ai/chat/completions', [
                    'model' => $model,
                    'messages' => [
                        ['role' => 'user', 'content' => 'Say "Connection successful" in exactly those words.']
                    ],
                    'max_tokens' => 20,
                ]);

                if ($response->successful()) {
                    return ApiResponse::make('Success', [
                        'status' => 'success',
                        'message' => 'Connection successful! Perplexity is configured correctly.',
                    ]);
                } else {
                    $error = $response->json()['error']['message'] ?? 'Unknown error';
                    return ApiResponse::make('Error', [
                        'status' => 'error',
                        'message' => 'Connection failed: ' . $error,
                    ]);
                }
            } elseif ($provider === 'ollama') {
                // Ollama uses OpenAI-compatible API at /v1/chat/completions
                $ollamaUrl = rtrim($baseUrl, '/') . '/v1/chat/completions';

                $headers = ['Content-Type' => 'application/json'];
                if (!empty($apiKey)) {
                    $headers['Authorization'] = 'Bearer ' . $apiKey;
                }

                $response = Http::withHeaders($headers)
                    ->timeout(30)
                    ->post($ollamaUrl, [
                        'model' => $model,
                        'messages' => [
                            ['role' => 'user', 'content' => 'Say "Connection successful" in exactly those words.']
                        ],
                        'max_tokens' => 20,
                    ]);

                if ($response->successful()) {
                    return ApiResponse::make('Success', [
                        'status' => 'success',
                        'message' => 'Connection successful! Ollama is configured correctly.',
                    ]);
                } else {
                    $error = $response->json()['error']['message'] ?? ($response->json()['error'] ?? 'Unknown error');
                    return ApiResponse::make('Error', [
                        'status' => 'error',
                        'message' => 'Connection failed: ' . (is_array($error) ? json_encode($error) : $error),
                    ]);
                }
            } elseif ($provider === 'openai_compatible') {
                // OpenAI-compatible providers (Groq, Together AI, OpenRouter, etc.)
                $apiUrl = rtrim($baseUrl, '/') . '/chat/completions';

                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ])->timeout(30)->post($apiUrl, [
                    'model' => $model,
                    'messages' => [
                        ['role' => 'user', 'content' => 'Say "Connection successful" in exactly those words.']
                    ],
                    'max_tokens' => 20,
                ]);

                if ($response->successful()) {
                    return ApiResponse::make('Success', [
                        'status' => 'success',
                        'message' => 'Connection successful! OpenAI-compatible provider is configured correctly.',
                    ]);
                } else {
                    $error = $response->json()['error']['message'] ?? ($response->json()['error'] ?? 'Unknown error');
                    return ApiResponse::make('Error', [
                        'status' => 'error',
                        'message' => 'Connection failed: ' . (is_array($error) ? json_encode($error) : $error),
                    ]);
                }
            }

            return ApiResponse::make('Error', [
                'status' => 'error',
                'message' => 'Unknown provider',
            ]);
        } catch (\Exception $e) {
            return ApiResponse::make('Error', [
                'status' => 'error',
                'message' => 'Connection failed: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * Get active AI configuration for use in website builder
     */
    public function getActiveConfig()
    {
        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            return ApiResponse::make('Success', [
                'configured' => false,
                'provider' => null,
            ]);
        }

        return ApiResponse::make('Success', [
            'configured' => true,
            'provider' => $aiSetting->name_key,
            'model' => $aiSetting->credentials['model'] ?? '',
        ]);
    }

    /**
     * Call AI API (OpenAI, Claude, Perplexity, Ollama, or OpenAI-compatible)
     */
    private function callAiApi($aiSetting, $systemPrompt, $userPrompt, $customMaxTokens = null)
    {
        $apiKey = $aiSetting->credentials['api_key'] ?? '';
        $model = $aiSetting->credentials['model'];
        $maxTokens = $customMaxTokens ?? ($aiSetting->credentials['max_tokens'] ?? 2000);
        $baseUrl = $aiSetting->credentials['base_url'] ?? '';

        if ($aiSetting->name_key === 'openai') {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', [
                'model' => $model,
                'messages' => [
                    ['role' => 'system', 'content' => $systemPrompt],
                    ['role' => 'user', 'content' => $userPrompt]
                ],
                'max_tokens' => (int)$maxTokens,
            ]);

            if ($response->successful()) {
                return $response->json()['choices'][0]['message']['content'] ?? '';
            } else {
                $error = $response->json()['error']['message'] ?? 'Unknown error';
                throw new \Exception($error);
            }
        } elseif ($aiSetting->name_key === 'claude') {
            $response = Http::withHeaders([
                'x-api-key' => $apiKey,
                'Content-Type' => 'application/json',
                'anthropic-version' => '2023-06-01',
            ])->post('https://api.anthropic.com/v1/messages', [
                'model' => $model,
                'max_tokens' => (int)$maxTokens,
                'system' => $systemPrompt,
                'messages' => [
                    ['role' => 'user', 'content' => $userPrompt]
                ],
            ]);

            if ($response->successful()) {
                return $response->json()['content'][0]['text'] ?? '';
            } else {
                $error = $response->json()['error']['message'] ?? 'Unknown error';
                throw new \Exception($error);
            }
        } elseif ($aiSetting->name_key === 'perplexity') {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.perplexity.ai/chat/completions', [
                'model' => $model,
                'messages' => [
                    ['role' => 'system', 'content' => $systemPrompt],
                    ['role' => 'user', 'content' => $userPrompt]
                ],
                'max_tokens' => (int)$maxTokens,
            ]);

            if ($response->successful()) {
                return $response->json()['choices'][0]['message']['content'] ?? '';
            } else {
                $error = $response->json()['error']['message'] ?? 'Unknown error';
                throw new \Exception($error);
            }
        } elseif ($aiSetting->name_key === 'ollama') {
            // Ollama uses OpenAI-compatible API
            $ollamaUrl = rtrim($baseUrl, '/') . '/v1/chat/completions';

            $headers = ['Content-Type' => 'application/json'];
            if (!empty($apiKey)) {
                $headers['Authorization'] = 'Bearer ' . $apiKey;
            }

            $response = Http::withHeaders($headers)
                ->timeout(120)
                ->post($ollamaUrl, [
                    'model' => $model,
                    'messages' => [
                        ['role' => 'system', 'content' => $systemPrompt],
                        ['role' => 'user', 'content' => $userPrompt]
                    ],
                    'max_tokens' => (int)$maxTokens,
                ]);

            if ($response->successful()) {
                return $response->json()['choices'][0]['message']['content'] ?? '';
            } else {
                $error = $response->json()['error']['message'] ?? ($response->json()['error'] ?? 'Unknown error');
                throw new \Exception(is_array($error) ? json_encode($error) : $error);
            }
        } elseif ($aiSetting->name_key === 'openai_compatible') {
            // OpenAI-compatible providers (Groq, Together AI, OpenRouter, etc.)
            $apiUrl = rtrim($baseUrl, '/') . '/chat/completions';

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(120)->post($apiUrl, [
                'model' => $model,
                'messages' => [
                    ['role' => 'system', 'content' => $systemPrompt],
                    ['role' => 'user', 'content' => $userPrompt]
                ],
                'max_tokens' => (int)$maxTokens,
            ]);

            if ($response->successful()) {
                return $response->json()['choices'][0]['message']['content'] ?? '';
            } else {
                $error = $response->json()['error']['message'] ?? ($response->json()['error'] ?? 'Unknown error');
                throw new \Exception(is_array($error) ? json_encode($error) : $error);
            }
        }

        throw new \Exception('Unknown AI provider');
    }

    /**
     * Extract website data from user description using AI
     */
    public function extractWebsiteData(Request $request)
    {
        $this->checkAiFeaturesAccess();

        // Extend execution time for AI requests (5 minutes)
        set_time_limit(300);

        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            throw new ApiException('AI is not configured. Please configure AI settings first.');
        }

        $description = $request->description;
        $categories = $request->categories ?? [];

        if (empty($description)) {
            throw new ApiException('Please provide a description of your website.');
        }

        // Build the category list for the prompt
        $categoryList = !empty($categories) ? implode(', ', array_column($categories, 'name')) : 'General';

        $systemPrompt = <<<PROMPT
You are a helpful assistant that extracts website configuration data from user descriptions.
Analyze the user's description and extract the following information in JSON format:

{
    "name": "Business/Website name (extract from description or suggest based on context)",
    "slug": "URL-friendly slug (lowercase, hyphens instead of spaces)",
    "category": "Best matching category from: {$categoryList}. If a good match exists, return the exact category name. If none match well, suggest a new appropriate category name.",
    "theme_primary_color": "Suggested primary color as hex code (e.g., #3B82F6) based on the business type",
    "theme_secondary_color": "ONLY include if user explicitly mentions secondary/accent color, otherwise set to null",
    "theme_mode": "light or dark (based on business type, most businesses use light)",
    "languages": [
        {
            "name": "Language name (e.g., English)",
            "key": "Language code (e.g., en)",
            "flag_code": "Country code for flag (e.g., US, GB, SA)",
            "is_rtl": false,
            "is_default": true
        }
    ],
    "website_settings": {
        "language_key": {
            "website_name": "Website/Business name in this language",
            "website_tagline": "Catchy tagline in this language (under 60 chars)",
            "website_description": "Business description (2-3 sentences) in this language",
            "seo_title": "SEO title for search engines (50-60 chars) in this language",
            "seo_description": "SEO meta description (150-160 chars) in this language",
            "seo_keywords": "Comma-separated keywords for SEO in this language"
        }
    }
}

Rules:
1. Extract languages mentioned in the description. If Arabic is mentioned, set is_rtl to true.
2. If no language is mentioned, default to English.
3. The first language should have is_default: true.
4. Choose colors that match the business type (e.g., green for eco/health, blue for tech/corporate, red/orange for food).
5. For "website_settings", create an entry for EACH language using the language key (e.g., "en", "es", "ar").
6. Translate all website_settings content into the respective language.
7. SEO content should be optimized for search engines in that language.
8. For theme_secondary_color: ONLY set a value if the user explicitly mentions a secondary or accent color. Otherwise, always set it to null.
9. For category: prefer existing categories from the list, but if none match the business type well, suggest an appropriate new category name.
10. Return ONLY the JSON object, no additional text or markdown.
PROMPT;

        try {
            $result = $this->callAiApi($aiSetting, $systemPrompt, $description);

            // Try to parse the JSON response
            $jsonResult = json_decode($result, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                // Try to extract JSON from the response
                preg_match('/\{[\s\S]*\}/', $result, $matches);
                if (!empty($matches[0])) {
                    $jsonResult = json_decode($matches[0], true);
                }
            }

            if (!$jsonResult) {
                throw new \Exception('Failed to parse AI response');
            }

            return ApiResponse::make('Success', [
                'extracted_data' => $jsonResult,
            ]);
        } catch (\Exception $e) {
            throw new ApiException('Failed to extract website data: ' . $e->getMessage());
        }
    }

    /**
     * Generate pages using AI based on website category and description
     */
    public function generatePages(Request $request)
    {
        $this->checkAiFeaturesAccess();

        // Extend execution time for AI requests (5 minutes)
        set_time_limit(300);

        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            throw new ApiException('AI is not configured. Please configure AI settings first.');
        }

        $websiteXid = $request->website_xid;
        $category = $request->category;
        $description = $request->description;
        $requirements = $request->requirements ?? '';
        $requestedLanguages = $request->languages ?? [];

        if (empty($websiteXid)) {
            throw new ApiException('Website ID is required.');
        }

        // Get website to verify languages
        $websiteId = $this->getIdFromHash($websiteXid);
        $website = Website::find($websiteId);

        if (!$website) {
            throw new ApiException('Website not found.');
        }

        // Get actual website languages from database - only use these
        $websiteLanguages = WebsiteLanguage::where('website_id', $websiteId)->get();

        if ($websiteLanguages->isEmpty()) {
            throw new ApiException('No languages found for this website. Please add languages first.');
        }

        // Build language list from actual database records, not from request
        // This ensures we only generate content for languages that actually exist
        $languageList = [];
        foreach ($websiteLanguages as $lang) {
            $languageList[] = [
                'key' => $lang->key,
                'name' => $lang->name,
                'is_rtl' => $lang->is_rtl ?? false
            ];
        }
        $languagesJson = json_encode($languageList);

        // Build example translations based on actual languages
        $exampleTranslations = [];
        foreach ($languageList as $lang) {
            $exampleName = $lang['key'] === 'en' ? 'Home' : ($lang['key'] === 'ar' ? 'الرئيسية' : 'Home');
            $exampleTranslations[$lang['key']] = ['name' => $exampleName];
        }
        $exampleTranslationsJson = json_encode($exampleTranslations, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

        $exampleTranslations2 = [];
        foreach ($languageList as $lang) {
            $exampleName = $lang['key'] === 'en' ? 'About Us' : ($lang['key'] === 'ar' ? 'من نحن' : 'About Us');
            $exampleTranslations2[$lang['key']] = ['name' => $exampleName];
        }
        $exampleTranslationsJson2 = json_encode($exampleTranslations2, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

        // Build the prompt
        $contextInfo = "Website Category: " . ($category ?? 'General') . "\n";
        $contextInfo .= "Website Description: " . ($description ?? 'A professional website') . "\n";

        if (!empty($requirements)) {
            $contextInfo .= "Additional Requirements: " . $requirements . "\n";
        }

        $systemPrompt = <<<PROMPT
You are a helpful assistant that generates website pages based on business type and requirements.
Generate a list of appropriate pages for the website.

For each page, provide:
1. page_key: A unique identifier (lowercase, hyphens for spaces, e.g., "home", "about-us", "services", "contact")
2. show_in_header: Whether this page should appear in the main navigation (true/false)
3. translations: Page names in ONLY these required languages

IMPORTANT - Languages required (ONLY generate translations for these languages, no others): {$languagesJson}

Rules:
1. Generate 4-8 pages appropriate for the business type
2. Always include: home, about (or about-us), contact
3. Add relevant pages based on the business category (e.g., services, portfolio, pricing, testimonials, blog, gallery, team, faq)
4. The "home" page should always have show_in_header: false (it's usually the logo link)
5. Translate page names appropriately for each language in the list above
6. For RTL languages (like Arabic), ensure names are properly translated
7. Return ONLY a JSON array, no additional text or markdown
8. IMPORTANT: Use hyphens (-) not underscores (_) in page_key
9. CRITICAL: Only include translations for the languages specified above. Do NOT add any other languages.

Return format (example with the exact languages you must use):
[
    {
        "page_key": "home",
        "show_in_header": false,
        "translations": {$exampleTranslationsJson}
    },
    {
        "page_key": "about-us",
        "show_in_header": true,
        "translations": {$exampleTranslationsJson2}
    }
]
PROMPT;

        try {
            $result = $this->callAiApi($aiSetting, $systemPrompt, $contextInfo);

            // Try to parse the JSON response
            $jsonResult = json_decode($result, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                // Try to extract JSON array from the response
                preg_match('/\[[\s\S]*\]/', $result, $matches);
                if (!empty($matches[0])) {
                    $jsonResult = json_decode($matches[0], true);
                }
            }

            if (!$jsonResult || !is_array($jsonResult)) {
                throw new \Exception('Failed to parse AI response');
            }

            // Get valid language keys for this website
            $validLanguageKeys = array_column($languageList, 'key');

            // Post-process: Filter out any unauthorized languages from AI response
            // This ensures only website-configured languages are included
            foreach ($jsonResult as &$page) {
                if (isset($page['translations']) && is_array($page['translations'])) {
                    // Filter translations to only include valid languages
                    $page['translations'] = array_filter(
                        $page['translations'],
                        function ($langKey) use ($validLanguageKeys) {
                            return in_array($langKey, $validLanguageKeys);
                        },
                        ARRAY_FILTER_USE_KEY
                    );

                    // Ensure all valid languages have translations
                    foreach ($validLanguageKeys as $langKey) {
                        if (!isset($page['translations'][$langKey])) {
                            // Add missing translation with fallback name
                            $fallbackName = ucfirst(str_replace('-', ' ', $page['page_key'] ?? 'Page'));
                            $page['translations'][$langKey] = ['name' => $fallbackName];
                        }
                    }
                }
            }
            unset($page); // Unset reference

            return ApiResponse::make('Success', [
                'pages' => $jsonResult,
            ]);
        } catch (\Exception $e) {
            throw new ApiException('Failed to generate pages: ' . $e->getMessage());
        }
    }

    /**
     * Save AI-generated pages to the database
     */
    public function saveGeneratedPages(Request $request)
    {
        $this->checkAiFeaturesAccess();

        $websiteXid = $request->website_xid;
        $pages = $request->pages ?? [];

        if (empty($websiteXid)) {
            throw new ApiException('Website ID is required.');
        }

        if (empty($pages)) {
            throw new ApiException('No pages to save.');
        }

        // Get website
        $website = Website::where('id', $this->getIdFromHash($websiteXid))->first();

        if (!$website) {
            throw new ApiException('Website not found.');
        }

        // Get website languages
        $websiteLanguages = WebsiteLanguage::where('website_id', $website->id)->get()->keyBy('key');

        if ($websiteLanguages->isEmpty()) {
            throw new ApiException('No languages found for this website.');
        }

        // Get existing pages for this website (keyed by page_key)
        $existingPages = WebsitePage::where('website_id', $website->id)
            ->get()
            ->keyBy('page_key');

        DB::beginTransaction();

        try {
            $sortOrder = WebsitePage::where('website_id', $website->id)->max('sort_order') ?? 0;
            $createdCount = 0;
            $updatedCount = 0;

            foreach ($pages as $pageData) {
                $translations = $pageData['translations'] ?? [];

                // Check if page already exists
                if ($existingPages->has($pageData['page_key'])) {
                    // Update existing page translations
                    $page = $existingPages->get($pageData['page_key']);

                    foreach ($translations as $langKey => $translationData) {
                        $websiteLang = $websiteLanguages->get($langKey);

                        if ($websiteLang) {
                            WebsitePageTranslation::updateOrCreate(
                                [
                                    'website_page_id' => $page->id,
                                    'website_language_id' => $websiteLang->id,
                                ],
                                [
                                    'name' => $translationData['name'] ?? ucfirst(str_replace('-', ' ', $pageData['page_key'])),
                                    'show_in_header' => $pageData['show_in_header'] ?? true,
                                ]
                            );
                        }
                    }

                    // Create translations for languages not in AI response
                    foreach ($websiteLanguages as $langKey => $websiteLang) {
                        if (!isset($translations[$langKey])) {
                            WebsitePageTranslation::updateOrCreate(
                                [
                                    'website_page_id' => $page->id,
                                    'website_language_id' => $websiteLang->id,
                                ],
                                [
                                    'name' => ucfirst(str_replace('-', ' ', $pageData['page_key'])),
                                    'show_in_header' => $pageData['show_in_header'] ?? true,
                                ]
                            );
                        }
                    }

                    $updatedCount++;
                } else {
                    // Create new page
                    $sortOrder++;

                    $page = WebsitePage::create([
                        'company_id' => $website->company_id,
                        'website_id' => $website->id,
                        'page_key' => $pageData['page_key'],
                        'sort_order' => $sortOrder,
                    ]);

                    // Create translations for each language
                    foreach ($translations as $langKey => $translationData) {
                        $websiteLang = $websiteLanguages->get($langKey);

                        if ($websiteLang) {
                            WebsitePageTranslation::create([
                                'website_page_id' => $page->id,
                                'website_language_id' => $websiteLang->id,
                                'name' => $translationData['name'] ?? ucfirst(str_replace('-', ' ', $pageData['page_key'])),
                                'show_in_header' => $pageData['show_in_header'] ?? true,
                            ]);
                        }
                    }

                    // Create translations for languages not in AI response
                    foreach ($websiteLanguages as $langKey => $websiteLang) {
                        if (!isset($translations[$langKey])) {
                            WebsitePageTranslation::create([
                                'website_page_id' => $page->id,
                                'website_language_id' => $websiteLang->id,
                                'name' => ucfirst(str_replace('-', ' ', $pageData['page_key'])),
                                'show_in_header' => $pageData['show_in_header'] ?? true,
                            ]);
                        }
                    }

                    $createdCount++;
                }
            }

            DB::commit();

            return ApiResponse::make('Pages saved successfully', [
                'created' => $createdCount,
                'updated' => $updatedCount,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new ApiException('Failed to save pages: ' . $e->getMessage());
        }
    }

    /**
     * Generate legal page suggestions using AI based on website info
     * Returns a list of suggested legal pages (similar to generatePages for website pages)
     */
    public function generateLegalPageSuggestions(Request $request)
    {
        $this->checkAiFeaturesAccess();

        // Extend execution time for AI requests (5 minutes)
        set_time_limit(300);

        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            throw new ApiException('AI is not configured. Please configure AI settings first.');
        }

        $websiteName = $request->website_name;
        $category = $request->category;
        $description = $request->description;
        $requirements = $request->requirements ?? '';
        $languages = $request->languages ?? [];

        // Build language list for the prompt
        $languageList = [];
        foreach ($languages as $lang) {
            $languageList[] = [
                'key' => $lang['key'],
                'name' => $lang['name'],
                'is_rtl' => $lang['is_rtl'] ?? false
            ];
        }
        $languagesJson = json_encode($languageList);

        // Build the context
        $contextInfo = "Business/Website Name: " . ($websiteName ?? 'Our Company') . "\n";
        $contextInfo .= "Business Category: " . ($category ?? 'General') . "\n";
        $contextInfo .= "Business Description: " . ($description ?? 'A professional business') . "\n";

        if (!empty($requirements)) {
            $contextInfo .= "Additional Requirements: " . $requirements . "\n";
        }

        $systemPrompt = <<<PROMPT
You are a helpful assistant that suggests appropriate legal pages for websites based on their business type.
Analyze the business information and suggest the most relevant legal pages.

For each legal page suggestion, provide:
1. slug: URL-friendly slug (lowercase, hyphens, e.g., "privacy-policy", "terms-of-service")
2. title: The page title in each language
3. description: A brief description of what this page covers

Languages required: {$languagesJson}

IMPORTANT RULES:
1. Suggest 3-6 legal pages appropriate for the business type
2. Common legal pages include: Privacy Policy, Terms of Service, Cookie Policy, Refund Policy, Disclaimer, DMCA Policy, Shipping Policy, Return Policy
3. For e-commerce/retail businesses: include Refund Policy, Shipping Policy, Return Policy
4. For service businesses: include Terms of Service, Service Agreement
5. For tech/software: include Privacy Policy, Terms of Service, Cookie Policy, DMCA Policy
6. All businesses should have: Privacy Policy, Terms of Service
7. Translate titles and descriptions to each language
8. For RTL languages, ensure proper translation
9. Return ONLY a JSON array, no additional text or markdown

Return format:
[
    {
        "slug": "privacy-policy",
        "translations": {
            "en": { "title": "Privacy Policy", "description": "Explains how you collect, use, and protect user data" },
            "ar": { "title": "سياسة الخصوصية", "description": "توضح كيفية جمع واستخدام وحماية بيانات المستخدمين" }
        },
        "recommended": true
    },
    {
        "slug": "terms-of-service",
        "translations": {
            "en": { "title": "Terms of Service", "description": "Rules and guidelines for using your website/service" },
            "ar": { "title": "شروط الخدمة", "description": "القواعد والإرشادات لاستخدام موقعك/خدمتك" }
        },
        "recommended": true
    }
]
PROMPT;

        try {
            $result = $this->callAiApi($aiSetting, $systemPrompt, $contextInfo, 2000);

            // Try to parse the JSON response
            $jsonResult = $this->parseAiJsonResponse($result);

            if (!$jsonResult || !is_array($jsonResult)) {
                Log::error('Failed to parse legal page suggestions AI response', ['raw_response' => substr($result, 0, 500)]);
                throw new \Exception('Failed to parse AI response. Please try again.');
            }

            return ApiResponse::make('Success', [
                'legal_pages' => $jsonResult,
            ]);
        } catch (\Exception $e) {
            throw new ApiException('Failed to generate legal page suggestions: ' . $e->getMessage());
        }
    }

    /**
     * Generate legal pages using AI based on website info
     */
    public function generateLegalPages(Request $request)
    {
        $this->checkAiFeaturesAccess();

        // Extend execution time for AI requests (5 minutes)
        set_time_limit(300);

        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            throw new ApiException('AI is not configured. Please configure AI settings first.');
        }

        $websiteXid = $request->website_xid;
        $websiteName = $request->website_name;
        $category = $request->category;
        $description = $request->description;
        $requirements = $request->requirements ?? '';
        $languages = $request->languages ?? [];

        if (empty($websiteXid)) {
            throw new ApiException('Website ID is required.');
        }

        // Build language list for the prompt
        $languageList = [];
        foreach ($languages as $lang) {
            $languageList[] = [
                'key' => $lang['key'],
                'name' => $lang['name'],
                'is_rtl' => $lang['is_rtl'] ?? false
            ];
        }
        $languagesJson = json_encode($languageList);

        $currentDate = date('F d, Y');

        // Build the context
        $contextInfo = "Business/Website Name: " . ($websiteName ?? 'Our Company') . "\n";
        $contextInfo .= "Business Category: " . ($category ?? 'General') . "\n";
        $contextInfo .= "Business Description: " . ($description ?? 'A professional business') . "\n";
        $contextInfo .= "Current Date: " . $currentDate . "\n";

        if (!empty($requirements)) {
            $contextInfo .= "Additional Requirements: " . $requirements . "\n";
        }

        $systemPrompt = <<<PROMPT
You are a helpful assistant that generates professional legal pages for websites.
Generate legal pages appropriate for the business type described.

For each legal page, provide:
1. title: The page title (e.g., "Privacy Policy", "Terms of Service")
2. slug: URL-friendly slug (lowercase, hyphens, e.g., "privacy-policy")
3. content: Full HTML content for the legal page. Use proper HTML tags (h1, h2, p, ul, li, section, etc.). Make the content SPECIFIC to the business - use the business name throughout the content.
4. meta_title: SEO title (50-60 chars)
5. meta_description: SEO description (150-160 chars)
6. meta_keywords: Comma-separated keywords
7. lang_key: Language key for this page

Languages required: {$languagesJson}

IMPORTANT RULES:
1. Generate 3-4 legal pages: Privacy Policy, Terms of Service, Cookie Policy, and optionally Refund Policy (if applicable to business type)
2. Create SEPARATE entries for EACH language - e.g., if there are 2 languages, generate 6-8 entries total (3-4 pages × 2 languages)
3. Use the EXACT business name "{$websiteName}" throughout the content
4. Make content relevant to the business category and type
5. Include proper sections with h2 headings
6. Content should be professional and legally-sounding
7. For RTL languages like Arabic, translate the entire content properly
8. Each page should have "Last updated: {$currentDate}" near the top
9. Return ONLY a JSON array, no additional text or markdown

Return format:
[
    {
        "title": "Privacy Policy",
        "slug": "privacy-policy",
        "content": "<div class='legal-page'><h1>Privacy Policy</h1><p class='last-updated'>Last updated: {$currentDate}</p><section><h2>1. Information We Collect</h2><p>{$websiteName} collects...</p></section>...</div>",
        "meta_title": "Privacy Policy - {$websiteName}",
        "meta_description": "Learn how {$websiteName} collects, uses, and protects your personal information.",
        "meta_keywords": "privacy policy, data protection, {$websiteName}",
        "lang_key": "en"
    },
    {
        "title": "سياسة الخصوصية",
        "slug": "privacy-policy",
        "content": "<div class='legal-page' dir='rtl'>...</div>",
        "meta_title": "سياسة الخصوصية - {$websiteName}",
        "meta_description": "...",
        "meta_keywords": "...",
        "lang_key": "ar"
    }
]
PROMPT;

        try {
            // Use higher token limit for legal pages (they contain a lot of HTML content)
            $result = $this->callAiApi($aiSetting, $systemPrompt, $contextInfo, 8000);

            // Try to parse the JSON response
            $jsonResult = $this->parseAiJsonResponse($result);

            if (!$jsonResult || !is_array($jsonResult)) {
                Log::error('Failed to parse legal pages AI response', ['raw_response' => substr($result, 0, 1000)]);
                throw new \Exception('Failed to parse AI response. Please try again.');
            }

            return ApiResponse::make('Success', [
                'legal_pages' => $jsonResult,
            ]);
        } catch (\Exception $e) {
            throw new ApiException('Failed to generate legal pages: ' . $e->getMessage());
        }
    }

    /**
     * Generate a single legal page content using AI
     */
    public function generateLegalPageContent(Request $request)
    {
        $this->checkAiFeaturesAccess();

        // Extend execution time for AI requests (5 minutes)
        set_time_limit(300);

        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            throw new ApiException('AI is not configured. Please configure AI settings first.');
        }

        $websiteXid = $request->website_xid;
        $websiteName = $request->website_name;
        $category = $request->category;
        $description = $request->description;
        $requirements = $request->requirements ?? '';
        $pageType = $request->page_type; // e.g., 'privacy-policy', 'terms-of-service'
        $pageTitle = $request->page_title; // e.g., 'Privacy Policy'
        $language = $request->language; // { key: 'en', name: 'English', is_rtl: false }

        if (empty($websiteXid)) {
            throw new ApiException('Website ID is required.');
        }

        if (empty($pageType) || empty($pageTitle)) {
            throw new ApiException('Page type and title are required.');
        }

        $currentDate = date('F d, Y');
        $langKey = $language['key'] ?? 'en';
        $langName = $language['name'] ?? 'English';
        $isRtl = $language['is_rtl'] ?? false;

        // Build the context
        $contextInfo = "Business/Website Name: " . ($websiteName ?? 'Our Company') . "\n";
        $contextInfo .= "Business Category: " . ($category ?? 'General') . "\n";
        $contextInfo .= "Business Description: " . ($description ?? 'A professional business') . "\n";
        $contextInfo .= "Current Date: " . $currentDate . "\n";
        $contextInfo .= "Page Type: " . $pageTitle . " (" . $pageType . ")\n";
        $contextInfo .= "Target Language: " . $langName . " (" . $langKey . ")\n";

        if (!empty($requirements)) {
            $contextInfo .= "Additional Requirements: " . $requirements . "\n";
        }

        $rtlNote = $isRtl ? "IMPORTANT: This is an RTL language. Add dir='rtl' to the main div and translate ALL content properly to {$langName}." : "";

        $systemPrompt = <<<PROMPT
You are a helpful assistant that generates professional legal page content for websites.
Generate a {$pageTitle} page for the business described.

Language: {$langName} ({$langKey})
{$rtlNote}

CONTENT STRUCTURE - Use this exact HTML structure:
<div class="legal-page">
    <h1>[Page Title in {$langName}]</h1>
    <p class="last-updated">Last updated: {$currentDate}</p>

    <section>
        <h2>1. [Section Title]</h2>
        <p>[Section content with business name: {$websiteName}]</p>
        <ul>
            <li>[Point 1]</li>
            <li>[Point 2]</li>
        </ul>
    </section>

    <section>
        <h2>2. [Next Section Title]</h2>
        <p>[Content...]</p>
    </section>

    <!-- Add 6-9 relevant sections based on page type -->
</div>

IMPORTANT RULES:
1. Use the EXACT business name "{$websiteName}" throughout the content (at least 5-10 times)
2. Make content relevant to the business category: {$category}
3. Include 6-9 properly numbered sections with h2 headings
4. Content should be professional and legally-sounding
5. Use proper HTML tags: div, h1, h2, p, ul, li, strong, section
6. Content must be written entirely in {$langName}
7. Return ONLY a JSON object, no additional text or markdown

Return format:
{
    "title": "[Page title in {$langName}]",
    "slug": "{$pageType}",
    "content": "<div class='legal-page'>...</div>",
    "meta_title": "[SEO title 50-60 chars in {$langName}]",
    "meta_description": "[SEO description 150-160 chars in {$langName}]",
    "meta_keywords": "[keywords in {$langName}]",
    "lang_key": "{$langKey}"
}
PROMPT;

        try {
            // Use moderate token limit for single page
            $result = $this->callAiApi($aiSetting, $systemPrompt, $contextInfo, 4000);

            // Try to parse the JSON response
            $jsonResult = $this->parseAiJsonResponseObject($result);

            if (!$jsonResult || !is_array($jsonResult)) {
                Log::error('Failed to parse legal page AI response', ['raw_response' => substr($result, 0, 500)]);
                throw new \Exception('Failed to parse AI response. Please try again.');
            }

            return ApiResponse::make('Success', [
                'legal_page' => $jsonResult,
            ]);
        } catch (\Exception $e) {
            throw new ApiException('Failed to generate legal page: ' . $e->getMessage());
        }
    }

    /**
     * Parse AI JSON response (object) with better error handling
     */
    private function parseAiJsonResponseObject($result)
    {
        // First, try direct parsing
        $jsonResult = json_decode($result, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($jsonResult) && isset($jsonResult['title'])) {
            return $jsonResult;
        }

        // Clean up the response - remove markdown code blocks if present
        $cleaned = $result;

        // Remove markdown code blocks (```json ... ``` or ``` ... ```)
        $cleaned = preg_replace('/^```(?:json)?\s*/i', '', $cleaned);
        $cleaned = preg_replace('/\s*```$/', '', $cleaned);

        // Try parsing the cleaned version
        $jsonResult = json_decode($cleaned, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($jsonResult) && isset($jsonResult['title'])) {
            return $jsonResult;
        }

        // Try to extract JSON object from the response
        if (preg_match('/\{[\s\S]*\}/s', $cleaned, $matches)) {
            $extracted = $matches[0];

            // Try parsing the extracted JSON
            $jsonResult = json_decode($extracted, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($jsonResult) && isset($jsonResult['title'])) {
                return $jsonResult;
            }
        }

        return null;
    }

    /**
     * Parse AI JSON response with better error handling
     */
    private function parseAiJsonResponse($result)
    {
        // First, try direct parsing
        $jsonResult = json_decode($result, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($jsonResult)) {
            return $jsonResult;
        }

        // Clean up the response - remove markdown code blocks if present
        $cleaned = $result;

        // Remove markdown code blocks (```json ... ``` or ``` ... ```)
        $cleaned = preg_replace('/^```(?:json)?\s*/i', '', $cleaned);
        $cleaned = preg_replace('/\s*```$/', '', $cleaned);

        // Try parsing the cleaned version
        $jsonResult = json_decode($cleaned, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($jsonResult)) {
            return $jsonResult;
        }

        // Try to extract JSON array from the response
        if (preg_match('/\[[\s\S]*\]/s', $cleaned, $matches)) {
            $extracted = $matches[0];

            // Try parsing the extracted JSON
            $jsonResult = json_decode($extracted, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($jsonResult)) {
                return $jsonResult;
            }

            // If still failing, try to fix common JSON issues
            // Fix unescaped newlines in strings
            $fixed = preg_replace('/(?<!\\\\)\n/', '\\n', $extracted);

            // Try parsing again
            $jsonResult = json_decode($fixed, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($jsonResult)) {
                return $jsonResult;
            }
        }

        return null;
    }

    /**
     * Save AI-generated legal pages to the database
     */
    public function saveGeneratedLegalPages(Request $request)
    {
        $this->checkAiFeaturesAccess();

        $websiteXid = $request->website_xid;
        $legalPages = $request->legal_pages ?? [];

        if (empty($websiteXid)) {
            throw new ApiException('Website ID is required.');
        }

        if (empty($legalPages)) {
            throw new ApiException('No legal pages to save.');
        }

        // Get website
        $website = Website::where('id', $this->getIdFromHash($websiteXid))->first();

        if (!$website) {
            throw new ApiException('Website not found.');
        }

        // Get website languages (keyed by lang key)
        $websiteLanguages = WebsiteLanguage::where('website_id', $website->id)->get()->keyBy('key');

        if ($websiteLanguages->isEmpty()) {
            throw new ApiException('No languages found for this website.');
        }

        DB::beginTransaction();

        try {
            $displayOrder = WebsiteLegalPage::where('website_id', $website->id)->max('display_order') ?? 0;
            $createdCount = 0;
            $updatedCount = 0;

            foreach ($legalPages as $pageData) {
                $langKey = $pageData['lang_key'] ?? 'en';
                $websiteLang = $websiteLanguages->get($langKey);

                if (!$websiteLang) {
                    continue; // Skip if language not found
                }

                // Use updateOrCreate to handle race conditions and duplicate entries
                $existingPage = WebsiteLegalPage::where('website_id', $website->id)
                    ->where('website_language_id', $websiteLang->id)
                    ->where('slug', $pageData['slug'])
                    ->first();

                if ($existingPage) {
                    // Update existing legal page
                    $existingPage->update([
                        'title' => $pageData['title'],
                        'content' => $pageData['content'],
                        'meta_title' => $pageData['meta_title'] ?? $pageData['title'],
                        'meta_description' => $pageData['meta_description'] ?? '',
                        'meta_keywords' => $pageData['meta_keywords'] ?? '',
                    ]);
                    $updatedCount++;
                } else {
                    // Create new legal page
                    $displayOrder++;

                    WebsiteLegalPage::create([
                        'company_id' => $website->company_id,
                        'website_id' => $website->id,
                        'website_language_id' => $websiteLang->id,
                        'title' => $pageData['title'],
                        'slug' => $pageData['slug'],
                        'content' => $pageData['content'],
                        'meta_title' => $pageData['meta_title'] ?? $pageData['title'],
                        'meta_description' => $pageData['meta_description'] ?? '',
                        'meta_keywords' => $pageData['meta_keywords'] ?? '',
                        'show_in_footer' => true,
                        'display_order' => $displayOrder,
                        'status' => 'active',
                    ]);
                    $createdCount++;
                }
            }

            DB::commit();

            return ApiResponse::make('Legal pages saved successfully', [
                'created' => $createdCount,
                'updated' => $updatedCount,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw new ApiException('Failed to save legal pages: ' . $e->getMessage());
        }
    }

    /**
     * Get theme sections configuration for AI
     */
    public function getThemeSectionsConfig()
    {
        $configPath = public_path('theme-sections-config.json');

        if (!file_exists($configPath)) {
            // Generate the config on demand
            $this->generateThemeSectionsConfig();
        }

        $config = json_decode(file_get_contents($configPath), true);

        return ApiResponse::make('Success', $config);
    }

    /**
     * Generate theme sections configuration file
     */
    protected function generateThemeSectionsConfig()
    {
        // Get all active sections from database
        $sections = ThemeSection::where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        // Build sections data
        $sectionsData = [];
        foreach ($sections as $section) {
            $sectionsData[$section->section_key] = [
                'name' => $section->name,
                'description' => $section->description,
                'content_schema' => $section->content_schema,
                'default_content' => $section->default_content ?? [],
                'sort_order' => $section->sort_order
            ];
        }

        // Get layouts from file system
        $componentsPath = resource_path('views/themes/tailwind_theme/components');
        $layoutsData = [];

        if (is_dir($componentsPath)) {
            $dirs = scandir($componentsPath);
            foreach ($dirs as $dir) {
                if ($dir === '.' || $dir === '..') continue;
                $sectionPath = $componentsPath . '/' . $dir;
                if (!is_dir($sectionPath)) continue;

                $layouts = [];
                $files = scandir($sectionPath);
                foreach ($files as $file) {
                    if (preg_match('/^⚡(.+)\.blade\.php$/', $file, $matches)) {
                        $layoutName = $matches[1];
                        $layouts[] = [
                            'value' => $layoutName,
                            'label' => ucwords(str_replace(['-', '_'], ' ', $layoutName))
                        ];
                    }
                }

                if (!empty($layouts)) {
                    $layoutsData[$dir] = $layouts;
                }
            }
        }

        // Combine data
        $result = [
            'sections' => $sectionsData,
            'layouts' => $layoutsData,
            'generated_at' => now()->toISOString()
        ];

        file_put_contents(public_path('theme-sections-config.json'), json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

        return $result;
    }

    /**
     * Generate section suggestions for a website page using AI
     */
    public function generateSectionSuggestions(Request $request)
    {
        $this->checkAiFeaturesAccess();

        // Extend execution time for AI requests (5 minutes)
        set_time_limit(300);

        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            throw new ApiException('AI is not configured. Please configure AI settings first.');
        }

        $websiteXid = $request->website_xid;
        $category = $request->category ?? '';
        $description = $request->description ?? '';
        $requirements = $request->requirements ?? '';
        $pages = $request->pages ?? []; // Array of page_key values

        if (empty($websiteXid)) {
            throw new ApiException('Website ID is required.');
        }

        // Get available sections from config
        $configPath = public_path('theme-sections-config.json');
        if (!file_exists($configPath)) {
            $this->generateThemeSectionsConfig();
        }
        $config = json_decode(file_get_contents($configPath), true);
        $availableSections = $config['sections'] ?? [];
        $availableLayouts = $config['layouts'] ?? [];

        // Build sections list for prompt (excluding header and footer as they are common)
        $sectionsForPrompt = [];
        foreach ($availableSections as $key => $section) {
            if (!in_array($key, ['header', 'footer'])) {
                $sectionsForPrompt[$key] = [
                    'name' => $section['name'],
                    'description' => $section['description'] ?? ''
                ];
            }
        }

        // Build layouts info for prompt (excluding header/footer for AI suggestions)
        $layoutsForPrompt = [];
        foreach ($availableLayouts as $sectionKey => $layouts) {
            if (!in_array($sectionKey, ['header', 'footer'])) {
                $layoutsForPrompt[$sectionKey] = array_column($layouts, 'value');
            }
        }

        // Note: availableLayouts includes header/footer for frontend use (for common section selection)

        $sectionsJson = json_encode($sectionsForPrompt);
        $layoutsJson = json_encode($layoutsForPrompt);
        $pagesJson = json_encode($pages);

        $contextInfo = "Business Category: " . ($category ?: 'General') . "\n";
        $contextInfo .= "Business Description: " . ($description ?: 'A professional business') . "\n";
        $contextInfo .= "Website Pages: " . implode(', ', $pages) . "\n";

        if (!empty($requirements)) {
            $contextInfo .= "Additional Requirements: " . $requirements . "\n";
        }

        $systemPrompt = <<<PROMPT
You are a web design expert that suggests appropriate website sections for different page types.

Available sections and their purposes:
{$sectionsJson}

Available layouts for each section:
{$layoutsJson}

Pages to generate sections for: {$pagesJson}

Based on the business type and each page's purpose, suggest which sections should be included on each page.

IMPORTANT RULES:
1. Header and footer are COMMON across all pages - DO NOT include them in suggestions
2. For 'home' page: include hero, features, services, stats, testimonials, cta at minimum
3. For 'about' page: include hero (or about section), team, stats
4. For 'services' page: include hero, services section with details
5. For 'pricing' page: include hero, pricing section, faq
6. For 'contact' page: include contact section with form, faq (optional)
7. For other pages: suggest appropriate sections based on page name/purpose
8. Suggest appropriate layout variants for visual variety
9. Return ONLY a JSON object, no additional text

Return format:
{
    "home": [
        {"section_key": "hero", "layout": "hero-split", "sort_order": 1},
        {"section_key": "features", "layout": "features-grid", "sort_order": 2},
        {"section_key": "services", "layout": "services-cards", "sort_order": 3}
    ],
    "about": [
        {"section_key": "about", "layout": "about", "sort_order": 1},
        {"section_key": "team", "layout": "team-grid-2", "sort_order": 2}
    ]
}
PROMPT;

        try {
            $result = $this->callAiApi($aiSetting, $systemPrompt, $contextInfo, 3000);

            // Try to parse the JSON response
            $jsonResult = $this->parseAiJsonResponse($result);

            if (!$jsonResult || !is_array($jsonResult)) {
                Log::error('Failed to parse section suggestions AI response', ['raw_response' => substr($result, 0, 500)]);
                throw new \Exception('Failed to parse AI response. Please try again.');
            }

            return ApiResponse::make('Success', [
                'suggestions' => $jsonResult,
                'available_sections' => $sectionsForPrompt,
                'available_layouts' => $availableLayouts
            ]);
        } catch (\Exception $e) {
            throw new ApiException('Failed to generate section suggestions: ' . $e->getMessage());
        }
    }

    /**
     * Generate section content for a specific section using AI
     */
    public function generateSectionContent(Request $request)
    {
        $this->checkAiFeaturesAccess();

        // Extend execution time for AI requests (5 minutes)
        set_time_limit(300);

        $aiSetting = Settings::where('setting_type', 'ai')
            ->where('status', 1)
            ->first();

        if (!$this->isAiConfigured($aiSetting)) {
            throw new ApiException('AI is not configured. Please configure AI settings first.');
        }

        $websiteXid = $request->website_xid;
        $category = $request->category ?? '';
        $description = $request->description ?? '';
        $sectionKey = $request->section_key;
        $pageKey = $request->page_key;
        $language = $request->language; // { key: 'en', name: 'English', is_rtl: false }

        if (empty($websiteXid) || empty($sectionKey)) {
            throw new ApiException('Website ID and section key are required.');
        }

        // Get section schema from database
        $themeSection = ThemeSection::where('section_key', $sectionKey)
            ->where('is_active', true)
            ->first();

        if (!$themeSection) {
            throw new ApiException("Section '{$sectionKey}' not found.");
        }

        $contentSchema = $themeSection->content_schema ?? [];
        $defaultContent = $themeSection->default_content ?? [];
        $defaultTitle = $themeSection->default_title ?? $themeSection->name;
        $defaultSubtitle = $themeSection->default_subtitle ?? '';

        // Build schema with content length hints from default_content
        $schemaWithHints = $this->buildSchemaWithContentHints($contentSchema, $defaultContent);
        $schemaJson = json_encode($schemaWithHints, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

        // Build example content from default_content for reference
        $exampleContentJson = json_encode($defaultContent, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

        $langKey = $language['key'] ?? 'en';
        $langName = $language['name'] ?? 'English';
        $isRtl = $language['is_rtl'] ?? false;

        // Calculate title/subtitle length hints
        $titleLengthHint = $defaultTitle ? strlen($defaultTitle) . ' characters' : '20-50 characters';
        $subtitleLengthHint = $defaultSubtitle ? strlen($defaultSubtitle) . ' characters' : '40-80 characters';

        $contextInfo = "Business Category: " . ($category ?: 'General') . "\n";
        $contextInfo .= "Business Description: " . ($description ?: 'A professional business') . "\n";
        $contextInfo .= "Section: " . $themeSection->name . " (" . $sectionKey . ")\n";
        $contextInfo .= "Page: " . $pageKey . "\n";
        $contextInfo .= "Language: " . $langName . " (" . $langKey . ")" . ($isRtl ? " [RTL]" : "") . "\n";

        $systemPrompt = <<<PROMPT
You are a content writer that generates website section content.

Generate content for a "{$themeSection->name}" section on the "{$pageKey}" page.
The content should be appropriate for a {$category} business.

SECTION TITLE AND SUBTITLE (generate these first):
- Example title: "{$defaultTitle}" (approximately {$titleLengthHint})
- Example subtitle: "{$defaultSubtitle}" (approximately {$subtitleLengthHint})

Content schema (fields to generate):
{$schemaJson}

EXAMPLE CONTENT (use similar structure and content length):
{$exampleContentJson}

CRITICAL RULES:
1. Generate content in {$langName} language (language code: {$langKey})
2. MUST include "title" and "subtitle" fields in your response with similar length to the examples
3. MATCH THE CONTENT LENGTH of the example content - if a field has 50 characters in the example, your content should be approximately 50 characters
4. For 'text' or 'textarea' fields: provide appropriate text content with SIMILAR LENGTH to the example
5. For 'repeater' fields: provide the SAME NUMBER of items as in the example content
6. For 'nested_repeater' fields: maintain the same nesting structure as the example
7. For 'switch' fields: provide true/false boolean
8. For 'image' fields: set to null (images will be uploaded separately)
9. For 'icon' fields: use Lucide icon names (e.g., "lucide:check", "lucide:star", "lucide:rocket", "lucide:shield")
10. For 'select' fields: choose from the available options
11. For 'number' fields: provide integer values (e.g., for "rating" field use 1-5, for counts use appropriate numbers)
12. Make content professional, relevant to the business type, and engaging
13. Return ONLY a JSON object with "title", "subtitle" and content fields matching the schema structure, no additional text
14. The JSON structure MUST exactly match the example content structure plus title/subtitle - include ALL fields from the example
PROMPT;

        try {
            $result = $this->callAiApi($aiSetting, $systemPrompt, $contextInfo, 4000);

            // Try to parse the JSON response
            $jsonResult = $this->parseAiJsonResponse($result);

            if (!$jsonResult || !is_array($jsonResult)) {
                Log::error('Failed to parse section content AI response', ['raw_response' => substr($result, 0, 500)]);
                // Fallback to default content with title/subtitle
                $jsonResult = $defaultContent;
                $jsonResult['title'] = $defaultTitle;
                $jsonResult['subtitle'] = $defaultSubtitle;
            }

            // Extract title and subtitle from response (AI should include them)
            $generatedTitle = $jsonResult['title'] ?? $defaultTitle;
            $generatedSubtitle = $jsonResult['subtitle'] ?? $defaultSubtitle;

            // Remove title/subtitle from content array (they're stored separately in website_builder)
            unset($jsonResult['title'], $jsonResult['subtitle']);

            // Validate and normalize AI content against content_schema
            $normalizedContent = $this->normalizeContentAgainstSchema($jsonResult, $contentSchema, $defaultContent);

            // Log normalized content for debugging
            Log::debug("Normalized content for section '{$sectionKey}'", [
                'has_items' => isset($normalizedContent['items']),
                'items_count' => isset($normalizedContent['items']) ? count($normalizedContent['items']) : 0,
                'first_item_keys' => isset($normalizedContent['items'][0]) ? array_keys($normalizedContent['items'][0]) : [],
                'first_item_rating' => $normalizedContent['items'][0]['rating'] ?? 'NOT_SET',
            ]);

            // Generate sample_data for image fields (with AI-generated URLs)
            $sampleData = $this->generateSampleDataForImages(
                $contentSchema,
                $normalizedContent,
                $sectionKey,
                $category,
                $aiSetting
            );

            return ApiResponse::make('Success', [
                'content' => $normalizedContent,
                'title' => $generatedTitle,
                'subtitle' => $generatedSubtitle,
                'sample_data' => !empty($sampleData) ? $sampleData : null,
                'section_key' => $sectionKey,
                'lang_key' => $langKey
            ]);
        } catch (\Exception $e) {
            // Generate sample_data for default content as well
            $sampleData = $this->generateSampleDataForImages(
                $contentSchema,
                $defaultContent,
                $sectionKey,
                $category
            );

            // On error, return default content with title/subtitle
            return ApiResponse::make('Using default content', [
                'content' => $defaultContent,
                'title' => $defaultTitle,
                'subtitle' => $defaultSubtitle,
                'sample_data' => !empty($sampleData) ? $sampleData : null,
                'section_key' => $sectionKey,
                'lang_key' => $langKey,
                'fallback' => true
            ]);
        }
    }

    /**
     * Save AI-generated sections to website_builder table
     */
    public function saveGeneratedSections(Request $request)
    {
        $this->checkAiFeaturesAccess();

        $websiteXid = $request->website_xid;
        $sections = $request->sections ?? [];

        if (empty($websiteXid)) {
            throw new ApiException('Website ID is required.');
        }

        // Decode website ID
        $websiteId = \Vinkla\Hashids\Facades\Hashids::decode($websiteXid);
        if (empty($websiteId)) {
            throw new ApiException('Invalid website ID.');
        }
        $websiteId = $websiteId[0];

        // Get the website
        $website = Website::find($websiteId);
        if (!$website) {
            throw new ApiException('Website not found.');
        }

        // Get all website languages
        $websiteLanguages = WebsiteLanguage::where('website_id', $websiteId)
            ->where('enabled', true)
            ->get()
            ->keyBy('key');

        // Get website pages
        $websitePages = WebsitePage::where('website_id', $websiteId)
            ->get()
            ->keyBy('page_key');

        DB::beginTransaction();

        try {
            $createdCount = 0;
            $updatedCount = 0;

            foreach ($sections as $sectionData) {
                $pageKey = $sectionData['page_key'] ?? null;
                $sectionKey = $sectionData['section_key'] ?? null;
                $layout = $sectionData['layout'] ?? $sectionKey;
                $sortOrder = $sectionData['sort_order'] ?? 0;
                $content = $sectionData['content'] ?? [];
                $title = $sectionData['title'] ?? '';
                $subtitle = $sectionData['subtitle'] ?? '';
                $langKey = $sectionData['lang_key'] ?? 'en';
                $sampleData = $sectionData['sample_data'] ?? null;

                if (!$pageKey || !$sectionKey) {
                    continue;
                }

                $websitePage = $websitePages->get($pageKey);
                $websiteLang = $websiteLanguages->get($langKey);

                if (!$websitePage || !$websiteLang) {
                    continue;
                }

                // Get theme section for schema
                $themeSection = ThemeSection::where('section_key', $sectionKey)
                    ->where('is_active', true)
                    ->first();

                if (!$themeSection) {
                    Log::warning('Theme section not found or inactive', [
                        'section_key' => $sectionKey,
                        'page_key' => $pageKey,
                    ]);
                    continue;
                }

                // Normalize content against theme section's content_schema
                if (!empty($content) && is_array($content)) {
                    $content = $this->normalizeContentAgainstSchema(
                        $content,
                        $themeSection->content_schema ?? [],
                        $themeSection->default_content ?? []
                    );
                }

                // Check if section already exists
                $existingSection = WebsiteBuilder::where('company_id', $website->company_id)
                    ->where('website_id', $websiteId)
                    ->where('website_page_id', $websitePage->id)
                    ->where('website_language_id', $websiteLang->id)
                    ->where('section_key', $sectionKey)
                    ->first();

                if ($existingSection) {
                    // Update existing section
                    $updateData = [
                        'section_layout' => $layout,
                        'title' => $title ?: $existingSection->title,
                        'subtitle' => $subtitle ?: $existingSection->subtitle,
                        'content' => !empty($content) ? $content : $existingSection->content,
                        'sort_order' => $sortOrder,
                    ];

                    // Update sample_data if provided
                    if ($sampleData !== null) {
                        $updateData['sample_data'] = $sampleData;
                    }

                    $existingSection->update($updateData);
                    $updatedCount++;
                } else {
                    // Create new section
                    WebsiteBuilder::create([
                        'company_id' => $website->company_id,
                        'website_id' => $websiteId,
                        'website_page_id' => $websitePage->id,
                        'website_language_id' => $websiteLang->id,
                        'section_key' => $sectionKey,
                        'section_layout' => $layout,
                        'title' => $title ?: ($themeSection->default_title ?? $themeSection->name),
                        'subtitle' => $subtitle ?: ($themeSection->default_subtitle ?? ''),
                        'content' => !empty($content) ? $content : ($themeSection->default_content ?? []),
                        'content_schema' => $themeSection->content_schema ?? [],
                        'sample_data' => $sampleData,
                        'sort_order' => $sortOrder,
                        'is_white_background' => $themeSection->is_white_background ?? false,
                    ]);
                    $createdCount++;
                }
            }

            DB::commit();

            return ApiResponse::make('Sections saved successfully', [
                'created' => $createdCount,
                'updated' => $updatedCount,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to save generated sections', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'website_xid' => $websiteXid,
            ]);
            throw new ApiException('Failed to save sections: ' . $e->getMessage());
        }
    }

    /**
     * Build schema with content length hints from default content
     */
    private function buildSchemaWithContentHints(array $schema, array $defaultContent): array
    {
        $result = [];

        foreach ($schema as $field) {
            $fieldName = $field['name'] ?? '';
            $fieldType = $field['type'] ?? 'text';

            $fieldWithHint = $field;

            // Add content length hint based on default content
            if (isset($defaultContent[$fieldName])) {
                $defaultValue = $defaultContent[$fieldName];

                if (is_string($defaultValue)) {
                    $fieldWithHint['_hint_length'] = strlen($defaultValue) . ' characters';
                } elseif (is_array($defaultValue)) {
                    $fieldWithHint['_hint_items_count'] = count($defaultValue) . ' items';

                    // For repeater fields, add hints for nested fields
                    if ($fieldType === 'repeater' && !empty($defaultValue[0]) && isset($field['fields'])) {
                        $fieldWithHint['fields'] = $this->buildSchemaWithContentHints(
                            $field['fields'],
                            $defaultValue[0]
                        );
                    }
                }
            }

            $result[] = $fieldWithHint;
        }

        return $result;
    }

    /**
     * Extract image fields from content_schema and generate sample_data with dummy images
     * First asks AI for relevant image search terms, then builds Unsplash URLs
     *
     * @param array $contentSchema The content schema from theme_sections
     * @param array $generatedContent The AI-generated content
     * @param string $sectionKey The section key (hero, services, etc.)
     * @param string $category The website category for context-aware images
     * @param mixed $aiSetting The AI settings for making API calls (optional)
     * @return array The sample_data with dummy image URLs
     */
    private function generateSampleDataForImages(array $contentSchema, array $generatedContent, string $sectionKey, string $category, $aiSetting = null): array
    {
        $sampleData = [];

        // Collect all image fields that need URLs
        $imageFieldsToGenerate = $this->collectImageFieldsInfo($contentSchema, $generatedContent, $sectionKey);

        if (empty($imageFieldsToGenerate)) {
            return $sampleData;
        }

        // Try to get AI-generated image URLs
        $aiGeneratedUrls = [];
        if ($aiSetting) {
            $aiGeneratedUrls = $this->getAiGeneratedImageUrls($aiSetting, $imageFieldsToGenerate, $category, $sectionKey);
        }

        // Fallback images
        $categoryImages = $this->getCategoryImageUrls($category);
        $genericImages = $this->getGenericImageUrls();

        // Process each image field
        foreach ($contentSchema as $field) {
            $fieldName = $field['name'] ?? '';
            $fieldType = $field['type'] ?? 'text';

            if ($fieldType === 'image') {
                // Top-level image field - use AI URL or fallback
                $sampleData[$fieldName] = $aiGeneratedUrls[$fieldName] ?? $this->getDummyImageUrl($fieldName, $sectionKey, $categoryImages, $genericImages);
            } elseif ($fieldType === 'repeater' && isset($field['fields'])) {
                // Check for image fields inside repeater
                $repeaterImageFields = $this->extractImageFieldsFromRepeater($field['fields']);

                if (!empty($repeaterImageFields)) {
                    // Get count of items from generated content
                    $itemCount = isset($generatedContent[$fieldName]) && is_array($generatedContent[$fieldName])
                        ? count($generatedContent[$fieldName])
                        : 3; // Default to 3 items

                    // Generate sample images for each item
                    $sampleImages = [];
                    for ($i = 0; $i < $itemCount; $i++) {
                        foreach ($repeaterImageFields as $imgField) {
                            $imgFieldName = $imgField['name'] ?? '';
                            $aiKey = "{$fieldName}.{$imgFieldName}.{$i}";

                            // Use AI-generated URL or fallback
                            $sampleImages[$imgFieldName][] = $aiGeneratedUrls[$aiKey] ?? $this->getDummyImageUrl(
                                $imgFieldName,
                                $sectionKey,
                                $categoryImages,
                                $genericImages,
                                $i
                            );
                        }
                    }

                    if (!empty($sampleImages)) {
                        $sampleData[$fieldName] = $sampleImages;
                    }
                }
            }
        }

        return $sampleData;
    }

    /**
     * Collect information about all image fields that need URLs
     */
    private function collectImageFieldsInfo(array $contentSchema, array $generatedContent, string $sectionKey): array
    {
        $imageFields = [];

        foreach ($contentSchema as $field) {
            $fieldName = $field['name'] ?? '';
            $fieldType = $field['type'] ?? 'text';
            $fieldLabel = $field['label'] ?? $fieldName;

            if ($fieldType === 'image') {
                $imageFields[] = [
                    'key' => $fieldName,
                    'name' => $fieldName,
                    'label' => $fieldLabel,
                    'section' => $sectionKey,
                    'is_repeater' => false,
                ];
            } elseif ($fieldType === 'repeater' && isset($field['fields'])) {
                $repeaterImageFields = $this->extractImageFieldsFromRepeater($field['fields']);

                if (!empty($repeaterImageFields)) {
                    $itemCount = isset($generatedContent[$fieldName]) && is_array($generatedContent[$fieldName])
                        ? count($generatedContent[$fieldName])
                        : 3;

                    for ($i = 0; $i < $itemCount; $i++) {
                        // Get context from generated content if available
                        $itemContext = '';
                        if (isset($generatedContent[$fieldName][$i])) {
                            $item = $generatedContent[$fieldName][$i];
                            // Try to get title/name/heading for context
                            $itemContext = $item['title'] ?? $item['name'] ?? $item['heading'] ?? '';
                        }

                        foreach ($repeaterImageFields as $imgField) {
                            $imgFieldName = $imgField['name'] ?? '';
                            $imgFieldLabel = $imgField['label'] ?? $imgFieldName;

                            $imageFields[] = [
                                'key' => "{$fieldName}.{$imgFieldName}.{$i}",
                                'name' => $imgFieldName,
                                'label' => $imgFieldLabel,
                                'section' => $sectionKey,
                                'is_repeater' => true,
                                'repeater_field' => $fieldName,
                                'index' => $i,
                                'item_context' => $itemContext,
                            ];
                        }
                    }
                }
            }
        }

        return $imageFields;
    }

    /**
     * Get image URLs for image fields using category-based images from getCategoryImageUrls
     * Uses pre-defined high-quality, relevant business images for each category
     *
     * @param mixed $aiSetting AI setting (kept for API compatibility, not used)
     * @param array $imageFields Array of image field definitions
     * @param string $category Business category
     * @param string $sectionKey Section key
     * @return array URLs mapped to field keys
     */
    private function getAiGeneratedImageUrls($aiSetting, array $imageFields, string $category, string $sectionKey): array
    {
        // Note: $aiSetting parameter kept for API compatibility but not used
        // Images are now sourced from getCategoryImageUrls for better relevance
        $urls = [];

        if (empty($imageFields)) {
            return $urls;
        }

        // Get category-specific and generic images
        $categoryImages = $this->getCategoryImageUrls($category);
        $genericImages = $this->getGenericImageUrls();

        // Assign appropriate images to each field based on field type and context
        foreach ($imageFields as $index => $field) {
            $fieldKey = $field['key'];
            $fieldName = $field['name'] ?? $fieldKey;

            // Use the getDummyImageUrl method which has intelligent field-type matching
            $urls[$fieldKey] = $this->getDummyImageUrl(
                $fieldName,
                $sectionKey,
                $categoryImages,
                $genericImages,
                $index
            );
        }

        return $urls;
    }

    /**
     * Extract image fields from repeater fields
     */
    private function extractImageFieldsFromRepeater(array $fields): array
    {
        $imageFields = [];

        foreach ($fields as $field) {
            $fieldType = $field['type'] ?? 'text';

            if ($fieldType === 'image') {
                $imageFields[] = $field;
            } elseif ($fieldType === 'repeater' && isset($field['fields'])) {
                // Nested repeater
                $nestedImages = $this->extractImageFieldsFromRepeater($field['fields']);
                $imageFields = array_merge($imageFields, $nestedImages);
            }
        }

        return $imageFields;
    }

    /**
     * Get dummy image URL based on field name and context
     */
    private function getDummyImageUrl(string $fieldName, string $sectionKey, array $categoryImages, array $genericImages, int $index = 0): string
    {
        // Map field names to appropriate image types
        $fieldNameLower = strtolower($fieldName);
        $sectionKeyLower = strtolower($sectionKey);

        // Hero images
        if (str_contains($fieldNameLower, 'hero') || ($sectionKeyLower === 'hero' && str_contains($fieldNameLower, 'image'))) {
            return $categoryImages['hero'] ?? $genericImages['hero'];
        }

        // Avatar images
        if (str_contains($fieldNameLower, 'avatar')) {
            $avatars = $genericImages['avatars'] ?? [];
            return $avatars[$index % count($avatars)] ?? $genericImages['avatar'];
        }

        // Team/person photos
        if (str_contains($fieldNameLower, 'photo') || ($sectionKeyLower === 'team' && str_contains($fieldNameLower, 'image'))) {
            $teamPhotos = $genericImages['team'] ?? [];
            return $teamPhotos[$index % count($teamPhotos)] ?? $genericImages['person'];
        }

        // Logo images
        if (str_contains($fieldNameLower, 'logo')) {
            $logos = $genericImages['logos'] ?? [];
            return $logos[$index % count($logos)] ?? $genericImages['logo'];
        }

        // Gallery images
        if ($sectionKeyLower === 'gallery' || str_contains($fieldNameLower, 'gallery')) {
            $gallery = $categoryImages['gallery'] ?? $genericImages['gallery'];
            return $gallery[$index % count($gallery)] ?? $genericImages['image'];
        }

        // Service images
        if ($sectionKeyLower === 'services' || str_contains($fieldNameLower, 'service')) {
            $services = $categoryImages['services'] ?? $genericImages['services'];
            return $services[$index % count($services)] ?? $genericImages['image'];
        }

        // Background images
        if (str_contains($fieldNameLower, 'background')) {
            return $categoryImages['background'] ?? $genericImages['background'];
        }

        // Thumbnail images
        if (str_contains($fieldNameLower, 'thumbnail')) {
            return $categoryImages['thumbnail'] ?? $genericImages['thumbnail'];
        }

        // App/mockup images
        if (str_contains($fieldNameLower, 'app') || str_contains($fieldNameLower, 'mockup')) {
            return $genericImages['app_mockup'];
        }

        // QR code
        if (str_contains($fieldNameLower, 'qr')) {
            return $genericImages['qr_code'];
        }

        // Default: use category-specific or generic image
        return $categoryImages['default'] ?? $genericImages['image'];
    }

    /**
     * Get category-specific image URLs
     */
    private function getCategoryImageUrls(string $category): array
    {
        $categoryImages = [
            'Laundry & Dry Cleaning' => [
                'hero' => 'https://images.unsplash.com/photo-1517677208171-0bc6725a3e60?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1545173168-9f1947eebb7f?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1517677208171-0bc6725a3e60?w=800&q=80',
                    'https://images.unsplash.com/photo-1582735689369-4fe89db7114c?w=800&q=80',
                    'https://images.unsplash.com/photo-1610557892470-55d9e80c0bce?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1517677208171-0bc6725a3e60?w=800&q=80',
            ],
            'Restaurant & Food' => [
                'hero' => 'https://images.unsplash.com/photo-1517248135467-4c7edcad34c4?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1414235077428-338989a2e8c0?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1504674900247-0877df9cc836?w=800&q=80',
                    'https://images.unsplash.com/photo-1414235077428-338989a2e8c0?w=800&q=80',
                    'https://images.unsplash.com/photo-1476224203421-9ac39bcb3327?w=800&q=80',
                ],
                'gallery' => [
                    'https://images.unsplash.com/photo-1504674900247-0877df9cc836?w=800&q=80',
                    'https://images.unsplash.com/photo-1414235077428-338989a2e8c0?w=800&q=80',
                    'https://images.unsplash.com/photo-1476224203421-9ac39bcb3327?w=800&q=80',
                    'https://images.unsplash.com/photo-1540189549336-e6e99c3679fe?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1517248135467-4c7edcad34c4?w=800&q=80',
            ],
            'Salon & Spa' => [
                'hero' => 'https://images.unsplash.com/photo-1560066984-138dadb4c035?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1519415510236-718bdfcd89c8?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1560066984-138dadb4c035?w=800&q=80',
                    'https://images.unsplash.com/photo-1519415510236-718bdfcd89c8?w=800&q=80',
                    'https://images.unsplash.com/photo-1582095133179-bfd08e2fc6b3?w=800&q=80',
                ],
                'gallery' => [
                    'https://images.unsplash.com/photo-1560066984-138dadb4c035?w=800&q=80',
                    'https://images.unsplash.com/photo-1519415510236-718bdfcd89c8?w=800&q=80',
                    'https://images.unsplash.com/photo-1582095133179-bfd08e2fc6b3?w=800&q=80',
                    'https://images.unsplash.com/photo-1516975080664-ed2fc6a32937?w=800&q=80',
                    'https://images.unsplash.com/photo-1604654894610-df63bc536371?w=800&q=80',
                    'https://images.unsplash.com/photo-1522337660859-02fbefca4702?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1560066984-138dadb4c035?w=800&q=80',
            ],
            'Healthcare & Medical' => [
                'hero' => 'https://images.unsplash.com/photo-1519494026892-80bbd2d6fd0d?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1504813184591-01572f98c85f?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1519494026892-80bbd2d6fd0d?w=800&q=80',
                    'https://images.unsplash.com/photo-1504813184591-01572f98c85f?w=800&q=80',
                    'https://images.unsplash.com/photo-1538108149393-fbbd81895907?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1519494026892-80bbd2d6fd0d?w=800&q=80',
            ],
            'Fitness & Gym' => [
                'hero' => 'https://images.unsplash.com/photo-1534438327276-14e5300c3a48?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1571902943202-507ec2618e8f?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1534438327276-14e5300c3a48?w=800&q=80',
                    'https://images.unsplash.com/photo-1571902943202-507ec2618e8f?w=800&q=80',
                    'https://images.unsplash.com/photo-1540497077202-7c8a3999166f?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1534438327276-14e5300c3a48?w=800&q=80',
            ],
            'Real Estate' => [
                'hero' => 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1512917774080-9991f1c4c750?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=800&q=80',
                    'https://images.unsplash.com/photo-1512917774080-9991f1c4c750?w=800&q=80',
                    'https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?w=800&q=80',
                ],
                'gallery' => [
                    'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=800&q=80',
                    'https://images.unsplash.com/photo-1512917774080-9991f1c4c750?w=800&q=80',
                    'https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?w=800&q=80',
                    'https://images.unsplash.com/photo-1600607687939-ce8a6c25118c?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=800&q=80',
            ],
            'E-Commerce' => [
                'hero' => 'https://images.unsplash.com/photo-1441986300917-64674bd600d8?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1472851294608-062f824d29cc?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=800&q=80',
                    'https://images.unsplash.com/photo-1472851294608-062f824d29cc?w=800&q=80',
                    'https://images.unsplash.com/photo-1556742049-0cfed4f6a45d?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=800&q=80',
            ],
            'Education & Training' => [
                'hero' => 'https://images.unsplash.com/photo-1427504494785-3a9ca7044f45?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1523050854058-8df90110c9f1?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1427504494785-3a9ca7044f45?w=800&q=80',
                    'https://images.unsplash.com/photo-1523050854058-8df90110c9f1?w=800&q=80',
                    'https://images.unsplash.com/photo-1509062522246-3755977927d7?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1427504494785-3a9ca7044f45?w=800&q=80',
            ],
            'Travel & Tourism' => [
                'hero' => 'https://images.unsplash.com/photo-1488646953014-85cb44e25828?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1476514525535-07fb3b4ae5f1?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1488646953014-85cb44e25828?w=800&q=80',
                    'https://images.unsplash.com/photo-1476514525535-07fb3b4ae5f1?w=800&q=80',
                    'https://images.unsplash.com/photo-1469474968028-56623f02e42e?w=800&q=80',
                ],
                'gallery' => [
                    'https://images.unsplash.com/photo-1488646953014-85cb44e25828?w=800&q=80',
                    'https://images.unsplash.com/photo-1476514525535-07fb3b4ae5f1?w=800&q=80',
                    'https://images.unsplash.com/photo-1469474968028-56623f02e42e?w=800&q=80',
                    'https://images.unsplash.com/photo-1507525428034-b723cf961d3e?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1488646953014-85cb44e25828?w=800&q=80',
            ],
            'Technology & IT' => [
                'hero' => 'https://images.unsplash.com/photo-1518770660439-4636190af475?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1451187580459-43490279c0fa?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1518770660439-4636190af475?w=800&q=80',
                    'https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=800&q=80',
                    'https://images.unsplash.com/photo-1550751827-4bd374c3f58b?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1518770660439-4636190af475?w=800&q=80',
            ],
            'Consulting & Professional Services' => [
                'hero' => 'https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1521737711867-e3b97375f902?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?w=800&q=80',
                    'https://images.unsplash.com/photo-1521737711867-e3b97375f902?w=800&q=80',
                    'https://images.unsplash.com/photo-1557804506-669a67965ba0?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?w=800&q=80',
            ],
            'Photography & Media' => [
                'hero' => 'https://images.unsplash.com/photo-1542038784456-1ea8e935640e?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1542038784456-1ea8e935640e?w=800&q=80',
                    'https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=800&q=80',
                    'https://images.unsplash.com/photo-1519741497674-611481863552?w=800&q=80',
                ],
                'gallery' => [
                    'https://images.unsplash.com/photo-1542038784456-1ea8e935640e?w=800&q=80',
                    'https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=800&q=80',
                    'https://images.unsplash.com/photo-1519741497674-611481863552?w=800&q=80',
                    'https://images.unsplash.com/photo-1554415707-6e8cfc93fe23?w=800&q=80',
                    'https://images.unsplash.com/photo-1531746790731-6c087fecd65a?w=800&q=80',
                    'https://images.unsplash.com/photo-1452587925148-ce544e77e70d?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1542038784456-1ea8e935640e?w=800&q=80',
            ],
            'Event Management' => [
                'hero' => 'https://images.unsplash.com/photo-1511578314322-379afb476865?q=80&w=1920&auto=format&fit=crop',
                'background' => 'https://images.unsplash.com/photo-1540575467063-178a50c2df87?q=80&w=1920&auto=format&fit=crop',
                'services' => [
                    'https://images.unsplash.com/photo-1511578314322-379afb476865?w=800&q=80',
                    'https://images.unsplash.com/photo-1540575467063-178a50c2df87?w=800&q=80',
                    'https://images.unsplash.com/photo-1464366400600-7168b8af9bc3?w=800&q=80',
                ],
                'gallery' => [
                    'https://images.unsplash.com/photo-1511578314322-379afb476865?w=800&q=80',
                    'https://images.unsplash.com/photo-1540575467063-178a50c2df87?w=800&q=80',
                    'https://images.unsplash.com/photo-1464366400600-7168b8af9bc3?w=800&q=80',
                    'https://images.unsplash.com/photo-1505236858219-8359eb29e329?w=800&q=80',
                    'https://images.unsplash.com/photo-1478147427282-58a87a120781?w=800&q=80',
                    'https://images.unsplash.com/photo-1530103862676-de8c9debad1d?w=800&q=80',
                ],
                'default' => 'https://images.unsplash.com/photo-1511578314322-379afb476865?w=800&q=80',
            ],
        ];

        return $categoryImages[$category] ?? [
            'hero' => 'https://images.unsplash.com/photo-1497366216548-37526070297c?q=80&w=1920&auto=format&fit=crop',
            'background' => 'https://images.unsplash.com/photo-1557683316-973673baf926?q=80&w=1920&auto=format&fit=crop',
            'services' => [
                'https://images.unsplash.com/photo-1497366216548-37526070297c?w=800&q=80',
                'https://images.unsplash.com/photo-1557683316-973673baf926?w=800&q=80',
                'https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=800&q=80',
            ],
            'default' => 'https://images.unsplash.com/photo-1497366216548-37526070297c?w=800&q=80',
        ];
    }

    /**
     * Get generic image URLs for common field types
     */
    private function getGenericImageUrls(): array
    {
        return [
            'hero' => 'https://images.unsplash.com/photo-1497366216548-37526070297c?q=80&w=1920&auto=format&fit=crop',
            'background' => 'https://images.unsplash.com/photo-1557683316-973673baf926?q=80&w=1920&auto=format&fit=crop',
            'thumbnail' => 'https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=800&q=80',
            'image' => 'https://images.unsplash.com/photo-1497366216548-37526070297c?w=800&q=80',
            'avatar' => 'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=150&q=80',
            'avatars' => [
                'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=150&q=80',
                'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=150&q=80',
                'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=150&q=80',
                'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=150&q=80',
                'https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=150&q=80',
            ],
            'person' => 'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=400&q=80',
            'team' => [
                'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=400&q=80',
                'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=400&q=80',
                'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&q=80',
                'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=400&q=80',
                'https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=400&q=80',
                'https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=400&q=80',
            ],
            'logo' => 'https://images.unsplash.com/photo-1599305445671-ac291c95aaa9?w=200&q=80',
            'logos' => [
                'https://images.unsplash.com/photo-1599305445671-ac291c95aaa9?w=200&q=80',
                'https://images.unsplash.com/photo-1611162616305-c69b3fa7fbe0?w=200&q=80',
                'https://images.unsplash.com/photo-1611162618071-b39a2ec055fb?w=200&q=80',
                'https://images.unsplash.com/photo-1611605698335-8b1569810432?w=200&q=80',
            ],
            'services' => [
                'https://images.unsplash.com/photo-1497366216548-37526070297c?w=800&q=80',
                'https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=800&q=80',
                'https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?w=800&q=80',
            ],
            'gallery' => [
                'https://images.unsplash.com/photo-1497366216548-37526070297c?w=800&q=80',
                'https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=800&q=80',
                'https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?w=800&q=80',
                'https://images.unsplash.com/photo-1557683316-973673baf926?w=800&q=80',
            ],
            'app_mockup' => 'https://images.unsplash.com/photo-1512941937669-90a1b58e7e9c?w=800&q=80',
            'qr_code' => 'https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=https://example.com',
        ];
    }

    /**
     * Normalize AI-generated content against the content_schema from theme_sections
     * This ensures all expected fields exist and have proper types
     *
     * @param array $aiContent The AI-generated content
     * @param array $contentSchema The content schema from theme_sections
     * @param array $defaultContent The default content from theme_sections
     * @return array Normalized content
     */
    private function normalizeContentAgainstSchema(array $aiContent, array $contentSchema, array $defaultContent): array
    {
        // If content_schema is empty, just merge AI content with default content
        if (empty($contentSchema)) {
            Log::warning('Empty content_schema, merging AI content with defaults');
            return array_merge($defaultContent, $aiContent);
        }

        $normalized = [];

        foreach ($contentSchema as $field) {
            $fieldName = $field['name'] ?? '';
            $fieldType = $field['type'] ?? 'text';

            if (empty($fieldName)) {
                continue;
            }

            // Get AI value or default value
            $aiValue = $aiContent[$fieldName] ?? null;
            $defaultValue = $defaultContent[$fieldName] ?? null;

            // Normalize based on field type
            switch ($fieldType) {
                case 'text':
                case 'textarea':
                    // Text fields should be strings
                    if (is_string($aiValue) && !empty($aiValue)) {
                        $normalized[$fieldName] = $aiValue;
                    } else {
                        $normalized[$fieldName] = is_string($defaultValue) ? $defaultValue : '';
                    }
                    break;

                case 'number':
                    // Number fields should be numeric
                    if (is_numeric($aiValue)) {
                        $normalized[$fieldName] = (int) $aiValue;
                    } elseif (is_numeric($defaultValue)) {
                        $normalized[$fieldName] = (int) $defaultValue;
                    } else {
                        $normalized[$fieldName] = 0;
                    }
                    break;

                case 'switch':
                    // Switch fields should be boolean
                    if (is_bool($aiValue)) {
                        $normalized[$fieldName] = $aiValue;
                    } elseif (is_string($aiValue)) {
                        $normalized[$fieldName] = in_array(strtolower($aiValue), ['true', '1', 'yes']);
                    } else {
                        $normalized[$fieldName] = is_bool($defaultValue) ? $defaultValue : false;
                    }
                    break;

                case 'image':
                    // Image fields should be null (images are uploaded separately)
                    $normalized[$fieldName] = null;
                    break;

                case 'icon':
                    // Icon fields should be strings with lucide: prefix
                    if (is_string($aiValue) && !empty($aiValue)) {
                        // Ensure lucide: prefix
                        if (!str_starts_with($aiValue, 'lucide:')) {
                            $aiValue = 'lucide:' . $aiValue;
                        }
                        $normalized[$fieldName] = $aiValue;
                    } else {
                        $normalized[$fieldName] = is_string($defaultValue) ? $defaultValue : 'lucide:star';
                    }
                    break;

                case 'select':
                    // Select fields should be one of the available options
                    $options = $field['options'] ?? [];
                    $validValues = array_column($options, 'value');
                    if (is_string($aiValue) && in_array($aiValue, $validValues)) {
                        $normalized[$fieldName] = $aiValue;
                    } elseif (is_string($defaultValue)) {
                        $normalized[$fieldName] = $defaultValue;
                    } elseif (!empty($validValues)) {
                        $normalized[$fieldName] = $validValues[0];
                    } else {
                        $normalized[$fieldName] = '';
                    }
                    break;

                case 'repeater':
                case 'nested_repeater':
                    // Repeater fields should be arrays
                    $repeaterFields = $field['fields'] ?? [];
                    $aiItems = is_array($aiValue) ? $aiValue : [];
                    $defaultItems = is_array($defaultValue) ? $defaultValue : [];

                    // Always use first default item as template for all items
                    // This ensures all required fields have default values
                    $templateItem = !empty($defaultItems) ? $defaultItems[0] : [];

                    // If AI returned items, normalize each item
                    if (!empty($aiItems)) {
                        $normalizedItems = [];
                        foreach ($aiItems as $index => $item) {
                            if (is_array($item)) {
                                // Use template item to ensure all fields have defaults
                                $normalizedItems[] = $this->normalizeRepeaterItem($item, $repeaterFields, $templateItem);
                            }
                        }
                        $normalized[$fieldName] = $normalizedItems;
                    } elseif (!empty($defaultItems)) {
                        // Use default items if AI returned nothing
                        $normalized[$fieldName] = $defaultItems;
                    } else {
                        $normalized[$fieldName] = [];
                    }
                    break;

                default:
                    // Unknown type - use AI value or default
                    $normalized[$fieldName] = $aiValue ?? $defaultValue;
                    break;
            }
        }

        // Also include any fields from AI that are not in schema (might be valid additional fields)
        foreach ($aiContent as $key => $value) {
            if (!isset($normalized[$key]) && !in_array($key, ['title', 'subtitle'])) {
                $normalized[$key] = $value;
            }
        }

        return $normalized;
    }

    /**
     * Normalize a single repeater item against its field definitions
     *
     * @param array $item The AI-generated item
     * @param array $fields The field definitions for the repeater
     * @param array $defaultItem Default values for the item
     * @return array Normalized item
     */
    private function normalizeRepeaterItem(array $item, array $fields, array $defaultItem): array
    {
        // If no fields defined, just merge with defaults
        if (empty($fields)) {
            return array_merge($defaultItem, $item);
        }

        $normalized = [];

        foreach ($fields as $field) {
            $fieldName = $field['name'] ?? '';
            $fieldType = $field['type'] ?? 'text';

            if (empty($fieldName)) {
                continue;
            }

            $aiValue = $item[$fieldName] ?? null;
            $defaultValue = $defaultItem[$fieldName] ?? null;

            switch ($fieldType) {
                case 'text':
                case 'textarea':
                    if (is_string($aiValue) && !empty($aiValue)) {
                        $normalized[$fieldName] = $aiValue;
                    } else {
                        $normalized[$fieldName] = is_string($defaultValue) ? $defaultValue : '';
                    }
                    break;

                case 'number':
                    if (is_numeric($aiValue)) {
                        $normalized[$fieldName] = (int) $aiValue;
                    } elseif (is_numeric($defaultValue)) {
                        $normalized[$fieldName] = (int) $defaultValue;
                    } else {
                        // Default to 5 for rating fields, 0 for others
                        $defaultNum = (strtolower($fieldName) === 'rating') ? 5 : 0;
                        $normalized[$fieldName] = $defaultNum;
                        Log::debug("Normalizer: Field '{$fieldName}' missing/invalid, using default: {$defaultNum}");
                    }
                    break;

                case 'switch':
                    if (is_bool($aiValue)) {
                        $normalized[$fieldName] = $aiValue;
                    } elseif (is_string($aiValue)) {
                        $normalized[$fieldName] = in_array(strtolower($aiValue), ['true', '1', 'yes']);
                    } else {
                        $normalized[$fieldName] = is_bool($defaultValue) ? $defaultValue : false;
                    }
                    break;

                case 'image':
                    $normalized[$fieldName] = null;
                    break;

                case 'icon':
                    if (is_string($aiValue) && !empty($aiValue)) {
                        if (!str_starts_with($aiValue, 'lucide:')) {
                            $aiValue = 'lucide:' . $aiValue;
                        }
                        $normalized[$fieldName] = $aiValue;
                    } else {
                        $normalized[$fieldName] = is_string($defaultValue) ? $defaultValue : 'lucide:star';
                    }
                    break;

                case 'repeater':
                case 'nested_repeater':
                    // Nested repeater
                    $nestedFields = $field['fields'] ?? [];
                    $aiNested = is_array($aiValue) ? $aiValue : [];
                    $defaultNested = is_array($defaultValue) ? $defaultValue : [];

                    // Use first default item as template for all items
                    $templateNestedItem = !empty($defaultNested) ? $defaultNested[0] : [];

                    if (!empty($aiNested)) {
                        $normalizedNested = [];
                        foreach ($aiNested as $idx => $nestedItem) {
                            if (is_array($nestedItem)) {
                                $normalizedNested[] = $this->normalizeRepeaterItem($nestedItem, $nestedFields, $templateNestedItem);
                            }
                        }
                        $normalized[$fieldName] = $normalizedNested;
                    } else {
                        $normalized[$fieldName] = $defaultNested;
                    }
                    break;

                case 'select':
                    // Select fields should be one of the available options
                    $options = $field['options'] ?? [];
                    $validValues = array_column($options, 'value');
                    if (is_string($aiValue) && in_array($aiValue, $validValues)) {
                        $normalized[$fieldName] = $aiValue;
                    } elseif (is_string($defaultValue)) {
                        $normalized[$fieldName] = $defaultValue;
                    } elseif (!empty($validValues)) {
                        $normalized[$fieldName] = $validValues[0];
                    } else {
                        $normalized[$fieldName] = '';
                    }
                    break;

                default:
                    $normalized[$fieldName] = $aiValue ?? $defaultValue;
                    break;
            }
        }

        // Include any additional fields from AI item
        foreach ($item as $key => $value) {
            if (!isset($normalized[$key])) {
                $normalized[$key] = $value;
            }
        }

        return $normalized;
    }
}
