<?php

namespace App\Http\Controllers\Api\Common;

use App\Classes\Common;
use App\Http\Controllers\ApiBaseController;
use Examyou\RestAPI\ApiResponse;
use Examyou\RestAPI\Exceptions\ApiException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class DatabaseBackupController extends ApiBaseController
{
    public function databaseBackups(Request $request)
    {
        $data = $this->allDatabases();

        return ApiResponse::make('Data fetched', $data);
    }

    public function allDatabases()
    {
        $data = [];

        // Supported backup file extensions
        $supportedExtensions = ['sql', 'zip', 'gz'];

        $allFiles = collect(Storage::disk('backup')->listContents('/'))
            ->filter(function ($file) use ($supportedExtensions) {
                $extension = File::extension($file['path']);
                // Handle .sql.gz files
                if ($extension === 'gz') {
                    return true;
                }
                return in_array($extension, $supportedExtensions);
            })
            ->sortByDesc('last_modified')
            ->values()->all();

        foreach ($allFiles as $currentFile) {
            $fileName = $currentFile['path'];
            $item['id'] = substr($fileName, 0, 8);
            $item['name'] = $fileName;
            $size = Common::formatSizeUnits($currentFile['file_size']);
            $item['size'] = $size;
            $item['url'] = route('api.settings.download-backups.v1', [$item['id']]);

            // Determine file type for display
            if (str_ends_with($fileName, '.sql.gz')) {
                $item['type'] = 'gzip';
            } elseif (str_ends_with($fileName, '.zip')) {
                $item['type'] = 'zip';
            } else {
                $item['type'] = 'sql';
            }

            $data[] = $item;
        }

        return [
            'data' => $data,
            'totals' => sizeof($data),
        ];
    }

    public function generateBackups(Request $request)
    {
        if (env('APP_ENV') == 'production') {
            throw new ApiException('Not Allowed In Demo Mode');
        }

        // Get backup type from request (default: sql)
        $backupType = $request->input('backup_type', 'sql');
        $validTypes = ['sql', 'zip', 'gzip'];

        if (!in_array($backupType, $validTypes)) {
            throw new ApiException('Invalid backup type. Supported types: ' . implode(', ', $validTypes));
        }

        $exitCode = Artisan::call('database:backup', [
            '--type' => $backupType
        ]);

        if ($exitCode !== 0) {
            $output = Artisan::output();
            throw new ApiException('Backup failed: ' . $output);
        }

        $data = $this->allDatabases();

        return ApiResponse::make('Success', $data);
    }

    public function downloadBackups($id)
    {
        if (env('APP_ENV') == 'production') {
            throw new ApiException('Not Allowed In Demo Mode');
        }

        $downloadedFileName = "";

        foreach (Storage::disk('backup')->files('/') as $filename) {
            if (strpos(basename($filename), $id) !== false) {
                $downloadedFileName = basename($filename);
            }
        }

        if (Storage::disk('backup')->exists($downloadedFileName)) {
            return Storage::disk('backup')->download($downloadedFileName);
        }
    }

    public function deleteBackup(Request $request)
    {
        $fileName = $request->file_name;

        if (Storage::disk('backup')->exists($fileName)) {
            Storage::disk('backup')->delete($fileName);
        }

        $data = $this->allDatabases();

        return ApiResponse::make('Success', $data);
    }
}
