<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiBaseController;
use App\Http\Requests\Api\Company\UpdateRequest;
use App\Http\Requests\Api\Company\UpdatePdfFontSettingRequest;
use App\Models\Company;
use App\Models\Settings;
use Examyou\RestAPI\ApiResponse;
use Examyou\RestAPI\Exceptions\ApiException;
use Illuminate\Http\Request;

class CompanyController extends ApiBaseController
{
    /**
     * The model associated with the controller.
     *
     * @var string
     */
    protected $model = Company::class;

    /**
     * The request class for update operations.
     *
     * @var string
     */
    protected $updateRequest = UpdateRequest::class;

    /**
     * The request class for PDF font setting update operations.
     *
     * @var string
     */
    protected $updatePdfFontSettingRequest = UpdatePdfFontSettingRequest::class;

    /**
     * Hook: Before updating a company
     * Prevents certain fields from being updated in production mode
     *
     * @param Company $company
     * @return Company
     * @throws ApiException
     */
    public function updating(Company $company)
    {
        if (env('APP_ENV') == 'production' && ($company->isDirty('name') ||
            $company->isDirty('short_name') || $company->isDirty('light_logo') ||
            $company->isDirty('dark_logo') || $company->isDirty('small_dark_logo') ||
            $company->isDirty('small_light_logo') || $company->isDirty('app_debug') ||
            $company->isDirty('update_app_notification') || $company->isDirty('app_debug')
        )) {
            throw new ApiException('Not Allowed In Demo Mode');
        }

        return $company;
    }

    /**
     * Update the create menu settings
     *
     * @param Request $request
     * @return \Examyou\RestAPI\ApiResponse
     */
    public function updateCreateMenu(Request $request)
    {
        $company = company();
        $company->shortcut_menus = $request->position;
        $company->save();

        // Setting for create menu
        $settingCreateMenu = Settings::where('setting_type', 'shortcut_menus')->first();

        if ($settingCreateMenu) {
            $settingCreateMenu->credentials = $request->shortcut_menus_settings;
            $settingCreateMenu->save();
        }

        return ApiResponse::make('Success', []);
    }

    /**
     * Update PDF font settings for the company
     *
     * @param UpdatePdfFontSettingRequest $updatePdfFontSettingRequest
     * @return \Examyou\RestAPI\ApiResponse
     * @throws ApiException
     */
    public function updatePdfFontSetting(UpdatePdfFontSettingRequest $updatePdfFontSettingRequest)
    {
        $company = company();

        // here we have to check permission
        $loggedUser = user();

        if (!$loggedUser->ability('admin', 'font_settings')) {
            throw new ApiException("Not have valid permission");
        }

        if ($company) {
            $company->use_custom_font = $updatePdfFontSettingRequest->use_custom_font;

            if ($updatePdfFontSettingRequest->use_custom_font == 0) {
                $company->pdf_font_id = null;
            } else {
                $company->pdf_font_id = $this->getIdFromHash($updatePdfFontSettingRequest->pdf_font_id);
            }

            $company->pdf_header_color = $updatePdfFontSettingRequest->pdf_header_color;
            $company->pdf_show_company_name = $updatePdfFontSettingRequest->pdf_show_company_name;
            $company->pdf_show_company_email = $updatePdfFontSettingRequest->pdf_show_company_email;
            $company->pdf_show_company_phone = $updatePdfFontSettingRequest->pdf_show_company_phone;
            $company->pdf_show_company_address = $updatePdfFontSettingRequest->pdf_show_company_address;
            $company->pdf_left_space = $updatePdfFontSettingRequest->pdf_left_space;
            $company->pdf_right_space = $updatePdfFontSettingRequest->pdf_right_space;
            $company->pdf_top_space = $updatePdfFontSettingRequest->pdf_top_space;
            $company->pdf_bottom_space = $updatePdfFontSettingRequest->pdf_bottom_space;
            $company->pdf_font_size = $updatePdfFontSettingRequest->pdf_font_size;
            $company->pdf_line_height = $updatePdfFontSettingRequest->pdf_line_height;
            $company->title_show_in_pdf = $updatePdfFontSettingRequest->title_show_in_pdf;
            $company->save();
        }

        return ApiResponse::make('Success', []);
    }
}
