<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Website;
use App\Models\WebsiteCategory;
use App\Models\WebsiteBuilder;
use App\Models\WebsiteLanguage;
use App\Models\WebsitePage;
use App\Models\WebsiteLegalPage;
use App\Classes\Common;
use Carbon\Carbon;
use Examyou\RestAPI\ApiResponse;
use Illuminate\Http\Request;
use Vinkla\Hashids\Facades\Hashids;

class DashboardController extends Controller
{
    /**
     * Get dashboard statistics
     *
     * @param Request $request
     * @return \Examyou\RestAPI\ApiResponse
     */
    public function index(Request $request)
    {
        $websiteId = $this->getWebsiteId($request);
        $languageId = $this->getLanguageId($request);
        $companyId = company()->id;

        // Get website statistics
        $websiteStats = $this->getWebsiteStats($companyId);

        // Get category statistics
        $categoryStats = $this->getCategoryStats($companyId);

        // Get section statistics (for selected website and language)
        $sectionStats = $this->getSectionStats($websiteId, $languageId);

        // Get language statistics (for selected website)
        $languageStats = $this->getLanguageStats($websiteId);

        // Get page statistics (for selected website)
        $pageStats = $this->getPageStats($websiteId);

        // Get legal page statistics (for selected website)
        $legalPageStats = $this->getLegalPageStats($websiteId, $languageId);

        // Get recent websites
        $recentWebsites = $this->getRecentWebsites($companyId);

        // Get website pages (for selected website and language)
        $websitePages = $this->getWebsitePages($websiteId, $languageId);

        // Get legal pages (for selected website and language)
        $legalPages = $this->getLegalPages($websiteId, $languageId);

        return ApiResponse::make('Dashboard data fetched', [
            'website_stats' => $websiteStats,
            'category_stats' => $categoryStats,
            'section_stats' => $sectionStats,
            'language_stats' => $languageStats,
            'page_stats' => $pageStats,
            'legal_page_stats' => $legalPageStats,
            'recent_websites' => $recentWebsites,
            'website_pages' => $websitePages,
            'legal_pages' => $legalPages,
        ]);
    }

    /**
     * Helper: Get decoded website ID from request
     *
     * @param Request $request
     * @return int|null
     */
    private function getWebsiteId(Request $request)
    {
        $websiteXid = $request->input('website_id');

        if ($websiteXid && $websiteXid !== 'null') {
            $decoded = Hashids::decode($websiteXid);
            if (!empty($decoded)) {
                return $decoded[0];
            }
        }

        return null;
    }

    /**
     * Helper: Get decoded language ID from request
     *
     * @param Request $request
     * @return int|null
     */
    private function getLanguageId(Request $request)
    {
        $languageXid = $request->input('language_id');

        if ($languageXid && $languageXid !== 'null') {
            $decoded = Hashids::decode($languageXid);
            if (!empty($decoded)) {
                return $decoded[0];
            }
        }

        return null;
    }

    /**
     * Get website statistics
     *
     * @param int $companyId
     * @return array
     */
    private function getWebsiteStats($companyId)
    {
        $total = Website::where('company_id', $companyId)->count();
        $primary = Website::where('company_id', $companyId)->where('is_primary', true)->count();

        // This month's websites
        $thisMonth = Website::where('company_id', $companyId)
            ->whereMonth('created_at', Carbon::now()->month)
            ->whereYear('created_at', Carbon::now()->year)
            ->count();

        return [
            'total' => $total,
            'primary' => $primary,
            'this_month' => $thisMonth,
        ];
    }

    /**
     * Get category statistics
     *
     * @param int $companyId
     * @return array
     */
    private function getCategoryStats($companyId)
    {
        $total = WebsiteCategory::where('company_id', $companyId)->count();

        return [
            'total' => $total,
        ];
    }

    /**
     * Get section statistics for website
     *
     * @param int|null $websiteId
     * @param int|null $languageId
     * @return array
     */
    private function getSectionStats($websiteId, $languageId = null)
    {
        if (!$websiteId) {
            return [
                'total' => 0,
                'unique_sections' => 0,
            ];
        }

        // Get the language ID to use (provided or default)
        $langId = $languageId ?: $this->getDefaultLanguageId($websiteId);

        $query = WebsiteBuilder::withoutGlobalScopes()
            ->where('website_id', $websiteId);

        if ($langId) {
            $query->where('website_language_id', $langId);
        }

        $total = $query->count();

        $uniqueQuery = WebsiteBuilder::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->select('section_key')
            ->distinct();

        if ($langId) {
            $uniqueQuery->where('website_language_id', $langId);
        }

        $uniqueSections = $uniqueQuery->count('section_key');

        return [
            'total' => $total,
            'unique_sections' => $uniqueSections,
        ];
    }

    /**
     * Get language statistics for website
     *
     * @param int|null $websiteId
     * @return array
     */
    private function getLanguageStats($websiteId)
    {
        if (!$websiteId) {
            return [
                'total' => 0,
                'enabled' => 0,
            ];
        }

        $total = WebsiteLanguage::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->count();

        $enabled = WebsiteLanguage::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->where('enabled', true)
            ->count();

        return [
            'total' => $total,
            'enabled' => $enabled,
        ];
    }

    /**
     * Get page statistics for website
     *
     * @param int|null $websiteId
     * @return array
     */
    private function getPageStats($websiteId)
    {
        if (!$websiteId) {
            return [
                'total' => 0,
            ];
        }

        $total = WebsitePage::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->count();

        return [
            'total' => $total,
        ];
    }

    /**
     * Get legal page statistics for website
     *
     * @param int|null $websiteId
     * @param int|null $languageId
     * @return array
     */
    private function getLegalPageStats($websiteId, $languageId = null)
    {
        if (!$websiteId) {
            return [
                'total' => 0,
                'active' => 0,
            ];
        }

        // Get the language ID to use (provided or default)
        $langId = $languageId ?: $this->getDefaultLanguageId($websiteId);

        $query = WebsiteLegalPage::withoutGlobalScopes()
            ->where('website_id', $websiteId);

        if ($langId) {
            $query->where('website_language_id', $langId);
        }

        $total = $query->count();

        $activeQuery = WebsiteLegalPage::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->where('status', 'active');

        if ($langId) {
            $activeQuery->where('website_language_id', $langId);
        }

        $active = $activeQuery->count();

        return [
            'total' => $total,
            'active' => $active,
        ];
    }

    /**
     * Get recent websites
     *
     * @param int $companyId
     * @param int $limit
     * @return \Illuminate\Support\Collection
     */
    private function getRecentWebsites($companyId, $limit = 5)
    {
        $websites = Website::where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();

        return $websites->map(function ($website) {
            // Primary websites in non-saas mode use root URL
            if ($website->is_primary && app_type() == 'non-saas') {
                $websiteUrl = route('landing.root');
            } else {
                $websiteUrl = route('landing.home', ['slug' => $website->slug]);
            }

            return [
                'xid' => $website->xid,
                'name' => $website->name,
                'slug' => $website->slug,
                'is_primary' => $website->is_primary,
                'theme_name' => $website->theme_name,
                'created_at' => $website->created_at->format('Y-m-d'),
                'time_ago' => $website->created_at->diffForHumans(),
                'website_url' => $websiteUrl,
            ];
        });
    }

    /**
     * Get website pages for selected website
     *
     * @param int|null $websiteId
     * @param int|null $languageId
     * @return \Illuminate\Support\Collection
     */
    private function getWebsitePages($websiteId, $languageId = null)
    {
        if (!$websiteId) {
            return collect([]);
        }

        // Get the website for URL generation
        $website = Website::find($websiteId);
        if (!$website) {
            return collect([]);
        }

        // Get the language ID to use (provided or default)
        $langId = $languageId ?: $this->getDefaultLanguageId($websiteId);

        $pages = WebsitePage::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->with(['translations'])
            ->withCount(['builders as sections_count' => function ($query) use ($langId) {
                $query->withoutGlobalScopes();
                if ($langId) {
                    $query->where('website_language_id', $langId);
                }
            }])
            ->orderBy('sort_order', 'asc')
            ->get();

        return $pages->map(function ($page) use ($langId, $website) {
            // Get translation for the selected language
            $translation = $page->translations->where('website_language_id', $langId)->first();
            $name = $translation?->name ?? ucwords(str_replace(['-', '_'], ' ', $page->page_key));

            // Generate URL for website page
            // Primary websites in non-saas mode use root URLs
            if ($website->is_primary && app_type() == 'non-saas') {
                if ($page->page_key === 'home') {
                    $pageUrl = route('landing.root');
                } else {
                    $pageUrl = route('landing.root.website-page', ['pageKey' => $page->page_key]);
                }
            } else {
                // Non-primary websites use /w/{slug} URLs
                if ($page->page_key === 'home') {
                    $pageUrl = route('landing.home', ['slug' => $website->slug]);
                } else {
                    $pageUrl = route('landing.website-page', ['slug' => $website->slug, 'pageKey' => $page->page_key]);
                }
            }

            return [
                'xid' => $page->xid,
                'page_key' => $page->page_key,
                'name' => $name,
                'sort_order' => $page->sort_order,
                'sections_count' => $page->sections_count ?? 0,
                'page_url' => $pageUrl,
            ];
        });
    }

    /**
     * Get legal pages for selected website
     *
     * @param int|null $websiteId
     * @param int|null $languageId
     * @param int $limit
     * @return \Illuminate\Support\Collection
     */
    private function getLegalPages($websiteId, $languageId = null, $limit = 5)
    {
        if (!$websiteId) {
            return collect([]);
        }

        // Get the language ID to use (provided or default)
        $langId = $languageId ?: $this->getDefaultLanguageId($websiteId);

        // Get legal pages for the selected language
        $query = WebsiteLegalPage::withoutGlobalScopes()
            ->where('website_id', $websiteId);

        if ($langId) {
            $query->where('website_language_id', $langId);
        }

        $legalPages = $query->orderBy('display_order', 'asc')
            ->limit($limit)
            ->get();

        return $legalPages->map(function ($page) {
            return [
                'xid' => $page->xid,
                'slug' => $page->slug,
                'title' => $page->title,
                'status' => $page->status,
                'page_url' => $page->page_url,
            ];
        });
    }

    /**
     * Get the default language ID for a website
     *
     * @param int $websiteId
     * @return int|null
     */
    private function getDefaultLanguageId($websiteId)
    {
        $defaultLang = WebsiteLanguage::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->where('is_default', true)
            ->first();

        if ($defaultLang) {
            return $defaultLang->id;
        }

        // Fallback to English if no default
        $enLang = WebsiteLanguage::withoutGlobalScopes()
            ->where('website_id', $websiteId)
            ->where('key', 'en')
            ->first();

        return $enLang?->id;
    }
}
