<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiBaseController;
use App\Http\Requests\Api\StaffMember\IndexRequest;
use App\Http\Requests\Api\StaffMember\StoreRequest;
use App\Http\Requests\Api\StaffMember\UpdateRequest;
use App\Http\Requests\Api\StaffMember\DeleteRequest;
use App\Http\Requests\Api\StaffMember\BulkStoreRequest;
use App\Http\Requests\Api\StaffMember\ImportRequest;
use App\Models\StaffMember;
use Examyou\RestAPI\ApiResponse;
use App\Traits\UserTraits;
use App\Classes\Common;
use Illuminate\Support\Facades\DB;

class StaffMembersController extends ApiBaseController
{
    use UserTraits;

    /**
     * The model associated with the controller.
     *
     * @var string
     */
    protected $model = StaffMember::class;

    /**
     * The request class for index operations.
     *
     * @var string
     */
    protected $indexRequest = IndexRequest::class;

    /**
     * The request class for store operations.
     *
     * @var string
     */
    protected $storeRequest = StoreRequest::class;

    /**
     * The request class for update operations.
     *
     * @var string
     */
    protected $updateRequest = UpdateRequest::class;

    /**
     * The request class for delete operations.
     *
     * @var string
     */
    protected $deleteRequest = DeleteRequest::class;

    /**
     * Constructor
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();

        $this->userType = "staff_members";
    }

    /**
     * Bulk store multiple staff members
     *
     * @param BulkStoreRequest $request
     * @return \Examyou\RestAPI\ApiResponse
     * @throws \Exception
     */
    public function bulkStore(BulkStoreRequest $request)
    {
        $loggedUser = user();
        $company = company();
        $staffMembers = $request->staff_members;
        $createdUsers = [];

        DB::beginTransaction();
        try {
            foreach ($staffMembers as $staffMemberData) {
                $staffMember = new StaffMember();
                $staffMember->company_id = $company->id;
                $staffMember->user_type = 'staff_members';
                $staffMember->name = $staffMemberData['name'];
                $staffMember->email = $staffMemberData['email'];
                $staffMember->status = $staffMemberData['status'] ?? 'active';
                $staffMember->allow_login = $staffMemberData['allow_login'] ?? false;

                if (!empty($staffMemberData['password'])) {
                    $staffMember->password = $staffMemberData['password'];
                }

                if ($loggedUser->ability('admin', 'assign_role') && !empty($staffMemberData['role_id'])) {
                    $staffMember->role_id = Common::getIdFromHash($staffMemberData['role_id']);
                }

                $staffMember->save();

                // Attach role if applicable
                if ($staffMember->role_id) {
                    DB::table('role_user')->where('user_id', $staffMember->id)->delete();
                    $staffMember->attachRole($staffMember->role_id);
                }

                $createdUsers[] = $staffMember;
            }

            // Update company total users count
            Common::calculateTotalUsers($company->id, true);

            DB::commit();

            return ApiResponse::make('Staff members created successfully', [
                'count' => count($createdUsers)
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Import staff members from file
     *
     * @param ImportRequest $request
     * @return \Examyou\RestAPI\ApiResponse
     * @throws \Exception
     */
    public function import(ImportRequest $request)
    {
        $company = company();
        $staffMembers = $request->staff_members;
        $imported = 0;
        $failed = 0;
        $errors = [];

        DB::beginTransaction();
        try {
            foreach ($staffMembers as $index => $staffMemberData) {
                try {
                    $staffMember = new StaffMember();
                    $staffMember->company_id = $company->id;
                    $staffMember->user_type = 'staff_members';
                    $staffMember->name = $staffMemberData['name'];
                    $staffMember->email = $staffMemberData['email'];
                    $staffMember->phone = $staffMemberData['phone'] ?? null;
                    $staffMember->address = $staffMemberData['address'] ?? null;
                    $staffMember->status = $staffMemberData['status'] ?? 'active';
                    $staffMember->allow_login = false;

                    $staffMember->save();
                    $imported++;
                } catch (\Exception $e) {
                    $failed++;
                    $errors[] = [
                        'row' => $index + 1,
                        'message' => $e->getMessage(),
                    ];
                }
            }

            // Update company total users count
            Common::calculateTotalUsers($company->id, true);

            DB::commit();

            return ApiResponse::make('Staff members imported successfully', [
                'imported' => $imported,
                'failed' => $failed,
                'errors' => $errors,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
}
