<?php

namespace App\Http\Controllers\Api;

use App\Classes\Common;
use App\Http\Controllers\ApiBaseController;
use App\Http\Requests\Api\WebsiteBuilder\IndexRequest;
use App\Http\Requests\Api\WebsiteBuilder\StoreRequest;
use App\Http\Requests\Api\WebsiteBuilder\UpdateRequest;
use App\Models\Website;
use App\Models\WebsiteBuilder;
use App\Models\WebsiteLanguage;
use App\Models\WebsitePage;
use App\Models\ThemeSection;
use Examyou\RestAPI\ApiResponse;
use Examyou\RestAPI\Exceptions\ApiException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class WebsiteBuilderController extends ApiBaseController
{
    /**
     * The model associated with the controller.
     *
     * @var string
     */
    protected $model = WebsiteBuilder::class;

    /**
     * The request class for index operations.
     *
     * @var string
     */
    protected $indexRequest = IndexRequest::class;

    /**
     * The request class for store operations.
     *
     * @var string
     */
    protected $storeRequest = StoreRequest::class;

    /**
     * The request class for update operations.
     *
     * @var string
     */
    protected $updateRequest = UpdateRequest::class;

    /**
     * Override index to get all website builders for a specific language, website and website_page.
     *
     * @return \Examyou\RestAPI\ApiResponse
     * @throws \Examyou\RestAPI\Exceptions\ApiException
     */
    public function index()
    {
        // Check permission
        $user = user();
        if (!$user->ability('admin', 'website_builder_view')) {
            throw new ApiException("Don't have valid permission");
        }

        $request = request();

        // Validate required parameters
        if (!$request->has('website_language_id') || !$request->input('website_language_id') || $request->input('website_language_id') === 'null') {
            throw new ApiException("website_language_id is required");
        }

        if (!$request->has('website_id') || !$request->input('website_id') || $request->input('website_id') === 'null') {
            throw new ApiException("website_id is required");
        }

        if (!$request->has('website_page_id') || !$request->input('website_page_id') || $request->input('website_page_id') === 'null') {
            throw new ApiException("website_page_id is required");
        }

        $langId = null;
        $websiteId = null;
        $websitePageId = null;

        // Convert website_language_id from xid to id
        $decodedLangId = \Vinkla\Hashids\Facades\Hashids::decode($request->input('website_language_id'));
        if (!empty($decodedLangId)) {
            $langId = $decodedLangId[0];
        } else {
            throw new ApiException("Invalid website_language_id");
        }

        // Convert website_id from xid to id
        $decodedWebsiteId = \Vinkla\Hashids\Facades\Hashids::decode($request->input('website_id'));
        if (!empty($decodedWebsiteId)) {
            $websiteId = $decodedWebsiteId[0];
        } else {
            throw new ApiException("Invalid website_id");
        }

        // Convert website_page_id from xid to id
        $decodedWebsitePageId = \Vinkla\Hashids\Facades\Hashids::decode($request->input('website_page_id'));
        if (!empty($decodedWebsitePageId)) {
            $websitePageId = $decodedWebsitePageId[0];
        } else {
            throw new ApiException("Invalid website_page_id");
        }

        // Build query with required filters
        $query = WebsiteBuilder::where('company_id', company()->id)
            ->where('website_language_id', $langId)
            ->where('website_id', $websiteId)
            ->where('website_page_id', $websitePageId);

        $settings = $query->orderBy('sort_order', 'asc')->get();

        return ApiResponse::make(null, $settings->toArray());
    }

    /**
     * Hook: Before storing a new section
     * Copy data from theme_sections table based on section_key
     *
     * @param WebsiteBuilder $websiteBuilder
     * @return WebsiteBuilder
     * @throws ApiException
     */
    public function storing(WebsiteBuilder $websiteBuilder)
    {
        $request = request();
        $sectionKey = $request->input('section_key');
        $companyId = company()->id;

        // Find the theme section template
        $themeSection = ThemeSection::where('section_key', $sectionKey)
            ->where('is_active', true)
            ->first();

        if (!$themeSection) {
            throw new ApiException("Section template '{$sectionKey}' not found");
        }

        // Handle website_language_id conversion
        if ($request->has('website_language_id') && $request->input('website_language_id')) {
            $websiteBuilder->website_language_id = Common::getIdFromHash($request->input('website_language_id'));
        } else {
            // Default to English language
            $enLang = WebsiteLanguage::where('key', 'en')->first();
            $websiteBuilder->website_language_id = $enLang ? $enLang->id : null;
        }

        // Handle website_id conversion (only if website exists)
        $websiteId = null;
        if ($request->has('website_id') && $request->input('website_id')) {
            $websiteId = Common::getIdFromHash($request->input('website_id'));
            if ($websiteId && Website::where('id', $websiteId)->exists()) {
                $websiteBuilder->website_id = $websiteId;
            }
        }

        // Handle website_page_id conversion
        $websitePageId = null;
        if ($request->has('website_page_id') && $request->input('website_page_id')) {
            $websitePageId = Common::getIdFromHash($request->input('website_page_id'));
            if ($websitePageId) {
                $websiteBuilder->website_page_id = $websitePageId;
            }
        }

        // Check if section already exists for this combination
        $existsQuery = WebsiteBuilder::where('company_id', $companyId)
            ->where('website_language_id', $websiteBuilder->website_language_id)
            ->where('section_key', $sectionKey);

        if ($websiteBuilder->website_id) {
            $existsQuery->where('website_id', $websiteBuilder->website_id);
        } else {
            $existsQuery->whereNull('website_id');
        }

        if ($websiteBuilder->website_page_id) {
            $existsQuery->where('website_page_id', $websiteBuilder->website_page_id);
        } else {
            $existsQuery->whereNull('website_page_id');
        }

        if ($existsQuery->exists()) {
            throw new ApiException("This section has already been added for this page and language.");
        }

        // Set company_id
        $websiteBuilder->company_id = $companyId;

        // Copy data from theme section
        $websiteBuilder->section_key = $themeSection->section_key;
        $websiteBuilder->title = $themeSection->default_title ?? $themeSection->name;
        $websiteBuilder->subtitle = $themeSection->default_subtitle ?? '';
        $websiteBuilder->content = $themeSection->default_content ?? [];
        $websiteBuilder->content_schema = $themeSection->content_schema ?? [];

        // Set section_layout from request if provided, otherwise use default from theme_section
        if ($request->has('section_layout') && $request->input('section_layout')) {
            $websiteBuilder->section_layout = $request->input('section_layout');
        } elseif ($themeSection->default_layout) {
            $websiteBuilder->section_layout = $themeSection->default_layout;
        }

        // If adding to a non-Home page, copy data from Home page's same section
        if ($websitePageId) {
            // Find the Home page for this website
            $homePage = \App\Models\WebsitePage::where('website_id', $websiteBuilder->website_id)
                ->where('page_key', 'home')
                ->first();

            if ($homePage) {
                // Look for the same section on the Home page
                $homePageSection = WebsiteBuilder::where('company_id', $companyId)
                    ->where('website_language_id', $websiteBuilder->website_language_id)
                    ->where('section_key', $sectionKey)
                    ->where('website_id', $websiteBuilder->website_id)
                    ->where('website_page_id', $homePage->id)
                    ->first();

                if ($homePageSection) {
                    // Copy section_layout
                    if ($homePageSection->section_layout) {
                        $websiteBuilder->section_layout = $homePageSection->section_layout;
                    }
                    // Copy content data
                    if ($homePageSection->content) {
                        $websiteBuilder->content = $homePageSection->content;
                    }
                    // Copy title
                    if ($homePageSection->title) {
                        $websiteBuilder->title = $homePageSection->title;
                    }
                    // Copy subtitle
                    if ($homePageSection->subtitle) {
                        $websiteBuilder->subtitle = $homePageSection->subtitle;
                    }
                    // Copy is_white_background
                    $websiteBuilder->is_white_background = $homePageSection->is_white_background ?? false;
                }
            }
        }

        // Calculate next sort_order based on existing sections for this lang/website/website_page combination
        // If adding a non-footer section and footer exists, insert before footer
        // Otherwise, add at the end

        $baseQuery = WebsiteBuilder::where('company_id', company()->id)
            ->where('website_language_id', $websiteBuilder->website_language_id)
            ->where(function ($query) use ($websiteBuilder) {
                if ($websiteBuilder->website_id) {
                    $query->where('website_id', $websiteBuilder->website_id);
                } else {
                    $query->whereNull('website_id');
                }
            })
            ->where(function ($query) use ($websiteBuilder) {
                if ($websiteBuilder->website_page_id) {
                    $query->where('website_page_id', $websiteBuilder->website_page_id);
                } else {
                    $query->whereNull('website_page_id');
                }
            });

        // Check if we're adding a footer or if footer exists
        $isAddingFooter = ($sectionKey === 'footer');

        if (!$isAddingFooter) {
            // Check if footer section exists for this combination
            $footerSection = (clone $baseQuery)
                ->where('section_key', 'footer')
                ->first();

            if ($footerSection) {
                // Footer exists - insert new section just before it
                // Get footer's current sort_order
                $footerSortOrder = $footerSection->sort_order;

                // Increment sort_order of footer and all sections after it
                WebsiteBuilder::where('company_id', company()->id)
                    ->where('website_language_id', $websiteBuilder->website_language_id)
                    ->where(function ($query) use ($websiteBuilder) {
                        if ($websiteBuilder->website_id) {
                            $query->where('website_id', $websiteBuilder->website_id);
                        } else {
                            $query->whereNull('website_id');
                        }
                    })
                    ->where(function ($query) use ($websiteBuilder) {
                        if ($websiteBuilder->website_page_id) {
                            $query->where('website_page_id', $websiteBuilder->website_page_id);
                        } else {
                            $query->whereNull('website_page_id');
                        }
                    })
                    ->where('sort_order', '>=', $footerSortOrder)
                    ->increment('sort_order');

                // Set new section's sort_order to where footer was
                $websiteBuilder->sort_order = $footerSortOrder;
            } else {
                // No footer exists - add at the end
                $maxSortOrder = $baseQuery->max('sort_order');
                $websiteBuilder->sort_order = ($maxSortOrder ?? -1) + 1;
            }
        } else {
            // Adding footer - always add at the end
            $maxSortOrder = $baseQuery->max('sort_order');
            $websiteBuilder->sort_order = ($maxSortOrder ?? -1) + 1;
        }

        return $websiteBuilder;
    }

    /**
     * Hook: After storing a new section
     * Create the same section for all other languages
     * Also sync header/footer to all pages if website setting is enabled
     *
     * @param WebsiteBuilder $websiteBuilder
     * @return WebsiteBuilder
     */
    public function stored(WebsiteBuilder $websiteBuilder)
    {
        $companyId = company()->id;

        // Get all front languages for this specific website except the one already created
        // Languages must belong to the same website (website_id filter)
        $otherLanguagesQuery = WebsiteLanguage::where('company_id', $companyId)
            ->where('id', '!=', $websiteBuilder->website_language_id);

        // Only get languages that belong to the same website
        if ($websiteBuilder->website_id) {
            $otherLanguagesQuery->where('website_id', $websiteBuilder->website_id);
        }

        $otherLanguages = $otherLanguagesQuery->get();

        // Create the same section for all other languages
        foreach ($otherLanguages as $lang) {
            // Check if section already exists for this language/website/website_page combination
            $exists = WebsiteBuilder::where('company_id', $companyId)
                ->where('website_language_id', $lang->id)
                ->where('section_key', $websiteBuilder->section_key)
                ->where(function ($query) use ($websiteBuilder) {
                    if ($websiteBuilder->website_id) {
                        $query->where('website_id', $websiteBuilder->website_id);
                    } else {
                        $query->whereNull('website_id');
                    }
                })
                ->where(function ($query) use ($websiteBuilder) {
                    if ($websiteBuilder->website_page_id) {
                        $query->where('website_page_id', $websiteBuilder->website_page_id);
                    } else {
                        $query->whereNull('website_page_id');
                    }
                })
                ->exists();

            if (!$exists) {
                // Start with data from the primary section
                $sectionData = [
                    'company_id' => $companyId,
                    'website_id' => $websiteBuilder->website_id,
                    'website_page_id' => $websiteBuilder->website_page_id,
                    'website_language_id' => $lang->id,
                    'section_key' => $websiteBuilder->section_key,
                    'section_layout' => $websiteBuilder->section_layout,
                    'title' => $websiteBuilder->title,
                    'subtitle' => $websiteBuilder->subtitle ?? '',
                    'content' => $websiteBuilder->content,
                    'content_schema' => $websiteBuilder->content_schema,
                    'sort_order' => $websiteBuilder->sort_order ?? 0,
                    'is_white_background' => $websiteBuilder->is_white_background ?? false,
                ];

                // If adding to a non-Home page, try to copy from Home page section for this language
                if ($websiteBuilder->website_page_id) {
                    $homePage = \App\Models\WebsitePage::where('website_id', $websiteBuilder->website_id)
                        ->where('page_key', 'home')
                        ->first();

                    if ($homePage) {
                        $homePageSection = WebsiteBuilder::where('company_id', $companyId)
                            ->where('website_language_id', $lang->id)
                            ->where('section_key', $websiteBuilder->section_key)
                            ->where('website_id', $websiteBuilder->website_id)
                            ->where('website_page_id', $homePage->id)
                            ->first();

                        if ($homePageSection) {
                            // Override with Home page data for this language
                            if ($homePageSection->section_layout) {
                                $sectionData['section_layout'] = $homePageSection->section_layout;
                            }
                            if ($homePageSection->content) {
                                $sectionData['content'] = $homePageSection->content;
                            }
                            if ($homePageSection->title) {
                                $sectionData['title'] = $homePageSection->title;
                            }
                            if ($homePageSection->subtitle) {
                                $sectionData['subtitle'] = $homePageSection->subtitle;
                            }
                            $sectionData['is_white_background'] = $homePageSection->is_white_background ?? false;
                        }
                    }
                }

                WebsiteBuilder::create($sectionData);
            }
        }

        // Sync header/footer to all other pages if website setting is enabled
        $this->syncHeaderFooterToAllPages($websiteBuilder, 'create');

        return $websiteBuilder;
    }

    /**
     * Sync header/footer section to all other pages based on website settings
     *
     * @param WebsiteBuilder $websiteBuilder
     * @param string $action 'create', 'update', or 'delete'
     * @return void
     */
    private function syncHeaderFooterToAllPages(WebsiteBuilder $websiteBuilder, string $action): void
    {
        $sectionKey = $websiteBuilder->section_key;

        // Only process header and footer sections
        if (!in_array($sectionKey, ['header', 'footer'])) {
            return;
        }

        // Get the website to check settings
        $website = Website::find($websiteBuilder->website_id);
        if (!$website) {
            return;
        }

        // Check if the relevant setting is enabled
        $settingField = $sectionKey === 'header' ? 'use_same_header_across_all_pages' : 'use_same_footer_across_all_pages';
        if (!$website->$settingField) {
            return;
        }

        $companyId = company()->id;

        // Get all pages for this website except the current one
        $otherPages = WebsitePage::where('website_id', $websiteBuilder->website_id)
            ->where('id', '!=', $websiteBuilder->website_page_id)
            ->get();

        // Get all languages for this website
        $allLanguages = WebsiteLanguage::where('company_id', $companyId)
            ->where('website_id', $websiteBuilder->website_id)
            ->get();

        foreach ($otherPages as $page) {
            foreach ($allLanguages as $lang) {
                if ($action === 'create') {
                    // Check if section already exists for this page/language combination
                    $exists = WebsiteBuilder::where('company_id', $companyId)
                        ->where('website_language_id', $lang->id)
                        ->where('section_key', $sectionKey)
                        ->where('website_id', $websiteBuilder->website_id)
                        ->where('website_page_id', $page->id)
                        ->exists();

                    if (!$exists) {
                        // Get the source section for this language (from the page where it was created)
                        $sourceSection = WebsiteBuilder::where('company_id', $companyId)
                            ->where('website_language_id', $lang->id)
                            ->where('section_key', $sectionKey)
                            ->where('website_id', $websiteBuilder->website_id)
                            ->where('website_page_id', $websiteBuilder->website_page_id)
                            ->first();

                        if ($sourceSection) {
                            WebsiteBuilder::create([
                                'company_id' => $companyId,
                                'website_id' => $websiteBuilder->website_id,
                                'website_page_id' => $page->id,
                                'website_language_id' => $lang->id,
                                'section_key' => $sectionKey,
                                'section_layout' => $sourceSection->section_layout,
                                'title' => $sourceSection->title,
                                'subtitle' => $sourceSection->subtitle ?? '',
                                'content' => $sourceSection->content,
                                'content_schema' => $sourceSection->content_schema,
                                'sort_order' => $sourceSection->sort_order ?? 0,
                                'is_white_background' => $sourceSection->is_white_background ?? false,
                                'padding_top' => $sourceSection->padding_top,
                                'padding_bottom' => $sourceSection->padding_bottom,
                            ]);
                        }
                    }
                } elseif ($action === 'update') {
                    // Update the section on other pages for this language
                    $targetSection = WebsiteBuilder::where('company_id', $companyId)
                        ->where('website_language_id', $websiteBuilder->website_language_id)
                        ->where('section_key', $sectionKey)
                        ->where('website_id', $websiteBuilder->website_id)
                        ->where('website_page_id', $page->id)
                        ->first();

                    if ($targetSection) {
                        $targetSection->update([
                            'section_layout' => $websiteBuilder->section_layout,
                            'title' => $websiteBuilder->title,
                            'subtitle' => $websiteBuilder->subtitle,
                            'content' => $websiteBuilder->content,
                            'is_white_background' => $websiteBuilder->is_white_background,
                            'padding_top' => $websiteBuilder->padding_top,
                            'padding_bottom' => $websiteBuilder->padding_bottom,
                        ]);
                    }
                } elseif ($action === 'delete') {
                    // Delete the section from other pages for all languages
                    WebsiteBuilder::where('company_id', $companyId)
                        ->where('section_key', $sectionKey)
                        ->where('website_id', $websiteBuilder->website_id)
                        ->where('website_page_id', $page->id)
                        ->delete();
                }
            }
        }
    }

    /**
     * Hook: After destroying a section
     * Sync header/footer deletion to all other pages if website setting is enabled
     *
     * @param WebsiteBuilder $websiteBuilder
     * @return WebsiteBuilder
     */
    public function destroyed(WebsiteBuilder $websiteBuilder)
    {
        // Sync header/footer deletion to all other pages if website setting is enabled
        $this->syncHeaderFooterToAllPages($websiteBuilder, 'delete');

        return $websiteBuilder;
    }

    /**
     * Override update to find and update the correct record for the given website_language_id, website_id, website_page_id and section_key
     * This ensures that updates only affect the record for the selected language and page
     *
     * @param mixed ...$args
     * @return \Examyou\RestAPI\ApiResponse
     * @throws ApiException
     */
    public function update(...$args)
    {
        $id = $args[0] ?? null;
        $request = request();
        $langId = null;
        $websiteId = null;
        $websitePageId = null;
        $sectionKey = $request->input('section_key');

        // Handle website_language_id conversion
        if ($request->has('website_language_id') && $request->input('website_language_id') && $request->input('website_language_id') !== 'null') {
            $decodedLangId = \Vinkla\Hashids\Facades\Hashids::decode($request->input('website_language_id'));
            if (!empty($decodedLangId)) {
                $langId = $decodedLangId[0];
            }
        }

        // Handle website_id conversion
        if ($request->has('website_id') && $request->input('website_id') && $request->input('website_id') !== 'null') {
            $decodedWebsiteId = \Vinkla\Hashids\Facades\Hashids::decode($request->input('website_id'));
            if (!empty($decodedWebsiteId)) {
                $websiteId = $decodedWebsiteId[0];
            }
        }

        // Handle website_page_id conversion
        if ($request->has('website_page_id') && $request->input('website_page_id') && $request->input('website_page_id') !== 'null') {
            $decodedWebsitePageId = \Vinkla\Hashids\Facades\Hashids::decode($request->input('website_page_id'));
            if (!empty($decodedWebsitePageId)) {
                $websitePageId = $decodedWebsitePageId[0];
            }
        }

        // Find the correct record for this website_language_id, website_id, website_page_id and section_key
        // This is the record we should actually update
        $websiteBuilder = null;

        if ($langId && $sectionKey) {
            $query = WebsiteBuilder::where('section_key', $sectionKey)
                ->where('website_language_id', $langId)
                ->where('company_id', company()->id);

            if ($websiteId) {
                $query->where('website_id', $websiteId);
            } else {
                $query->whereNull('website_id');
            }

            if ($websitePageId) {
                $query->where('website_page_id', $websitePageId);
            } else {
                $query->whereNull('website_page_id');
            }

            $websiteBuilder = $query->first();

            // If no record exists for this language/website/page combination, create one by copying from original
            if (!$websiteBuilder) {
                $websiteBuilder = WebsiteBuilder::getOrCreateForLang($sectionKey, $langId, $websiteId, $websitePageId);
            }
        }

        // Fallback to original record if no website_language_id provided
        if (!$websiteBuilder) {
            $websiteBuilder = WebsiteBuilder::find(Common::getIdFromHash($id));
        }

        if (!$websiteBuilder) {
            throw new ApiException("Website builder not found");
        }

        // Update basic fields
        if ($request->has('section_layout')) {
            $websiteBuilder->section_layout = $request->input('section_layout');
        }
        if ($request->has('title')) {
            $websiteBuilder->title = $request->input('title');
        }
        if ($request->has('subtitle')) {
            $websiteBuilder->subtitle = $request->input('subtitle');
        }
        if ($request->has('is_white_background')) {
            $websiteBuilder->is_white_background = $request->input('is_white_background') ? true : false;
        }

        // Handle padding_top (nullable integer)
        if ($request->has('padding_top')) {
            $paddingTop = $request->input('padding_top');
            $websiteBuilder->padding_top = ($paddingTop !== null && $paddingTop !== '' && $paddingTop !== 'null')
                ? (int) $paddingTop
                : null;
        }

        // Handle padding_bottom (nullable integer)
        if ($request->has('padding_bottom')) {
            $paddingBottom = $request->input('padding_bottom');
            $websiteBuilder->padding_bottom = ($paddingBottom !== null && $paddingBottom !== '' && $paddingBottom !== 'null')
                ? (int) $paddingBottom
                : null;
        }

        // Get ORIGINAL content from database BEFORE any modifications
        // This is needed to get old image filenames for deletion
        $originalContent = $websiteBuilder->content ?? [];
        if (is_string($originalContent)) {
            $originalContent = json_decode($originalContent, true) ?? [];
        }

        // Handle content - start with current content for image merging
        $content = $originalContent;

        // Parse new content from request
        if ($request->has('content')) {
            $contentInput = $request->input('content');
            if (is_string($contentInput)) {
                $decoded = json_decode($contentInput, true);
                if (is_array($decoded)) {
                    $content = $decoded;
                }
            } elseif (is_array($contentInput)) {
                $content = $contentInput;
            }
        }

        // Remove all _url fields from content - they are dynamically generated by the model accessor
        // This prevents stale URLs from being stored in the database
        $content = $this->removeUrlFieldsFromContent($content);

        // Handle content images
        $folderPath = $websiteBuilder->imageFileFolder;
        $allFiles = $request->allFiles();
        \Illuminate\Support\Facades\Log::info('[WebsiteBuilder] All files received:', array_keys($allFiles));
        \Illuminate\Support\Facades\Log::info('[WebsiteBuilder] Content image files:', array_filter(array_keys($allFiles), fn($k) => str_starts_with($k, 'content_image_')));

        foreach ($allFiles as $key => $file) {
            if (str_starts_with($key, 'content_image_')) {
                $fieldName = substr($key, strlen('content_image_'));
                \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Processing image: {$key} -> fieldName: {$fieldName}");

                // Try to detect group_with_repeater by checking content structure
                // Format: groupField_repeaterField_index_subField (e.g., trusted_users_avatars_0_image)
                $groupWithRepeaterHandled = false;

                // Find pattern: something_NUMBER_something at the end
                // Extract index and subField from the end of fieldName
                if (preg_match('/_(\d+)_([^_]+)$/', $fieldName, $indexMatch)) {
                    $index = (int) $indexMatch[1];
                    $subField = $indexMatch[2];
                    // Get the prefix by removing the matched part from the end
                    $prefix = preg_replace('/_\d+_[^_]+$/', '', $fieldName);

                    // Now try to find the split between groupField and repeaterField in the prefix
                    // We need to check against actual content structure
                    $parts = explode('_', $prefix);

                    // Try different combinations from longest group name to shortest
                    for ($i = count($parts) - 1; $i >= 1; $i--) {
                        $potentialGroup = implode('_', array_slice($parts, 0, $i));
                        $potentialRepeater = implode('_', array_slice($parts, $i));

                        // Check if this structure exists in content (group with nested repeater array)
                        if (isset($content[$potentialGroup]) &&
                            is_array($content[$potentialGroup]) &&
                            !array_is_list($content[$potentialGroup]) &&
                            isset($content[$potentialGroup][$potentialRepeater]) &&
                            is_array($content[$potentialGroup][$potentialRepeater]) &&
                            array_is_list($content[$potentialGroup][$potentialRepeater])) {

                            $groupField = $potentialGroup;
                            $repeaterField = $potentialRepeater;

                            \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Group with repeater image: group={$groupField}, repeater={$repeaterField}, index={$index}, subField={$subField}");

                            // Get old image filename from ORIGINAL database content
                            $oldFileName = $originalContent[$groupField][$repeaterField][$index][$subField] ?? null;
                            \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Old group repeater image from DB: " . ($oldFileName ?? 'null'));

                            // Upload the new file first
                            $fileName = Common::uploadFileUsingDriver($file, $folderPath);
                            \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Uploaded file: {$fileName}");

                            // Delete old image if exists (after successful upload)
                            if ($oldFileName && is_string($oldFileName) && !empty($oldFileName)) {
                                $deleteResult = Common::deleteFileUsingDriver($oldFileName, $folderPath);
                                \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Delete result for group repeater image: " . ($deleteResult ? 'success' : 'failed'));
                            }

                            // Ensure the item at index exists
                            if (!isset($content[$groupField][$repeaterField][$index]) || !is_array($content[$groupField][$repeaterField][$index])) {
                                $content[$groupField][$repeaterField][$index] = [];
                            }

                            // Set the image filename
                            $content[$groupField][$repeaterField][$index][$subField] = $fileName;

                            // Remove any stale _url field
                            unset($content[$groupField][$repeaterField][$index][$subField . '_url']);

                            $groupWithRepeaterHandled = true;
                            break;
                        }
                    }
                }

                // Check if this is a regular repeater image (format: fieldName_index_subFieldName)
                // Use non-greedy matching to properly parse field names with underscores
                if (!$groupWithRepeaterHandled && preg_match('/^(.+?)_(\d+)_(.+)$/', $fieldName, $matches)) {
                    $repeaterField = $matches[1];
                    $index = (int) $matches[2];
                    $subField = $matches[3];

                    // Get old image filename from ORIGINAL database content (not request content)
                    $oldFileName = $originalContent[$repeaterField][$index][$subField] ?? null;
                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Old repeater image from DB: " . ($oldFileName ?? 'null'));
                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Original content for repeater: ", $originalContent[$repeaterField][$index] ?? []);

                    // Upload the new file first
                    $fileName = Common::uploadFileUsingDriver($file, $folderPath);
                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Uploaded file: {$fileName}");

                    // Delete old image if exists (after successful upload)
                    if ($oldFileName && is_string($oldFileName) && !empty($oldFileName)) {
                        $fullPath = $folderPath . '/' . $oldFileName;
                        \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Attempting to delete: {$fullPath}");
                        $deleteResult = Common::deleteFileUsingDriver($oldFileName, $folderPath);
                        \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Delete result for repeater image: " . ($deleteResult ? 'success' : 'failed'));
                    } else {
                        \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] No old repeater image to delete (oldFileName is empty/null)");
                    }

                    // Ensure the repeater array exists and is a proper array
                    if (!isset($content[$repeaterField]) || !is_array($content[$repeaterField])) {
                        $content[$repeaterField] = [];
                    }

                    // Ensure the item at index exists
                    if (!isset($content[$repeaterField][$index]) || !is_array($content[$repeaterField][$index])) {
                        $content[$repeaterField][$index] = [];
                    }

                    // Set the image filename
                    $content[$repeaterField][$index][$subField] = $fileName;

                    // Remove any stale _url field that might have been sent from frontend
                    // The model's accessor will regenerate it with the correct URL
                    unset($content[$repeaterField][$index][$subField . '_url']);
                } elseif (!$groupWithRepeaterHandled) {
                    // Regular content image - get old image filename from ORIGINAL database content
                    $oldFileName = $originalContent[$fieldName] ?? null;
                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Old image from DB: " . ($oldFileName ?? 'null'));
                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Original content keys: ", array_keys($originalContent));

                    // Upload the new file first
                    $fileName = Common::uploadFileUsingDriver($file, $folderPath);
                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Uploaded file: {$fileName}");

                    // Delete old image if exists (after successful upload)
                    if ($oldFileName && is_string($oldFileName) && !empty($oldFileName)) {
                        $fullPath = $folderPath . '/' . $oldFileName;
                        \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Attempting to delete: {$fullPath}");
                        $deleteResult = Common::deleteFileUsingDriver($oldFileName, $folderPath);
                        \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Delete result: " . ($deleteResult ? 'success' : 'failed'));
                    } else {
                        \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] No old image to delete (oldFileName is empty/null)");
                    }

                    // Set the new image filename
                    $content[$fieldName] = $fileName;
                    // Remove any stale _url field - the model's accessor will regenerate it
                    unset($content[$fieldName . '_url']);
                }
            }
        }

        // Handle image deletions - when a field is set to null but had an image before
        // This handles the case when user clicks the remove button on an uploaded image
        $this->handleImageDeletions($originalContent, $content, $folderPath);

        // Re-index any repeater arrays to ensure proper JSON encoding
        // (prevents objects instead of arrays when indices are non-sequential)
        foreach ($content as $key => $value) {
            if (is_array($value) && array_is_list($value)) {
                $content[$key] = array_values($value);
            }
        }

        $websiteBuilder->content = $content;
        $websiteBuilder->save();

        // Sync header/footer to all other pages if website setting is enabled
        $this->syncHeaderFooterToAllPages($websiteBuilder, 'update');

        return ApiResponse::make('Website builder updated successfully', $websiteBuilder->fresh()->toArray());
    }

    /**
     * Update sort order for multiple sections
     *
     * @param Request $request
     * @return \Examyou\RestAPI\ApiResponse
     */
    public function updateSortOrder(Request $request)
    {
        $sections = $request->input('sections', []);

        if (empty($sections)) {
            return ApiResponse::make('No sections to update', []);
        }

        foreach ($sections as $sectionData) {
            $xid = $sectionData['xid'] ?? null;
            $sortOrder = $sectionData['sort_order'] ?? 0;

            if (!$xid) {
                continue;
            }

            $id = Common::getIdFromHash($xid);
            if ($id) {
                WebsiteBuilder::where('id', $id)
                    ->where('company_id', company()->id)
                    ->update(['sort_order' => $sortOrder]);
            }
        }

        return ApiResponse::make('Sort order updated successfully', []);
    }

    /**
     * Get available section layouts for a given section_key
     * Scans the theme's components folder for available layout files
     *
     * @param Request $request
     * @return \Examyou\RestAPI\ApiResponse
     */
    public function getSectionLayouts(Request $request)
    {
        $sectionKey = $request->input('section_key');
        $websiteXid = $request->input('website_id');

        if (!$sectionKey) {
            return ApiResponse::make(null, []);
        }

        // Get theme name from website or use default
        $themeName = 'tailwind_theme'; // Default theme

        if ($websiteXid && $websiteXid !== 'null') {
            $websiteId = Common::getIdFromHash($websiteXid);
            $website = Website::find($websiteId);
            if ($website && $website->theme_name) {
                $themeName = $website->theme_name;
            }
        }

        // Build path to the section's component folder
        $componentPath = resource_path("views/themes/{$themeName}/components/{$sectionKey}");

        $layouts = [];

        if (File::isDirectory($componentPath)) {
            // Get all blade files in the folder
            $files = File::files($componentPath);

            foreach ($files as $file) {
                $filename = $file->getFilename();

                // Match files like ⚡hero-section.blade.php or ⚡pricing.blade.php
                if (preg_match('/^⚡(.+)\.blade\.php$/', $filename, $matches)) {
                    $layoutName = $matches[1];

                    // Generate a human-readable label
                    $label = $this->generateLayoutLabel($layoutName, $sectionKey);

                    $layouts[] = [
                        'value' => $layoutName,
                        'label' => $label,
                    ];
                }
            }

            // Sort layouts by value for consistent ordering
            usort($layouts, function ($a, $b) {
                return strnatcmp($a['value'], $b['value']);
            });
        }

        // If no layouts found, return the section_key as default
        if (empty($layouts)) {
            $layouts[] = [
                'value' => $sectionKey,
                'label' => ucwords(str_replace('_', ' ', $sectionKey)),
            ];
        }

        return ApiResponse::make(null, $layouts);
    }

    /**
     * Generate a human-readable label for a layout name
     *
     * @param string $layoutName
     * @param string $sectionKey
     * @return string
     */
    private function generateLayoutLabel(string $layoutName, string $sectionKey): string
    {
        // Convert layout name to title case
        $label = str_replace(['-', '_'], ' ', $layoutName);
        $label = ucwords($label);

        // If the layout name is the same as section_key (base layout), mark it as default
        if ($layoutName === $sectionKey || $layoutName === str_replace('_', '-', $sectionKey)) {
            $label .= ' (Default)';
        }

        return $label;
    }

    /**
     * Recursively remove all _url fields from content array
     * These fields are dynamically generated by the model accessor and should not be stored
     * Exception: Certain fields like cta_url, google_map_url are legitimate user input fields and should be preserved
     *
     * @param array $content
     * @return array
     */
    private function removeUrlFieldsFromContent(array $content): array
    {
        // List of URL fields that are user inputs and should be preserved
        $preserveUrlFields = ['cta_url', 'google_map_url', 'google_map_link_url', 'button_url', 'video_url', 'view_all_url', 'qr_url'];

        $result = [];

        foreach ($content as $key => $value) {
            // Skip any key that ends with _url, EXCEPT those in the preserve list
            if (str_ends_with($key, '_url') && !in_array($key, $preserveUrlFields)) {
                continue;
            }

            // If value is an array, process it recursively
            if (is_array($value)) {
                // Check if it's a sequential array (like repeater items)
                if (array_is_list($value)) {
                    $result[$key] = array_map(function ($item) {
                        if (is_array($item)) {
                            return $this->removeUrlFieldsFromContent($item);
                        }
                        return $item;
                    }, $value);
                } else {
                    // Associative array - process recursively
                    $result[$key] = $this->removeUrlFieldsFromContent($value);
                }
            } else {
                $result[$key] = $value;
            }
        }

        return $result;
    }

    /**
     * Handle image deletions when fields are set to null
     * Compares original content with new content and deletes images that were removed
     *
     * @param array $originalContent Original content from database
     * @param array $newContent New content from request
     * @param string $folderPath Folder path for images
     * @return void
     */
    private function handleImageDeletions(array $originalContent, array $newContent, string $folderPath): void
    {
        foreach ($originalContent as $key => $value) {
            // Skip _url fields
            if (str_ends_with($key, '_url')) {
                continue;
            }

            // Check if this is a repeater array (sequential array of items)
            if (is_array($value) && array_is_list($value)) {
                foreach ($value as $index => $item) {
                    if (is_array($item)) {
                        foreach ($item as $subKey => $subValue) {
                            // Skip _url fields
                            if (str_ends_with($subKey, '_url')) {
                                continue;
                            }

                            // Check if this was an image (string value that's not a URL)
                            if (is_string($subValue) && !empty($subValue) && !str_starts_with($subValue, 'http')) {
                                // Check if the new value is null or empty (image was removed)
                                $newValue = $newContent[$key][$index][$subKey] ?? null;
                                if ($newValue === null || $newValue === '') {
                                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Deleting removed repeater image: {$subValue}");
                                    Common::deleteFileUsingDriver($subValue, $folderPath);
                                }
                            }

                            // Handle nested repeaters (e.g., features inside items)
                            if (is_array($subValue) && array_is_list($subValue)) {
                                foreach ($subValue as $nestedIndex => $nestedItem) {
                                    if (is_array($nestedItem)) {
                                        foreach ($nestedItem as $nestedKey => $nestedValue) {
                                            if (str_ends_with($nestedKey, '_url')) {
                                                continue;
                                            }
                                            if (is_string($nestedValue) && !empty($nestedValue) && !str_starts_with($nestedValue, 'http')) {
                                                $newNestedValue = $newContent[$key][$index][$subKey][$nestedIndex][$nestedKey] ?? null;
                                                if ($newNestedValue === null || $newNestedValue === '') {
                                                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Deleting removed nested repeater image: {$nestedValue}");
                                                    Common::deleteFileUsingDriver($nestedValue, $folderPath);
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            // Check if this is a group_with_repeater (object/associative array containing repeaters)
            elseif (is_array($value) && !array_is_list($value)) {
                // This could be a group_with_repeater structure like trusted_users: { enabled, title, avatars: [...] }
                foreach ($value as $subKey => $subValue) {
                    if (str_ends_with($subKey, '_url')) {
                        continue;
                    }

                    // Check if subValue is a repeater array within the group
                    if (is_array($subValue) && array_is_list($subValue)) {
                        foreach ($subValue as $index => $item) {
                            if (is_array($item)) {
                                foreach ($item as $itemKey => $itemValue) {
                                    if (str_ends_with($itemKey, '_url')) {
                                        continue;
                                    }
                                    // Check if this was an image
                                    if (is_string($itemValue) && !empty($itemValue) && !str_starts_with($itemValue, 'http')) {
                                        $newValue = $newContent[$key][$subKey][$index][$itemKey] ?? null;
                                        if ($newValue === null || $newValue === '') {
                                            \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Deleting removed group repeater image: {$itemValue}");
                                            Common::deleteFileUsingDriver($itemValue, $folderPath);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            // Check if this is a regular image field (string value that's not a URL)
            elseif (is_string($value) && !empty($value) && !str_starts_with($value, 'http')) {
                // Check if the new value is null or empty (image was removed)
                $newValue = $newContent[$key] ?? null;
                if ($newValue === null || $newValue === '') {
                    \Illuminate\Support\Facades\Log::info("[WebsiteBuilder] Deleting removed image: {$value}");
                    Common::deleteFileUsingDriver($value, $folderPath);
                }
            }
        }
    }
}
