<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiBaseController;
use App\Http\Requests\Api\Website\IndexRequest;
use App\Http\Requests\Api\Website\StoreRequest;
use App\Http\Requests\Api\Website\UpdateRequest;
use App\Http\Requests\Api\Website\DeleteRequest;
use App\Models\Website;
use App\Models\WebsitePage;
use App\Models\WebsitePageTranslation;
use App\Models\WebsiteSetting;
use App\Models\WebsiteLanguage;

class WebsiteController extends ApiBaseController
{
    /**
     * The model associated with the controller.
     *
     * @var string
     */
    protected $model = Website::class;

    /**
     * The request class for index operations.
     *
     * @var string
     */
    protected $indexRequest = IndexRequest::class;

    /**
     * The request class for store operations.
     *
     * @var string
     */
    protected $storeRequest = StoreRequest::class;

    /**
     * The request class for update operations.
     *
     * @var string
     */
    protected $updateRequest = UpdateRequest::class;

    /**
     * The request class for delete operations.
     *
     * @var string
     */
    protected $deleteRequest = DeleteRequest::class;

    /**
     * Hook: After storing a new business
     * Create front_langs from request, business settings, default Home page
     *
     * @param Website $website
     * @return Business
     */
    public function stored(Website $website)
    {
        $companyId = company()->id;
        $websiteName = $website->name;
        $request = request();
        $languages = $request->languages ?? [];

        // If no languages provided, create default English
        if (empty($languages)) {
            $languages = [[
                'name' => 'English',
                'key' => 'en',
                'flag_code' => 'US',
                'is_rtl' => false,
                'enabled' => true,
                'is_default' => true,
            ]];
        }

        $defaultLang = null;
        $createdLangs = [];

        // Create all languages for this website
        foreach ($languages as $langData) {
            $frontLang = WebsiteLanguage::create([
                'company_id' => $companyId,
                'website_id' => $website->id,
                'name' => $langData['name'],
                'key' => $langData['key'],
                'flag_code' => $langData['flag_code'] ?? 'US',
                'is_rtl' => $langData['is_rtl'] ?? false,
                'enabled' => $langData['enabled'] ?? true,
                'is_default' => $langData['is_default'] ?? false,
            ]);

            $createdLangs[] = $frontLang;

            if ($langData['is_default'] ?? false) {
                $defaultLang = $frontLang;
            }
        }

        // If no default was set, use the first language
        if (!$defaultLang && !empty($createdLangs)) {
            $defaultLang = $createdLangs[0];
            $defaultLang->is_default = true;
            $defaultLang->save();
        }

        // Create business settings for default language
        $defaultSettings = WebsiteSetting::getDefaultSettings($companyId);
        $defaultSettings['website_id'] = $website->id;
        $defaultSettings['website_language_id'] = $defaultLang->id;

        // Replace default values with actual business name
        $websiteSlug = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $websiteName));
        $twitterHandle = '@' . $websiteSlug;
        $websiteEmail = 'hello@' . $websiteSlug . '.com';

        $defaultSettings['website_name'] = $websiteName;
        $defaultSettings['website_tagline'] = $websiteName . ' Business';
        $defaultSettings['website_email'] = $websiteEmail;
        $defaultSettings['seo_title'] = $websiteName;
        $defaultSettings['seo_description'] = 'Welcome to ' . $websiteName . '. We provide quality services and products to our customers.';
        $defaultSettings['seo_keywords'] = strtolower($websiteName) . ', ' . $websiteSlug . ', business, services';
        $defaultSettings['seo_author'] = $websiteName;
        $defaultSettings['seo_og_site_name'] = $websiteName;
        $defaultSettings['seo_twitter_site'] = $twitterHandle;
        $defaultSettings['seo_twitter_creator'] = $twitterHandle;

        WebsiteSetting::create($defaultSettings);

        // Create default "Home" page in website_pages
        $homePage = new WebsitePage();
        $homePage->company_id = $companyId;
        $homePage->website_id = $website->id;
        $homePage->page_key = 'home';
        $homePage->save();

        // Create translation for the home page for each language
        foreach ($createdLangs as $lang) {
            $translation = new WebsitePageTranslation();
            $translation->website_page_id = $homePage->id;
            $translation->website_language_id = $lang->id;
            $translation->name = 'Home';
            $translation->show_in_header = true;
            $translation->save();
        }

        // Note: website_builder sections are NOT created automatically
        // User will add them manually through the admin panel

        return $website;
    }

    /**
     * Hook: After updating a business
     * Update front_langs based on request (respecting permissions)
     *
     * @param Website $website
     * @return Business
     */
    public function updated(Website $website)
    {
        $companyId = company()->id;
        $request = request();
        $languages = $request->languages ?? [];
        $user = user();

        if (empty($languages)) {
            return $website;
        }

        // Check user permissions for language operations
        $canCreateLanguage = $user->ability('admin', 'website_languages_create');
        $canEditLanguage = $user->ability('admin', 'website_languages_edit');
        $canDeleteLanguage = $user->ability('admin', 'website_languages_delete');

        // Get existing language IDs for this website
        $existingLangs = WebsiteLanguage::where('website_id', $website->id)->get();
        $existingLangIds = $existingLangs->pluck('id')->toArray();
        $processedIds = [];

        foreach ($languages as $langData) {
            if (!empty($langData['xid'])) {
                // Update existing language (only if user has edit permission)
                $langId = \Vinkla\Hashids\Facades\Hashids::decode($langData['xid']);
                if (!empty($langId)) {
                    $langId = $langId[0];
                    $frontLang = WebsiteLanguage::find($langId);
                    if ($frontLang && $frontLang->website_id == $website->id) {
                        if ($canEditLanguage) {
                            $frontLang->update([
                                'name' => $langData['name'],
                                'key' => $langData['key'],
                                'flag_code' => $langData['flag_code'] ?? $frontLang->flag_code,
                                'is_rtl' => $langData['is_rtl'] ?? $frontLang->is_rtl,
                                'enabled' => $langData['enabled'] ?? $frontLang->enabled,
                                'is_default' => $langData['is_default'] ?? false,
                            ]);
                        }
                        $processedIds[] = $langId;
                    }
                }
            } else {
                // Create new language (only if user has create permission)
                if ($canCreateLanguage) {
                    $frontLang = WebsiteLanguage::create([
                        'company_id' => $companyId,
                        'website_id' => $website->id,
                        'name' => $langData['name'],
                        'key' => $langData['key'],
                        'flag_code' => $langData['flag_code'] ?? 'US',
                        'is_rtl' => $langData['is_rtl'] ?? false,
                        'enabled' => $langData['enabled'] ?? true,
                        'is_default' => $langData['is_default'] ?? false,
                    ]);
                    $processedIds[] = $frontLang->id;

                    // Create translations for all existing website pages for this new language
                    $websitePages = WebsitePage::where('website_id', $website->id)->get();
                    foreach ($websitePages as $page) {
                        // Check if translation already exists
                        $existingTranslation = WebsitePageTranslation::where('website_page_id', $page->id)
                            ->where('website_language_id', $frontLang->id)
                            ->first();

                        if (!$existingTranslation) {
                            $translation = new WebsitePageTranslation();
                            $translation->website_page_id = $page->id;
                            $translation->website_language_id = $frontLang->id;
                            $translation->name = $page->page_key === 'home' ? 'Home' : ucfirst(str_replace('-', ' ', $page->page_key));
                            $translation->show_in_header = true;
                            $translation->save();
                        }
                    }
                }
            }
        }

        // Delete languages that were removed (only if user has delete permission)
        if ($canDeleteLanguage) {
            $langsToDelete = array_diff($existingLangIds, $processedIds);
            if (!empty($langsToDelete)) {
                // Don't delete if it's the only language
                if (count($processedIds) > 0) {
                    WebsiteLanguage::whereIn('id', $langsToDelete)->delete();
                }
            }
        }

        // Ensure exactly one language is default
        $defaultExists = WebsiteLanguage::where('website_id', $website->id)
            ->where('is_default', true)
            ->exists();

        if (!$defaultExists) {
            $firstLang = WebsiteLanguage::where('website_id', $website->id)->first();
            if ($firstLang) {
                $firstLang->is_default = true;
                $firstLang->save();
            }
        }

        return $website;
    }

    /**
     * Toggle favourite status for a business
     *
     * @param string $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggleFavourite($id)
    {
        try {
            $decodedId = $this->getIdFromHash($id);

            if (!$decodedId) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid business ID',
                ], 400);
            }

            $website = Website::find($decodedId);

            if (!$website) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Website not found',
                ], 404);
            }

            $website->is_favourite = !$website->is_favourite;
            $website->save();

            return response()->json([
                'status' => 'success',
                'message' => $website->is_favourite ? 'Added to favourites' : 'Removed from favourites',
                'is_favourite' => $website->is_favourite,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to toggle favourite: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Create a website from AI Wizard data
     * This endpoint creates the website along with its settings in one go
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function aiWizardCreate()
    {
        $request = request();

        // Validate required fields
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|regex:/^[a-z0-9-]+$/',
            'theme_primary_color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'theme_mode' => 'required|in:light,dark',
            'languages' => 'required|array|min:1',
            'languages.*.name' => 'required|string',
            'languages.*.key' => 'required|string',
            'languages.*.flag_code' => 'required|string',
            'website_settings' => 'required|array',
        ], [
            'slug.regex' => 'Slug must contain only lowercase letters, numbers, and hyphens',
            'theme_primary_color.regex' => 'Primary color must be a valid hex color (e.g., #3B82F6)',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $companyId = company()->id;

            // Handle category: use existing or create new
            $categoryId = null;
            if ($request->website_category_id) {
                // Use existing category
                $decodedCategoryId = \Vinkla\Hashids\Facades\Hashids::decode($request->website_category_id);
                if (!empty($decodedCategoryId)) {
                    $categoryId = $decodedCategoryId[0];
                }
            } elseif ($request->category_name) {
                // Create new category based on AI suggestion
                $newCategory = \App\Models\WebsiteCategory::create([
                    'company_id' => $companyId,
                    'name' => $request->category_name,
                ]);
                $categoryId = $newCategory->id;
            }

            // Create the website - always use 'tailwind_theme' theme
            $website = Website::create([
                'company_id' => $companyId,
                'name' => $request->name,
                'slug' => $request->slug,
                'website_category_id' => $categoryId,
                'theme_name' => 'tailwind_theme',
                'theme_primary_color' => $request->theme_primary_color,
                'theme_secondary_color' => $request->theme_secondary_color ?? null,
                'theme_mode' => $request->theme_mode,
                'use_same_header_across_all_pages' => true,
                'use_same_footer_across_all_pages' => true,
            ]);

            $languages = $request->languages ?? [];
            $websiteSettings = $request->website_settings ?? [];

            $defaultLang = null;
            $createdLangs = [];

            // Create all languages for this website
            foreach ($languages as $index => $langData) {
                $frontLang = WebsiteLanguage::create([
                    'company_id' => $companyId,
                    'website_id' => $website->id,
                    'name' => $langData['name'],
                    'key' => $langData['key'],
                    'flag_code' => $langData['flag_code'] ?? 'US',
                    'is_rtl' => $langData['is_rtl'] ?? false,
                    'enabled' => $langData['enabled'] ?? true,
                    'is_default' => $index === 0, // First language is default
                ]);

                $createdLangs[$langData['key']] = $frontLang;

                if ($index === 0) {
                    $defaultLang = $frontLang;
                }
            }

            // Create website settings for each language
            foreach ($createdLangs as $langKey => $frontLang) {
                $langSettings = $websiteSettings[$langKey] ?? [];
                $defaultSettings = WebsiteSetting::getDefaultSettings($companyId);

                $defaultSettings['website_id'] = $website->id;
                $defaultSettings['website_language_id'] = $frontLang->id;

                // Apply AI-generated settings
                $defaultSettings['website_name'] = $langSettings['website_name'] ?? $website->name;
                $defaultSettings['website_tagline'] = $langSettings['website_tagline'] ?? '';
                $defaultSettings['website_description'] = $langSettings['website_description'] ?? '';
                $defaultSettings['seo_title'] = $langSettings['seo_title'] ?? $website->name;
                $defaultSettings['seo_description'] = $langSettings['seo_description'] ?? '';
                $defaultSettings['seo_keywords'] = $langSettings['seo_keywords'] ?? '';
                $defaultSettings['seo_author'] = $langSettings['website_name'] ?? $website->name;
                $defaultSettings['seo_og_site_name'] = $langSettings['website_name'] ?? $website->name;

                // Generate social handles
                $websiteSlug = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $website->name));
                $twitterHandle = '@' . $websiteSlug;
                $websiteEmail = 'hello@' . $websiteSlug . '.com';

                $defaultSettings['website_email'] = $websiteEmail;
                $defaultSettings['seo_twitter_site'] = $twitterHandle;
                $defaultSettings['seo_twitter_creator'] = $twitterHandle;

                WebsiteSetting::create($defaultSettings);
            }

            // Create default "Home" page
            $homePage = new WebsitePage();
            $homePage->company_id = $companyId;
            $homePage->website_id = $website->id;
            $homePage->page_key = 'home';
            $homePage->save();

            // Create translation for the home page for each language
            foreach ($createdLangs as $langKey => $frontLang) {
                $translation = new WebsitePageTranslation();
                $translation->website_page_id = $homePage->id;
                $translation->website_language_id = $frontLang->id;
                $translation->name = 'Home';
                $translation->show_in_header = true;
                $translation->save();
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Website created successfully',
                'xid' => $website->xid,
                'name' => $website->name,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create website: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Toggle primary status for a business (non-saas only)
     * Only one business can be primary at a time
     *
     * @param string $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function togglePrimary($id)
    {
        try {
            // Only allow in non-saas mode
            if (app_type() != 'non-saas') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Primary business feature is only available in non-saas mode',
                ], 403);
            }

            $decodedId = $this->getIdFromHash($id);

            if (!$decodedId) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid business ID',
                ], 400);
            }

            $website = Website::find($decodedId);

            if (!$website) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Website not found',
                ], 404);
            }

            // If setting as primary, unset any other primary business first
            if (!$website->is_primary) {
                Website::where('is_primary', true)->update(['is_primary' => false]);
                $website->is_primary = true;
            } else {
                // Just toggle off
                $website->is_primary = false;
            }

            $website->save();

            return response()->json([
                'status' => 'success',
                'message' => $website->is_primary ? 'Set as primary business' : 'Removed as primary business',
                'is_primary' => $website->is_primary,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to toggle primary: ' . $e->getMessage(),
            ], 500);
        }
    }
}
