<?php

namespace App\Http\Controllers\Api;

use App\Classes\Common;
use App\Http\Controllers\ApiBaseController;
use App\Http\Requests\Api\FrontLang\IndexRequest;
use App\Http\Requests\Api\FrontLang\StoreRequest;
use App\Http\Requests\Api\FrontLang\UpdateRequest;
use App\Http\Requests\Api\FrontLang\DeleteRequest;
use App\Models\Website;
use App\Models\WebsitePage;
use App\Models\WebsitePageTranslation;
use App\Models\WebsiteSetting;
use App\Models\WebsiteLanguage;
use App\Models\WebsiteBuilder;
use Examyou\RestAPI\ApiResponse;
use Illuminate\Support\Str;
use Vinkla\Hashids\Facades\Hashids;

class WebsiteLanguageController extends ApiBaseController
{
    /**
     * The model associated with the controller.
     *
     * @var string
     */
    protected $model = WebsiteLanguage::class;

    /**
     * The request class for index operations.
     *
     * @var string
     */
    protected $indexRequest = IndexRequest::class;

    /**
     * The request class for store operations.
     *
     * @var string
     */
    protected $storeRequest = StoreRequest::class;

    /**
     * The request class for update operations.
     *
     * @var string
     */
    protected $updateRequest = UpdateRequest::class;

    /**
     * The request class for delete operations.
     *
     * @var string
     */
    protected $deleteRequest = DeleteRequest::class;

    /**
     * Modify index query to filter by website_id
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    protected function modifyIndex($query)
    {
        $request = request();

        // Filter by website_id if provided
        if ($request->has('website_id') && $request->input('website_id') && $request->input('website_id') !== 'null') {
            $decodedWebsiteId = Hashids::decode($request->input('website_id'));
            if (!empty($decodedWebsiteId)) {
                $query->where('website_id', $decodedWebsiteId[0]);
            }
        }

        return $query;
    }

    /**
     * Hook: Before storing a new front language
     * Generate key from name and set website_id
     *
     * @param WebsiteLanguage $websiteLanguage
     * @return WebsiteLanguage
     */
    public function storing(WebsiteLanguage $websiteLanguage)
    {
        $request = request();

        // Use key from request if provided (set from flag's langCode in frontend)
        // Otherwise fallback to generating from name
        if ($request->has('key') && !empty($request->key)) {
            $websiteLanguage->key = Str::limit(Str::slug($request->key), 10, '');
        } else {
            $key = Str::slug($request->name);
            $key = Str::limit($key, 10, '');
            $websiteLanguage->key = $key;
        }

        // Set website_id from request
        if ($request->has('website_id')) {
            $websiteLanguage->website_id = Common::getIdFromHash($request->website_id);
        }

        return $websiteLanguage;
    }

    /**
     * Hook: After storing a new language
     * Create website_page_translations, website_settings, and website_builder for the new language
     *
     * @param WebsiteLanguage $websiteLanguage
     * @return WebsiteLanguage
     */
    public function stored(WebsiteLanguage $websiteLanguage)
    {
        $companyId = company()->id;
        $websiteId = $websiteLanguage->website_id;

        if (!$websiteId) {
            return $websiteLanguage;
        }

        // Get the website to check header/footer settings
        $website = Website::find($websiteId);

        // Get all website pages for this website
        $websitePages = WebsitePage::where('company_id', $companyId)
            ->where('website_id', $websiteId)
            ->get();

        // Find the home page to use as source for header/footer sections
        $homePage = $websitePages->firstWhere('page_key', 'home');

        foreach ($websitePages as $websitePage) {
            // Create website_page_translation for this language
            $existingTranslation = WebsitePageTranslation::where('website_page_id', $websitePage->id)
                ->where('website_language_id', $websiteLanguage->id)
                ->first();

            if (!$existingTranslation) {
                // Try to get translation from another language to copy the name
                $sourceTranslation = WebsitePageTranslation::where('website_page_id', $websitePage->id)->first();

                WebsitePageTranslation::create([
                    'website_page_id' => $websitePage->id,
                    'website_language_id' => $websiteLanguage->id,
                    'name' => $sourceTranslation ? $sourceTranslation->name : 'Page',
                    'show_in_header' => $sourceTranslation ? $sourceTranslation->show_in_header : true,
                ]);
            }

            // Copy website_builder from another language for this website_page
            // Use GROUP BY to get unique section_keys (avoid duplicates from multiple source languages)
            $existingSectionKeys = WebsiteBuilder::where('company_id', $companyId)
                ->where('website_id', $websiteId)
                ->where('website_page_id', $websitePage->id)
                ->where('website_language_id', '!=', $websiteLanguage->id)
                ->distinct()
                ->pluck('section_key');

            foreach ($existingSectionKeys as $sectionKey) {
                // Check if this section already exists for the new language
                $sectionExists = WebsiteBuilder::where('company_id', $companyId)
                    ->where('website_id', $websiteId)
                    ->where('website_page_id', $websitePage->id)
                    ->where('website_language_id', $websiteLanguage->id)
                    ->where('section_key', $sectionKey)
                    ->exists();

                if (!$sectionExists) {
                    // For header/footer with sync enabled, copy from home page if this is not the home page
                    $sourcePageId = $websitePage->id;
                    if ($homePage && $websitePage->id !== $homePage->id) {
                        if (($sectionKey === 'header' && $website && $website->use_same_header_across_all_pages) ||
                            ($sectionKey === 'footer' && $website && $website->use_same_footer_across_all_pages)) {
                            $sourcePageId = $homePage->id;
                        }
                    }

                    // Get the source section from another language
                    $setting = WebsiteBuilder::where('company_id', $companyId)
                        ->where('website_id', $websiteId)
                        ->where('website_page_id', $sourcePageId)
                        ->where('website_language_id', '!=', $websiteLanguage->id)
                        ->where('section_key', $sectionKey)
                        ->first();

                    if ($setting) {
                        WebsiteBuilder::create([
                            'company_id' => $companyId,
                            'website_id' => $websiteId,
                            'website_page_id' => $websitePage->id,
                            'website_language_id' => $websiteLanguage->id,
                            'section_key' => $setting->section_key,
                            'section_layout' => $setting->section_layout,
                            'title' => $setting->title,
                            'subtitle' => $setting->subtitle ?? '',
                            'content' => $setting->content,
                            'content_schema' => $setting->content_schema,
                            'sort_order' => $setting->sort_order ?? 0,
                            'is_white_background' => $setting->is_white_background ?? false,
                            'padding_top' => $setting->padding_top,
                            'padding_bottom' => $setting->padding_bottom,
                        ]);
                    }
                }
            }
        }

        // Create website_settings for this website and new language
        $existingWebsiteSetting = WebsiteSetting::where('website_id', $websiteId)
            ->where('website_language_id', $websiteLanguage->id)
            ->first();

        if (!$existingWebsiteSetting) {
            // Copy from another language or use defaults
            $sourceWebsiteSetting = WebsiteSetting::where('website_id', $websiteId)->first();

            if ($sourceWebsiteSetting) {
                $newSettings = $sourceWebsiteSetting->replicate();
                $newSettings->website_language_id = $websiteLanguage->id;
                $newSettings->save();
            } else {
                // Create with defaults
                $defaultSettings = WebsiteSetting::getDefaultSettings($companyId);
                $defaultSettings['website_id'] = $websiteId;
                $defaultSettings['website_language_id'] = $websiteLanguage->id;
                WebsiteSetting::create($defaultSettings);
            }
        }

        return $websiteLanguage;
    }

    /**
     * Hook: Before updating a front language
     * Update key and handle is_default toggle
     *
     * @param WebsiteLanguage $websiteLanguage
     * @return WebsiteLanguage
     */
    public function updating(WebsiteLanguage $websiteLanguage)
    {
        $request = request();

        // Use key from request if provided (set from flag's langCode in frontend)
        // Otherwise fallback to generating from name
        if ($request->has('key') && !empty($request->key)) {
            $websiteLanguage->key = Str::limit(Str::slug($request->key), 10, '');
        } else {
            $key = Str::slug($request->name);
            $key = Str::limit($key, 10, '');
            $websiteLanguage->key = $key;
        }

        // Handle is_default toggle - ensure only one default per website
        if ($request->has('is_default') && $request->is_default) {
            WebsiteLanguage::where('website_id', $websiteLanguage->website_id)
                ->where('id', '!=', $websiteLanguage->id)
                ->update(['is_default' => false]);
        }

        return $websiteLanguage;
    }

    /**
     * Override destroy to prevent deletion of default or only language
     *
     * @param mixed ...$args
     * @return \Examyou\RestAPI\ApiResponse
     */
    public function destroy(...$args)
    {
        $xid = last(func_get_args());
        $convertedId = Hashids::decode($xid);
        $id = $convertedId[0];

        $websiteLanguage = WebsiteLanguage::find($id);

        if (!$websiteLanguage) {
            return ApiResponse::make("Resource not found", null, [], 404);
        }

        // Prevent deletion of default language
        if ($websiteLanguage->is_default) {
            return ApiResponse::make("Default language cannot be deleted. Set another language as default first.", null, [], 403);
        }

        // Prevent deletion if it's the only language for the website
        $languageCount = WebsiteLanguage::where('website_id', $websiteLanguage->website_id)->count();
        if ($languageCount <= 1) {
            return ApiResponse::make("Cannot delete the only language for this website.", null, [], 403);
        }

        return parent::destroy(...$args);
    }

    /**
     * Set a language as default for its website
     *
     * @param string $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefault($id)
    {
        try {
            $decodedId = Common::getIdFromHash($id);

            if (!$decodedId) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid language ID',
                ], 400);
            }

            $websiteLanguage = WebsiteLanguage::find($decodedId);

            if (!$websiteLanguage) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Language not found',
                ], 404);
            }

            // Unset other defaults for this website
            WebsiteLanguage::where('website_id', $websiteLanguage->website_id)
                ->where('id', '!=', $websiteLanguage->id)
                ->update(['is_default' => false]);

            // Set this as default and enable it if disabled
            $websiteLanguage->is_default = true;
            if (!$websiteLanguage->enabled) {
                $websiteLanguage->enabled = true;
            }
            $websiteLanguage->save();

            return response()->json([
                'status' => 'success',
                'message' => 'Language set as default',
                'is_default' => true,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to set default: ' . $e->getMessage(),
            ], 500);
        }
    }
}
