<?php

namespace App\Http\Controllers\Api;

use App\Classes\Common;
use App\Http\Controllers\ApiBaseController;
use App\Http\Requests\Api\WebsiteLegalPage\IndexRequest;
use App\Http\Requests\Api\WebsiteLegalPage\StoreRequest;
use App\Http\Requests\Api\WebsiteLegalPage\UpdateRequest;
use App\Http\Requests\Api\WebsiteLegalPage\DeleteRequest;
use App\Models\Website;
use App\Models\WebsiteLegalPage;
use App\Models\Lang;
use Vinkla\Hashids\Facades\Hashids;

class WebsiteLegalPageController extends ApiBaseController
{
    /**
     * The model associated with the controller.
     *
     * @var string
     */
    protected $model = WebsiteLegalPage::class;

    /**
     * The request class for index operations.
     *
     * @var string
     */
    protected $indexRequest = IndexRequest::class;

    /**
     * The request class for store operations.
     *
     * @var string
     */
    protected $storeRequest = StoreRequest::class;

    /**
     * The request class for update operations.
     *
     * @var string
     */
    protected $updateRequest = UpdateRequest::class;

    /**
     * The request class for delete operations.
     *
     * @var string
     */
    protected $deleteRequest = DeleteRequest::class;

    /**
     * Modify query to filter by website_language_id, website_id, and status
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function modifyIndex($query)
    {
        $request = request();

        // Filter by website_language_id if provided
        if ($request->has('website_language_id') && $request->input('website_language_id') && $request->input('website_language_id') !== 'null') {
            $decodedLangId = Hashids::decode($request->input('website_language_id'));
            if (!empty($decodedLangId)) {
                $query->where('website_language_id', $decodedLangId[0]);
            }
        }

        // Filter by website_id if provided
        if ($request->has('website_id') && $request->input('website_id') && $request->input('website_id') !== 'null') {
            $decodedWebsiteId = Hashids::decode($request->input('website_id'));
            if (!empty($decodedWebsiteId)) {
                $query->where('website_id', $decodedWebsiteId[0]);
            }
        }

        // Filter by status if provided (not 'all')
        if ($request->has('status') && $request->input('status') && $request->input('status') !== 'all') {
            $query->where('status', $request->input('status'));
        }

        return $query;
    }

    /**
     * Hook: Before storing a new website legal page
     * Set company_id, website_id, website_language_id, and display_order
     *
     * @param WebsiteLegalPage $websiteLegalPage
     * @return WebsiteLegalPage
     */
    public function storing(WebsiteLegalPage $websiteLegalPage)
    {
        $request = request();

        // Set company_id
        $websiteLegalPage->company_id = company()->id;

        // Handle website_language_id conversion
        if ($request->has('website_language_id') && $request->input('website_language_id')) {
            $websiteLegalPage->website_language_id = Common::getIdFromHash($request->input('website_language_id'));
        } else {
            // Default to English language
            $enLang = Lang::where('key', 'en')->first();
            $websiteLegalPage->website_language_id = $enLang ? $enLang->id : null;
        }

        // Handle website_id conversion
        if ($request->has('website_id') && $request->input('website_id')) {
            $websiteId = Common::getIdFromHash($request->input('website_id'));
            if ($websiteId && Website::where('id', $websiteId)->exists()) {
                $websiteLegalPage->website_id = $websiteId;
            }
        }

        // Handle show_in_footer boolean conversion
        if ($request->has('show_in_footer')) {
            $showInFooter = $request->input('show_in_footer');
            $websiteLegalPage->show_in_footer = filter_var($showInFooter, FILTER_VALIDATE_BOOLEAN);
        }

        // Auto-set display_order to the next available order for this website_id
        $maxDisplayOrder = WebsiteLegalPage::where('company_id', company()->id)
            ->where('website_id', $websiteLegalPage->website_id)
            ->where('website_language_id', $websiteLegalPage->website_language_id)
            ->max('display_order');

        $websiteLegalPage->display_order = ($maxDisplayOrder !== null) ? $maxDisplayOrder + 1 : 0;

        return $websiteLegalPage;
    }

    /**
     * Hook: After storing a new website legal page
     * Create entries for all other languages with the same slug and website_id
     *
     * @param WebsiteLegalPage $websiteLegalPage
     * @return void
     */
    public function stored(WebsiteLegalPage $websiteLegalPage)
    {
        // Get all languages except the one just created
        $allLangs = Lang::where('id', '!=', $websiteLegalPage->website_language_id)->get();

        foreach ($allLangs as $lang) {
            // Check if entry already exists for this slug, website_id, and website_language_id
            $exists = WebsiteLegalPage::where('company_id', $websiteLegalPage->company_id)
                ->where('website_id', $websiteLegalPage->website_id)
                ->where('slug', $websiteLegalPage->slug)
                ->where('website_language_id', $lang->id)
                ->exists();

            if (!$exists) {
                // Get max display_order for this lang
                $maxDisplayOrder = WebsiteLegalPage::where('company_id', $websiteLegalPage->company_id)
                    ->where('website_id', $websiteLegalPage->website_id)
                    ->where('website_language_id', $lang->id)
                    ->max('display_order');

                // Create entry for this language
                WebsiteLegalPage::create([
                    'company_id' => $websiteLegalPage->company_id,
                    'website_id' => $websiteLegalPage->website_id,
                    'website_language_id' => $lang->id,
                    'title' => $websiteLegalPage->title,
                    'slug' => $websiteLegalPage->slug,
                    'content' => $websiteLegalPage->content,
                    'meta_title' => $websiteLegalPage->meta_title,
                    'meta_description' => $websiteLegalPage->meta_description,
                    'meta_keywords' => $websiteLegalPage->meta_keywords,
                    'show_in_footer' => $websiteLegalPage->show_in_footer,
                    'display_order' => ($maxDisplayOrder !== null) ? $maxDisplayOrder + 1 : 0,
                    'status' => $websiteLegalPage->status,
                ]);
            }
        }
    }

    /**
     * Hook: Before updating a website legal page
     * Note: website_id and website_language_id are not updated - they remain fixed after creation
     *
     * @param WebsiteLegalPage $websiteLegalPage
     * @return WebsiteLegalPage
     */
    public function updating(WebsiteLegalPage $websiteLegalPage)
    {
        $request = request();

        // Prevent website_id and website_language_id from being updated
        // Restore original values if they were changed
        $websiteLegalPage->website_id = $websiteLegalPage->getOriginal('website_id');
        $websiteLegalPage->website_language_id = $websiteLegalPage->getOriginal('website_language_id');

        // Handle show_in_footer boolean conversion
        if ($request->has('show_in_footer')) {
            $showInFooter = $request->input('show_in_footer');
            $websiteLegalPage->show_in_footer = filter_var($showInFooter, FILTER_VALIDATE_BOOLEAN);
        }

        return $websiteLegalPage;
    }

    /**
     * Hook: After updating a website legal page
     * If slug is changed, update slug for all other language entries
     *
     * @param WebsiteLegalPage $websiteLegalPage
     * @return void
     */
    public function updated(WebsiteLegalPage $websiteLegalPage)
    {
        $originalSlug = $websiteLegalPage->getOriginal('slug');
        $newSlug = $websiteLegalPage->slug;

        // If slug has changed, update all other language entries
        if ($originalSlug !== $newSlug) {
            WebsiteLegalPage::where('company_id', $websiteLegalPage->company_id)
                ->where('website_id', $websiteLegalPage->website_id)
                ->where('slug', $originalSlug)
                ->where('id', '!=', $websiteLegalPage->id)
                ->update(['slug' => $newSlug]);
        }
    }

    /**
     * Hook: After deleting a website legal page
     * Delete all other language entries with same slug and website_id
     *
     * @param WebsiteLegalPage $websiteLegalPage
     * @return void
     */
    public function destroyed(WebsiteLegalPage $websiteLegalPage)
    {
        // Delete all other language entries with same slug and website_id
        WebsiteLegalPage::where('company_id', $websiteLegalPage->company_id)
            ->where('website_id', $websiteLegalPage->website_id)
            ->where('slug', $websiteLegalPage->slug)
            ->delete();
    }

    /**
     * Update sort order for multiple website legal pages
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateSortOrder()
    {
        $request = request();
        $items = $request->input('items', []);

        if (empty($items)) {
            return response()->json(['message' => 'No items to update']);
        }

        foreach ($items as $item) {
            $xid = $item['xid'] ?? null;
            $displayOrder = $item['display_order'] ?? 0;

            if (!$xid) {
                continue;
            }

            $id = Common::getIdFromHash($xid);
            if ($id) {
                WebsiteLegalPage::where('id', $id)
                    ->where('company_id', company()->id)
                    ->update(['display_order' => $displayOrder]);
            }
        }

        return response()->json(['message' => 'Sort order updated successfully']);
    }
}
