<?php

namespace App\Http\Middleware;

use App\Models\Website;
use App\Models\WebsiteSetting;
use App\Models\WebsiteLanguage;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class FrontBusinessMiddleware
{
    /**
     * Handle an incoming request.
     *
     * Checks if the route is /w/{slug} pattern and fetches website settings.
     * The website data and theme_name are shared with the view.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $slug = $request->route('slug');

        if ($slug) {
            // Find website by slug (without company scope since this is public frontend)
            $website = Website::withoutGlobalScopes()->where('slug', $slug)->first();

            if (!$website) {
                abort(404, 'Website not found');
            }

            // Get language from request or default to English
            // Use website_languages table filtered by website's company_id
            $langKey = $request->input('lang', 'en');
            $lang = WebsiteLanguage::withoutGlobalScopes()
                ->where('key', $langKey)
                ->where('company_id', $website->company_id)
                ->first();
            $langId = $lang ? $lang->id : null;
            $enLang = WebsiteLanguage::withoutGlobalScopes()
                ->where('key', 'en')
                ->where('company_id', $website->company_id)
                ->first();
            $enLangId = $enLang ? $enLang->id : null;

            // Fetch website settings for this website (without company scope)
            $websiteSettings = WebsiteSetting::withoutGlobalScopes()
                ->where('website_id', $website->id)
                ->where('front_lang_id', $langId)
                ->first();

            // Fallback to English if not found
            if (!$websiteSettings && $langId !== $enLangId) {
                $websiteSettings = WebsiteSetting::withoutGlobalScopes()
                    ->where('website_id', $website->id)
                    ->where('front_lang_id', $enLangId)
                    ->first();
            }

            // Get theme name from websites table (not website_settings)
            $themeName = $website->theme_name ?? 'tailwind_theme';

            // Store in request for access by Livewire components
            $request->attributes->set('front_website', $website);
            $request->attributes->set('front_website_settings', $websiteSettings);
            $request->attributes->set('front_theme_name', $themeName);
            $request->attributes->set('front_lang_id', $langId);

            // Share with all views
            view()->share('frontWebsite', $website);
            view()->share('frontWebsiteSettings', $websiteSettings);
            view()->share('frontThemeName', $themeName);
        }

        return $next($request);
    }
}
