<?php

namespace App\Http\Requests\Api\AiSettings;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class TestConnectionRequest extends FormRequest
{
    /**
     * Handle a failed validation attempt.
     */
    protected function failedValidation(Validator $validator)
    {
        $response = response()->json([
            'error' => [
                'code' => 422,
                'message' => 'The given data was invalid.',
                'details' => $validator->errors()
            ],
        ], 422);

        throw new HttpResponseException($response);
    }

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        $user = auth('api')->user();

        if (app_type() == 'saas') {
            return $user && $user->is_superadmin;
        } else {
            return $user && $user->ability('admin', 'ai_settings_edit');
        }
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation()
    {
        $provider = $this->input('provider');

        // If provider is 'none', cannot test connection
        if ($provider === 'none') {
            $response = response()->json([
                'status' => 'error',
                'message' => 'Cannot test connection when no provider is selected.',
            ], 400);

            throw new HttpResponseException($response);
        }

        // Restrict self-hosted providers to local environment only
        if (in_array($provider, ['ollama', 'openai_compatible']) && env('APP_ENV') !== 'local') {
            $response = response()->json([
                'status' => 'error',
                'message' => 'Self-hosted providers are only available in local environment.',
            ], 403);

            throw new HttpResponseException($response);
        }
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $provider = $this->input('provider');

        $rules = [
            'provider' => [
                'required',
                'string',
                'in:none,openai,claude,perplexity,ollama,openai_compatible'
            ],
            'model' => 'required|string|max:255',
        ];

        // Add conditional validation based on provider type
        if ($provider === 'ollama') {
            // Ollama requires base_url, API key is optional
            $rules['base_url'] = 'required|url|max:255';
        } elseif ($provider === 'openai_compatible') {
            // OpenAI-compatible requires api_key and base_url
            $rules['api_key'] = 'required|string|max:255';
            $rules['base_url'] = 'required|url|max:255';
        } else {
            // Cloud providers require API key
            $rules['api_key'] = 'required|string|max:255';
        }

        return $rules;
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages()
    {
        $provider = $this->input('provider');

        $messages = [
            'provider.required' => 'Please select a provider',
            'provider.in' => 'Invalid provider selected',
            'model.required' => 'Model is required',
        ];

        // Add provider-specific messages
        if ($provider === 'ollama') {
            $messages['base_url.required'] = 'Base URL is required for Ollama';
            $messages['base_url.url'] = 'Please enter a valid URL';
        } elseif ($provider === 'openai_compatible') {
            $messages['api_key.required'] = 'API key is required for OpenAI-compatible provider';
            $messages['base_url.required'] = 'Base URL is required for OpenAI-compatible provider';
            $messages['base_url.url'] = 'Please enter a valid URL';
        } else {
            $messages['api_key.required'] = 'API key is required';
        }

        return $messages;
    }
}
