<?php

namespace App\Http\Requests\Api\AiSettings;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class UpdateRequest extends FormRequest
{
    /**
     * Handle a failed validation attempt.
     */
    protected function failedValidation(Validator $validator)
    {
        $response = response()->json([
            'error' => [
                'code' => 422,
                'message' => 'The given data was invalid.',
                'details' => $validator->errors()
            ],
        ], 422);

        throw new HttpResponseException($response);
    }

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        $user = auth('api')->user();

        if (app_type() == 'saas') {
            return $user && $user->is_superadmin;
        } else {
            return $user && $user->ability('admin', 'ai_settings_edit');
        }
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation()
    {
        $provider = $this->input('provider');

        // If provider is 'none', no further validation needed
        if ($provider === 'none') {
            return;
        }

        // Restrict self-hosted providers to local environment only
        if (in_array($provider, ['ollama', 'openai_compatible']) && env('APP_ENV') !== 'local') {
            $response = response()->json([
                'error' => [
                    'code' => 403,
                    'message' => 'Self-hosted providers are only available in local environment.',
                    'details' => [
                        'provider' => ['Self-hosted providers are only available in local environment.']
                    ]
                ],
            ], 403);

            throw new HttpResponseException($response);
        }
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $provider = $this->input('provider');

        $rules = [
            'provider' => [
                'required',
                'string',
                'in:none,openai,claude,perplexity,ollama,openai_compatible'
            ],
        ];

        // If provider is 'none', no other fields are required
        if ($provider === 'none') {
            return $rules;
        }

        // Add base rules for all providers except 'none'
        $rules['model'] = 'required|string|max:255';
        $rules['max_tokens'] = 'nullable|integer|min:100|max:8000';

        // Add conditional validation based on provider type
        if ($provider === 'ollama') {
            // Ollama requires base_url and model, API key is optional
            $rules['base_url'] = 'required|url|max:255';
        } elseif ($provider === 'openai_compatible') {
            // OpenAI-compatible requires api_key, base_url, and model
            $rules['api_key'] = 'required|string|max:255';
            $rules['base_url'] = 'required|url|max:255';
        } else {
            // Cloud providers (openai, claude, perplexity) require API key
            $rules['api_key'] = 'required|string|max:255';
        }

        return $rules;
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages()
    {
        $provider = $this->input('provider');

        $messages = [
            'provider.required' => 'Please select a provider',
            'provider.in' => 'Invalid provider selected',
            'model.required' => 'Model is required',
            'model.string' => 'Model must be a valid string',
            'max_tokens.integer' => 'Max tokens must be a number',
            'max_tokens.min' => 'Max tokens must be at least 100',
            'max_tokens.max' => 'Max tokens cannot exceed 8000',
        ];

        // Add provider-specific messages
        if ($provider === 'ollama') {
            $messages['base_url.required'] = 'Base URL is required for Ollama';
            $messages['base_url.url'] = 'Please enter a valid URL';
        } elseif ($provider === 'openai_compatible') {
            $messages['api_key.required'] = 'API key is required for OpenAI-compatible provider';
            $messages['base_url.required'] = 'Base URL is required for OpenAI-compatible provider';
            $messages['base_url.url'] = 'Please enter a valid URL';
        } else {
            $messages['api_key.required'] = 'API key is required';
        }

        return $messages;
    }
}
