<?php

namespace App\Http\Requests\Api\WebsiteBuilder;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Vinkla\Hashids\Facades\Hashids;

class StoreRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     * Only section_key is required - data is copied from theme_sections table
     *
     * @return array
     */
    public function rules()
    {
        $companyId = company()->id;

        // Get website_language_id from request for unique validation
        $langId = null;
        if ($this->has('website_language_id') && $this->input('website_language_id')) {
            $langIdDecoded = Hashids::decode($this->input('website_language_id'));
            $langId = $langIdDecoded[0] ?? null;
        }

        // Get website_id from request for unique validation
        $websiteId = null;
        if ($this->has('website_id') && $this->input('website_id')) {
            $websiteIdDecoded = Hashids::decode($this->input('website_id'));
            $websiteId = $websiteIdDecoded[0] ?? null;
        }

        // Get website_page_id from request for unique validation
        $websitePageId = null;
        if ($this->has('website_page_id') && $this->input('website_page_id')) {
            $websitePageIdDecoded = Hashids::decode($this->input('website_page_id'));
            $websitePageId = $websitePageIdDecoded[0] ?? null;
        }

        return [
            'section_key' => [
                'required',
                'string',
                'max:100',
                // Must exist in theme_sections table
                Rule::exists('theme_sections', 'section_key')->where(function ($query) {
                    $query->where('is_active', true);
                }),
                // Must be unique per company, language, website, and website_page
                Rule::unique('website_builder')->where(function ($query) use ($companyId, $langId, $websiteId, $websitePageId) {
                    $query->where('company_id', $companyId);
                    if ($langId) {
                        $query->where('website_language_id', $langId);
                    }
                    if ($websiteId) {
                        $query->where('website_id', $websiteId);
                    } else {
                        $query->whereNull('website_id');
                    }
                    if ($websitePageId) {
                        $query->where('website_page_id', $websitePageId);
                    } else {
                        $query->whereNull('website_page_id');
                    }
                    return $query;
                })
            ],
            'website_language_id' => 'nullable|string',
            'website_id' => 'nullable|string',
            'website_page_id' => 'nullable|string',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'section_key.exists' => 'The selected section template does not exist or is not active.',
            'section_key.unique' => 'This section has already been added.',
        ];
    }
}
