<?php

namespace App\Http\Requests\Api\WebsiteBuilder;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Vinkla\Hashids\Facades\Hashids;

class UpdateRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */

    public function authorize()
    {
        return true;
    }

    /**
     * Prepare the data for validation.
     * Decode content JSON string if needed.
     *
     * @return void
     */
    protected function prepareForValidation()
    {
        $data = [];

        if ($this->has('content') && is_string($this->content)) {
            $decoded = json_decode($this->content, true);
            $data['content'] = is_array($decoded) ? $decoded : [];
        }

        if (!empty($data)) {
            $this->merge($data);
        }
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $companyId = company()->id;
        $convertedId = Hashids::decode($this->route('website_builder'));
        $id = $convertedId[0] ?? null;

        // Get website_language_id from request for unique validation
        $langId = null;
        if ($this->has('website_language_id') && $this->input('website_language_id')) {
            $langIdDecoded = Hashids::decode($this->input('website_language_id'));
            $langId = $langIdDecoded[0] ?? null;
        }

        // Check if this is a layout section (header/footer) - title not required for these
        $sectionKey = $this->input('section_key');
        $isLayoutSection = in_array($sectionKey, ['header', 'footer']);

        return [
            'section_key' => [
                'required',
                'string',
                'max:100',
                Rule::unique('website_builder')->where(function ($query) use ($companyId, $langId) {
                    $query->where('company_id', $companyId);
                    if ($langId) {
                        $query->where('website_language_id', $langId);
                    }
                    return $query;
                })->ignore($id)
            ],
            'website_language_id' => 'nullable|string',
            'section_layout' => 'required|string|max:100',
            'title' => $isLayoutSection ? 'nullable|string|max:255' : 'required|string|max:255',
            'subtitle' => 'nullable|string|max:500',
            'content' => 'nullable|array',
            'image' => ($this->hasFile('image') && $this->file('image')->isValid())
                ? '' . \App\Classes\Files::fileValidationRule('image')
                : 'nullable'
        ];
    }
}
