<?php

namespace App\Http\Requests\Api\WebsitePage;

use Illuminate\Foundation\Http\FormRequest;

class UpdateRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */

    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $company = company();
        $websiteId = $this->website_id;
        $currentId = $this->route('website_page');

        return [
            'page_key' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-z0-9]+(?:-[a-z0-9]+)*$/',
                function ($attribute, $value, $fail) use ($company, $websiteId, $currentId) {
                    $decodedCurrentId = $this->getDecodedId($currentId);

                    $exists = \App\Models\WebsitePage::where('page_key', $value)
                        ->where('website_id', $this->getDecodedWebsiteId($websiteId))
                        ->where('id', '!=', $decodedCurrentId)
                        ->when($company && !$company->is_global, function ($query) use ($company) {
                            return $query->where('company_id', $company->id);
                        })
                        ->exists();

                    if ($exists) {
                        $fail('The page key already exists for this website.');
                    }
                },
            ],
            'website_id' => 'required',
            'translations' => 'required|array|min:1',
            'translations.*.website_language_id' => 'required|string',
            'translations.*.name' => 'required|string|max:255',
            'translations.*.show_in_header' => 'nullable|boolean',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'page_key.required' => 'The page key is required.',
            'page_key.regex' => 'The page key must be lowercase letters, numbers, and hyphens only (e.g., about-us).',
            'translations.required' => 'At least one translation is required.',
            'translations.*.name.required' => 'The page name is required for all languages.',
        ];
    }

    /**
     * Get decoded ID from hashed value
     */
    protected function getDecodedId($id)
    {
        if (!$id) {
            return null;
        }

        return \App\Classes\Common::getIdFromHash($id);
    }

    /**
     * Get decoded website ID from hashed value
     */
    protected function getDecodedWebsiteId($websiteId)
    {
        if (!$websiteId) {
            return null;
        }

        return \App\Classes\Common::getIdFromHash($websiteId);
    }
}
