<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Website;
use App\Models\WebsiteSetting;
use App\Models\WebsiteBuilder;
use App\Models\Service;
use App\Models\WebsiteLegalPage;
use App\Models\WebsiteLanguage;

/**
 * Base Livewire Component for Laundry Frontend Pages
 *
 * This class provides common functionality for all frontend Livewire components
 * including theme, SEO, company info, navigation, social links, and footer data.
 *
 * Uses magic getter/setter pattern similar to stockifly-saas FrontBaseController
 * to allow dynamic property access.
 *
 * Child components should extend this class and call parent::mount() to initialize
 * common data, then add their page-specific data.
 *
 * Supports website-specific settings when accessed via /w/{slug} route pattern.
 */
abstract class LivewireBaseComponent extends Component
{
    /**
     * Data container for dynamic properties
     */
    protected array $data = [];

    // ============================================================
    // Common properties available to all frontend pages
    // ============================================================
    public string $layout = 'tailwind_theme::layouts.app';
    public array $theme = [];
    public array $seo = [];
    public array $company = [];
    public array $header = [];
    public array $nav = [];
    public array $social = [];
    public array $footer = [];
    public array $services = [];

    // Website-specific properties (for /w/{slug} routes)
    public ?Website $frontWebsite = null;
    public ?string $websiteSlug = null;
    public string $themeName = 'tailwind_theme';

    // Language properties
    public array $languages = [];
    public string $currentLangKey = 'en';
    public ?int $currentLangId = null;

    // Dynamic navigation items from website_builder
    public array $navItems = [];

    /**
     * Magic setter for dynamic properties
     */
    public function __set($name, $value)
    {
        $this->data[$name] = $value;
    }

    /**
     * Magic getter for dynamic properties
     */
    public function __get($name)
    {
        return $this->data[$name] ?? null;
    }

    /**
     * Magic isset check for dynamic properties
     */
    public function __isset($name)
    {
        return isset($this->data[$name]);
    }

    /**
     * Initialize common data for all frontend pages
     * Child components should call parent::mount() first
     *
     * @param string|null $slug Website slug for /w/{slug} routes
     */
    public function mount($slug = null)
    {
        $this->websiteSlug = $slug;
        $this->loadWebsite(); // Load website first for language filtering by company_id
        $this->loadLanguages();
        $this->initializeCommonData();
        $this->loadNavigationItems();
    }

    /**
     * Load available languages and detect current language from cookie or shared view data
     */
    protected function loadLanguages(): void
    {
        // Check for temp_lang query parameter (for live preview purposes)
        // This takes priority and doesn't affect the stored language preference
        $tempLang = request()->query('temp_lang');

        if ($tempLang) {
            // Use temp_lang for preview - will be resolved to ID after loading languages
            $this->currentLangKey = $tempLang;
        } else {
            // Try to get language from shared view data (set by route), fallback to cookie
            $sharedLangKey = view()->shared('frontLangKey');
            $sharedLangId = view()->shared('frontLangId');

            if ($sharedLangKey && $sharedLangId) {
                $this->currentLangKey = $sharedLangKey;
                $this->currentLangId = $sharedLangId;
            } else {
                // Get current language from slug-based cookie (same pattern as theme mode)
                // Cookie key format: lang_{slug} or lang_default
                $langCookieKey = 'lang_' . ($this->websiteSlug ?? 'default');
                $this->currentLangKey = request()->cookie($langCookieKey) ?? 'en';
            }
        }

        // Load all active languages from front_langs table
        // Filter by website_id if we have a website context
        try {
            $query = WebsiteLanguage::where('enabled', 1)->orderBy('name');

            // If we have a website loaded, filter by its website_id to show only languages specific to this website
            if ($this->frontWebsite) {
                $query->withoutGlobalScopes()->where('website_id', $this->frontWebsite->id);
            }

            $langs = $query->get();

            $this->languages = $langs->map(function ($lang) {
                return [
                    'id' => $lang->id,
                    'xid' => $lang->xid,
                    'key' => $lang->key,
                    'name' => $lang->name,
                    'flag_code' => $lang->flag_code,
                    'flag_emoji' => $this->getFlagEmoji($lang->flag_code),
                    'is_rtl' => $lang->is_rtl,
                ];
            })->toArray();

            // If lang_id not yet set from shared data or temp_lang is used, get it from the key
            // temp_lang always needs to resolve the ID from the key
            $tempLang = request()->query('temp_lang');
            if (!$this->currentLangId || $tempLang) {
                $currentLang = $langs->firstWhere('key', $this->currentLangKey);
                if ($currentLang) {
                    $this->currentLangId = $currentLang->id;
                } else {
                    // Fallback to English if the language key is not found
                    $englishLang = $langs->firstWhere('key', 'en');
                    $this->currentLangId = $englishLang?->id;
                    $this->currentLangKey = 'en';
                }
            }
        } catch (\Exception $e) {
            // Default to English if error
            $this->languages = [
                ['id' => 1, 'xid' => null, 'key' => 'en', 'name' => 'English', 'flag_code' => 'us', 'flag_emoji' => '🇺🇸', 'is_rtl' => false],
            ];
            $this->currentLangId = $this->currentLangId ?? 1;
        }
    }

    /**
     * Convert country code to flag emoji
     */
    protected function getFlagEmoji(?string $countryCode): string
    {
        if (!$countryCode) {
            return '';
        }

        $countryCode = strtoupper($countryCode);

        // Regional indicator symbols start at 0x1F1E6 for 'A'
        $flagEmoji = '';
        for ($i = 0; $i < strlen($countryCode); $i++) {
            $char = ord($countryCode[$i]) - ord('A') + 0x1F1E6;
            $flagEmoji .= mb_chr($char);
        }

        return $flagEmoji;
    }

    /**
     * Load website from slug if provided, or from shared view data (for primary website root URL)
     */
    protected function loadWebsite(): void
    {
        // First check if website is shared via view()->share() (from routes/front.php)
        // This handles the primary website root URL case where no slug is passed
        try {
            $sharedWebsite = app('view')->shared('frontWebsite', null);
            $sharedSlug = app('view')->shared('frontWebsiteSlug', null);

            if ($sharedWebsite) {
                $this->frontWebsite = $sharedWebsite;
                $this->websiteSlug = $sharedSlug ?? $sharedWebsite->slug;
                return;
            }
        } catch (\Exception $e) {
            // Continue if shared data not available
        }

        if ($this->websiteSlug) {
            // Try to get from middleware first
            $this->frontWebsite = request()->attributes->get('front_website');

            // If not from middleware, fetch directly
            if (!$this->frontWebsite) {
                $this->frontWebsite = Website::withoutGlobalScopes()
                    ->where('slug', $this->websiteSlug)
                    ->first();
            }
        }
    }

    /**
     * Load navigation items from website_page_translations where show_in_header is enabled
     * Navigation now comes from website pages, not from website_builder
     */
    protected function loadNavigationItems(): void
    {
        try {
            if (!$this->frontWebsite) {
                $this->navItems = [];
                return;
            }

            // Get website pages with their translations for the current language
            $websitePages = \App\Models\WebsitePage::withoutGlobalScopes()
                ->where('website_id', $this->frontWebsite->id)
                ->with(['translations' => function ($query) {
                    $query->where('website_language_id', $this->currentLangId)
                        ->where('show_in_header', true);
                }])
                ->orderBy('sort_order')
                ->get();

            // Filter to only pages with show_in_header translation
            $navPages = $websitePages->filter(function ($page) {
                return $page->translations->isNotEmpty();
            });

            // Fallback to English if no nav items found for current language
            if ($navPages->isEmpty() && $this->currentLangKey !== 'en') {
                $enLangQuery = WebsiteLanguage::where('key', 'en');
                $enLangQuery->withoutGlobalScopes()->where('company_id', $this->frontWebsite->company_id);
                $enLang = $enLangQuery->first();

                if ($enLang) {
                    $websitePages = \App\Models\WebsitePage::withoutGlobalScopes()
                        ->where('website_id', $this->frontWebsite->id)
                        ->with(['translations' => function ($query) use ($enLang) {
                            $query->where('website_language_id', $enLang->id)
                                ->where('show_in_header', true);
                        }])
                        ->orderBy('sort_order')
                        ->get();

                    $navPages = $websitePages->filter(function ($page) {
                        return $page->translations->isNotEmpty();
                    });
                }
            }

            $this->navItems = $navPages->map(function ($page) {
                $translation = $page->translations->first();
                $pageKey = $page->page_key;

                // Generate URL based on page_key
                // 'home' links to root, other pages link to their page_key URL
                $url = $this->generateWebsitePageUrl($pageKey);

                return [
                    'name' => $translation->name ?? $pageKey,
                    'page_key' => $pageKey,
                    'url' => $url,
                    'sort_order' => $page->sort_order,
                ];
            })->toArray();
        } catch (\Exception $e) {
            // Fallback to empty array on error
            $this->navItems = [];
        }
    }

    /**
     * Generate URL for a website page based on page_key
     * 'home' links to root URL, other pages link to their page_key URL
     *
     * @param string $pageKey
     * @return string
     */
    protected function generateWebsitePageUrl(string $pageKey): string
    {
        // For 'home' page, return the root URL for this website
        if ($pageKey === 'home') {
            if (app_type() === 'non-saas' && $this->frontWebsite && $this->frontWebsite->is_primary) {
                return '/';
            }
            return '/' . website_route_prefix() . '/' . ($this->websiteSlug ?? $this->frontWebsite->slug ?? '');
        }

        // For other pages, generate the page URL
        if (app_type() === 'non-saas' && $this->frontWebsite && $this->frontWebsite->is_primary) {
            return '/' . $pageKey;
        }

        return '/' . website_route_prefix() . '/' . ($this->websiteSlug ?? $this->frontWebsite->slug ?? '') . '/' . $pageKey;
    }

    /**
     * Initialize all common data for frontend pages
     */
    protected function initializeCommonData(): void
    {
        // Check if we have website-specific settings from middleware
        $request = request();
        $settings = null;

        // Website is already loaded in loadWebsite(), use it to get settings
        // Theme configuration is now stored in Website table, not WebsiteSetting
        if ($this->frontWebsite) {
            $settings = $request->attributes->get('front_website_settings');
            // Get theme name from website table (not website_settings)
            $this->themeName = $this->frontWebsite->theme_name ?? 'tailwind_theme';

            // Fetch settings for specific website and language (for non-theme settings like SEO, etc.)
            if (!$settings) {
                $settings = WebsiteSetting::withoutGlobalScopes()
                    ->where('website_id', $this->frontWebsite->id)
                    ->where('website_language_id', $this->currentLangId)
                    ->first();

                // Fallback to English if no settings found for the current language
                if (!$settings && $this->currentLangKey !== 'en') {
                    $enLang = WebsiteLanguage::withoutGlobalScopes()
                        ->where('key', 'en')
                        ->where('company_id', $this->frontWebsite->company_id)
                        ->first();
                    $settings = WebsiteSetting::withoutGlobalScopes()
                        ->where('website_id', $this->frontWebsite->id)
                        ->where('website_language_id', $enLang?->id)
                        ->first();
                }

                // Final fallback: get any settings for this website
                if (!$settings) {
                    $settings = WebsiteSetting::withoutGlobalScopes()
                        ->where('website_id', $this->frontWebsite->id)
                        ->first();
                }
            }
        }

        // Fallback to default settings if not found (for SEO, header, footer, etc.)
        if (!$settings) {
            $settings = WebsiteSetting::getSettings($this->currentLangId);
        }

        // Get theme configuration from Website table (not WebsiteSetting)
        // If no frontWebsite, try to get a default website for theme settings
        $themeWebsite = $this->frontWebsite;
        if (!$themeWebsite) {
            $themeWebsite = Website::withoutGlobalScopes()->first();
        }

        // Set theme name from website
        if ($themeWebsite) {
            $this->themeName = $themeWebsite->theme_name ?? 'tailwind_theme';
        }

        // Set dynamic layout based on theme name
        $this->layout = $this->themeName . '::layouts.app';

        // ============================================================
        // THEME CONFIGURATION (from Website table, not WebsiteSetting)
        // ============================================================
        $primaryColor = $themeWebsite->theme_primary_color ?? '#2563eb';
        $secondaryColor = $themeWebsite->theme_secondary_color ?? null;

        // Get theme mode from cookie (slug-based) or fallback to database setting
        // Cookie key format: theme_{slug} or theme_default
        $themeCookieKey = 'theme_' . ($this->websiteSlug ?? 'default');
        $userThemeMode = request()->cookie($themeCookieKey);
        $themeMode = $userThemeMode ?? ($themeWebsite->theme_mode ?? 'light');

        // Determine direction based on current language's is_rtl property
        $currentLang = collect($this->languages)->firstWhere('key', $this->currentLangKey);
        $directionMode = ($currentLang && $currentLang['is_rtl']) ? 'rtl' : 'ltr';

        $this->theme = [
            'mode' => $themeMode,
            'direction' => $directionMode,
            'primary' => $this->generateColorPalette($primaryColor),
            'secondary' => $this->generateColorPalette(
                $secondaryColor ?? $this->generateSecondaryColor($primaryColor)
            ),
        ];

        // ============================================================
        // SEO DEFAULTS
        // ============================================================
        $this->seo = [
            'title' => $settings->seo_title ?? 'SparkleWash - Premium Laundry Services | Free Pickup & Delivery',
            'description' => $settings->seo_description ?? 'Professional laundry and dry cleaning services with free pickup & delivery.',
            'keywords' => $settings->seo_keywords ?? 'laundry service, dry cleaning, wash and fold, pickup delivery laundry',
            'author' => $settings->seo_author ?? 'SparkleWash',
            'robots' => $settings->seo_robots ?? 'index, follow',
            'canonical' => url()->current(),
            'og' => [
                'type' => $settings->seo_og_type ?? 'website',
                'locale' => $settings->seo_og_locale ?? 'en_US',
                'site_name' => $settings->seo_og_site_name ?? ($settings->company_name ?? 'SparkleWash'),
                'image' => $settings->seo_og_image_url ?? asset('images/small_dark.png'),
                'image_width' => '1200',
                'image_height' => '630',
                'image_alt' => ($settings->company_name ?? 'SparkleWash') . ' - ' . ($settings->company_tagline ?? 'Premium Laundry Services'),
            ],
            'twitter' => [
                'card' => $settings->seo_twitter_card ?? 'summary_large_image',
                'site' => $settings->seo_twitter_site ?? '@sparklewash',
                'creator' => $settings->seo_twitter_creator ?? '@sparklewash',
            ],
        ];

        // ============================================================
        // COMPANY INFO
        // ============================================================
        // Default logos from public/images folder
        $defaultLightLogo = asset('images/light.png');
        $defaultDarkLogo = asset('images/dark.png');
        $defaultSmallLightLogo = asset('images/small_light.png');
        $defaultSmallDarkLogo = asset('images/small_dark.png');
        $defaultFavicon = asset('images/favicon.png');

        $this->company = [
            'name' => $settings->website_name ?? 'SparkleWash',
            'tagline' => $settings->website_tagline ?? 'Premium Laundry Services',
            'description' => $settings->website_description ?? '',
            'phone' => $settings->website_phone ?? '+1 (555) 123-4567',
            'email' => $settings->website_email ?? 'hello@sparklewash.com',
            'address' => $settings->website_address ?? '123 Clean Street, Washville, WV 12345',
            'hours' => $settings->website_hours ?? 'Mon - Sat: 7:00 AM - 9:00 PM | Sun: 8:00 AM - 6:00 PM',
            // Logo fields for theme mode support
            'light_logo' => $settings->light_logo_url ?? $defaultLightLogo,
            'dark_logo' => $settings->dark_logo_url ?? $defaultDarkLogo,
            'light_logo_small' => $settings->light_logo_small_url ?? $defaultSmallLightLogo,
            'dark_logo_small' => $settings->dark_logo_small_url ?? $defaultSmallDarkLogo,
            // Legacy logo field (uses light_logo as default for backwards compatibility)
            'logo' => $settings->light_logo_url ?? $defaultLightLogo,
            'favicon' => $settings->website_favicon_url ?? $defaultFavicon,
            // Header/Footer logo display type
            'header_logo_type' => $settings->header_logo_type ?? 'logo_with_name',
        ];

        // ============================================================
        // HEADER SETTINGS
        // ============================================================
        $this->header = [
            'show_phone' => $settings->header_show_phone ?? true,
            'sticky' => $settings->header_sticky ?? true,
            'show_theme_toggle' => $settings->header_show_theme_toggle ?? true,
        ];

        // ============================================================
        // NAVIGATION
        // ============================================================
        $this->nav = [
            'services' => $settings->nav_services ?? 'Services',
            'how_it_works' => $settings->nav_how_it_works ?? 'How It Works',
            'pricing' => $settings->nav_pricing ?? 'Pricing',
            'reviews' => $settings->nav_reviews ?? 'Reviews',
            'faq' => $settings->nav_faq ?? 'FAQ',
            'contact' => $settings->nav_contact ?? 'Contact',
            'schedule_pickup' => $settings->nav_schedule_pickup ?? 'Schedule Pickup',
        ];

        // ============================================================
        // SOCIAL LINKS
        // ============================================================
        $socialLinks = $settings->social_links ?? [];
        $this->social = is_array($socialLinks) && !empty($socialLinks) ? $socialLinks : [
            ['name' => 'Facebook', 'url' => '#', 'icon' => 'facebook'],
            ['name' => 'Instagram', 'url' => '#', 'icon' => 'instagram'],
            ['name' => 'Twitter', 'url' => '#', 'icon' => 'twitter'],
            ['name' => 'LinkedIn', 'url' => '#', 'icon' => 'linkedin'],
        ];

        // ============================================================
        // FOOTER DATA
        // ============================================================
        $footerCopyright = $settings->footer_copyright ?? '© %year% %company%. All rights reserved.';
        $footerCopyright = str_replace('%year%', date('Y'), $footerCopyright);
        $footerCopyright = str_replace('%company%', $this->company['name'], $footerCopyright);

        $this->footer = [
            'description' => $settings->footer_description ?? 'Professional laundry and dry cleaning services with free pickup & delivery.',
            'quick_links_title' => $settings->footer_quick_links_title ?? 'Quick Links',
            'link_groups' => $settings->footer_link_groups ?? [
                [
                    'title' => 'Our Services',
                    'links' => [
                        ['title' => 'Wash & Fold', 'url' => '#services'],
                        ['title' => 'Dry Cleaning', 'url' => '#services'],
                        ['title' => 'Ironing', 'url' => '#services'],
                        ['title' => 'Alterations', 'url' => '#services'],
                    ],
                ],
                [
                    'title' => 'Company',
                    'links' => [
                        ['title' => 'About Us', 'url' => '#about'],
                        ['title' => 'Our Team', 'url' => '#team'],
                        ['title' => 'Careers', 'url' => '#careers'],
                        ['title' => 'Contact', 'url' => '#contact'],
                    ],
                ],
                [
                    'title' => 'Support',
                    'links' => [
                        ['title' => 'FAQ', 'url' => '#faq'],
                        ['title' => 'Help Center', 'url' => '#help'],
                        ['title' => 'Service Areas', 'url' => '#service-areas'],
                    ],
                ],
            ],
            'contact_title' => $settings->footer_contact_title ?? 'Contact Us',
            'copyright' => $footerCopyright,
        ];

        // Add show_social_icon to social array for global use
        $this->social['show_icon'] = $settings->show_social_icon ?? true;
    }

    /**
     * Get the base layout with common data
     * Child components can override this to customize layout data
     */
    protected function getLayoutData(): array
    {
        return [
            'title' => $this->seo['title'] ?? 'Laundry Services',
            'seo' => $this->seo,
            'theme' => $this->theme,
            'themeMode' => $this->theme['mode'] ?? 'light',
            'themeColor' => $this->theme['primary']['600'] ?? '#0284c7',
            'direction' => $this->theme['direction'] ?? 'ltr',
            // Pass data to layout for header/footer includes
            'company' => $this->company,
            'header' => $this->header,
            'nav' => $this->nav,
            'navItems' => $this->navItems,
            'services' => $this->services,
            'social' => $this->social,
            'footerData' => $this->footer,
            // Language data for header dropdown
            'languages' => $this->languages,
            'currentLangKey' => $this->currentLangKey,
            // Website slug for route generation
            'websiteSlug' => $this->websiteSlug,
            // Custom code from website
            'customCss' => $this->frontWebsite?->custom_css ?? '',
            'customJs' => $this->frontWebsite?->custom_js ?? '',
        ];
    }

    /**
     * Render with the default layout
     * Child components should override this to return their view
     */
    public function render()
    {
        return $this->view()
            ->layout($this->layout, $this->getLayoutData());
    }

    /**
     * Set layout for this page
     */
    protected function setLayout(string $layout): void
    {
        $this->layout = $layout;
    }

    /**
     * Update SEO data for specific pages
     */
    protected function setSeo(array $seoData): void
    {
        $this->seo = array_merge($this->seo, $seoData);
    }

    /**
     * Set page title (also updates SEO title)
     */
    protected function setTitle(string $title): void
    {
        $this->seo['title'] = $title;
    }

    /**
     * Set page description (also updates SEO description)
     */
    protected function setDescription(string $description): void
    {
        $this->seo['description'] = $description;
    }

    /**
     * Get all common data as array
     */
    public function getCommonData(): array
    {
        return [
            'theme' => $this->theme,
            'seo' => $this->seo,
            'company' => $this->company,
            'header' => $this->header,
            'nav' => $this->nav,
            'social' => $this->social,
            'footer' => $this->footer,
            'services' => $this->services,
        ];
    }

    // ============================================================
    // COLOR PALETTE GENERATION METHODS
    // ============================================================

    protected function generateColorPalette(string $hexColor): array
    {
        $hex = ltrim($hexColor, '#');
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        $hsl = $this->rgbToHsl($r, $g, $b);

        return [
            '50'  => $this->hslToHex($hsl['h'], max(5, $hsl['s'] - 30), 97),
            '100' => $this->hslToHex($hsl['h'], max(10, $hsl['s'] - 20), 93),
            '200' => $this->hslToHex($hsl['h'], max(15, $hsl['s'] - 10), 86),
            '300' => $this->hslToHex($hsl['h'], $hsl['s'], 75),
            '400' => $this->hslToHex($hsl['h'], $hsl['s'], 62),
            '500' => $this->hslToHex($hsl['h'], $hsl['s'], 50),
            '600' => $this->hslToHex($hsl['h'], $hsl['s'], 43),
            '700' => $this->hslToHex($hsl['h'], $hsl['s'], 35),
            '800' => $this->hslToHex($hsl['h'], min(100, $hsl['s'] + 5), 28),
            '900' => $this->hslToHex($hsl['h'], min(100, $hsl['s'] + 10), 22),
        ];
    }

    protected function rgbToHsl(int $r, int $g, int $b): array
    {
        $r /= 255;
        $g /= 255;
        $b /= 255;
        $max = max($r, $g, $b);
        $min = min($r, $g, $b);
        $l = ($max + $min) / 2;
        if ($max === $min) {
            $h = $s = 0;
        } else {
            $d = $max - $min;
            $s = $l > 0.5 ? $d / (2 - $max - $min) : $d / ($max + $min);
            switch ($max) {
                case $r:
                    $h = (($g - $b) / $d + ($g < $b ? 6 : 0)) / 6;
                    break;
                case $g:
                    $h = (($b - $r) / $d + 2) / 6;
                    break;
                case $b:
                    $h = (($r - $g) / $d + 4) / 6;
                    break;
            }
        }
        return ['h' => round($h * 360), 's' => round($s * 100), 'l' => round($l * 100)];
    }

    protected function hslToHex(float $h, float $s, float $l): string
    {
        $h /= 360;
        $s /= 100;
        $l /= 100;
        if ($s === 0.0) {
            $r = $g = $b = $l;
        } else {
            $q = $l < 0.5 ? $l * (1 + $s) : $l + $s - $l * $s;
            $p = 2 * $l - $q;
            $r = $this->hueToRgb($p, $q, $h + 1 / 3);
            $g = $this->hueToRgb($p, $q, $h);
            $b = $this->hueToRgb($p, $q, $h - 1 / 3);
        }
        return sprintf('#%02x%02x%02x', round($r * 255), round($g * 255), round($b * 255));
    }

    protected function hueToRgb(float $p, float $q, float $t): float
    {
        if ($t < 0) $t += 1;
        if ($t > 1) $t -= 1;
        if ($t < 1 / 6) return $p + ($q - $p) * 6 * $t;
        if ($t < 1 / 2) return $q;
        if ($t < 2 / 3) return $p + ($q - $p) * (2 / 3 - $t) * 6;
        return $p;
    }

    protected function generateSecondaryColor(string $hexColor): string
    {
        $hex = ltrim($hexColor, '#');
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        $hsl = $this->rgbToHsl($r, $g, $b);
        $newHue = ($hsl['h'] + 30) % 360;
        return $this->hslToHex($newHue, $hsl['s'], $hsl['l']);
    }
}
