<?php


namespace App\Models;

use App\Classes\Common;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Vinkla\Hashids\Facades\Hashids;

/**
 * BaseModel with automatic attribute appending, hiding, and casting
 *
 * Automatically appends the following to $appends:
 * 1. 'xid' - Always appended
 * 2. Image URL attributes from $images array (e.g., 'image_url', 'profile_image_url') - returns default image when null
 * 3. File URL attributes from $files array (e.g., 'bill_url', 'document_url') - returns null when field is empty
 * 4. Hashable getter attributes from $hashableGetterFunctions (e.g., 'x_company_id', 'x_role_id')
 * 5. Hashable getter attributes from $foreignKeys (auto-generated, e.g., 'x_company_id', 'x_role_id')
 *
 * Automatically hides the following fields:
 * - 'id' field (always hidden)
 * - Fields from $foreignKeys (e.g., 'company_id', 'role_id')
 * - Fields from $hashableGetterFunctions values (e.g., 'company_id', 'role_id')
 * - Merges with any fields defined in child model's $hidden array
 *
 * Automatically adds Hash cast to foreign keys:
 * - All fields in $foreignKeys get Hash::class . ':hash' cast
 *
 * Usage:
 *
 * // Define images in your model (for image files)
 * // Option 1: Simple array (uses table.png as default)
 * protected $images = ['image', 'profile_image', 'logo'];
 *
 * // Option 2: With custom default images (key => value format)
 * protected $images = [
 *     'login_image' => 'login_background.svg',  // Custom default
 *     'light_logo' => 'light.png',              // Custom default
 *     'dark_logo',                              // Uses table.png as default
 * ];
 *
 * // Define files in your model (for non-image files like PDFs, documents, etc.)
 * // Option 1: Simple array (uses table.png as default)
 * protected $files = ['bill', 'document', 'invoice'];
 *
 * // Option 2: With custom default files (key => value format)
 * protected $files = [
 *     'bill' => 'default_bill.pdf',  // Custom default
 *     'invoice' => 'default_invoice.pdf',  // Custom default
 *     'document',  // Uses table.png as default
 * ];
 *
 * // Option 1: Define foreign keys (simplest - automatic hashable getters & auto-hide)
 * protected $foreignKeys = ['company_id', 'role_id', 'category_id'];
 *
 * // Option 2: Define hashable getters manually (for custom behavior, also auto-hide)
 * protected $hashableGetterFunctions = [
 *     'getXCompanyIdAttribute' => 'company_id',
 *     'getXRoleIdAttribute' => 'role_id',
 * ];
 *
 * // Define folder path for images/files
 * // Option 1: All images/files in the same folder (simple string)
 * protected $folderPath = 'userImagePath';
 *
 * // Option 2: Different folders for different images/files (array)
 * protected $folderPath = [
 *     'profile_image' => 'userImagePath',
 *     'logo' => 'companyLogoPath',
 *     'bill' => 'billsPath',
 *     'document' => 'documentsPath',
 * ];
 *
 * // Alternative: Use $imageFileFolder (takes priority over $folderPath)
 * // Option 1: All images in the same folder (simple string)
 * protected $imageFileFolder = 'customImagePath';
 *
 * // Option 2: Different folders for different images (array)
 * protected $imageFileFolder = [
 *     'profile_image' => 'userImagePath',
 *     'logo' => 'companyLogoPath',
 * ];
 *
 * // NO need to manually add to $appends, $hidden, or $casts!
 * // Auto-appended: 'xid', 'image_url', 'profile_image_url', 'logo_url', 'bill_url', 'document_url', 'x_company_id', 'x_role_id', 'x_category_id'
 * // Auto-hidden: 'id', 'company_id', 'role_id', 'category_id' (plus any fields in child model's $hidden)
 * // Auto-casted: 'company_id' => Hash::class . ':hash', 'role_id' => Hash::class . ':hash', etc.
 *
 * // If no custom folder path is defined, the table name is used as the folder
 * // If no custom default image/file is defined, {table}.png is used as the default
 */
class BaseModel extends ApiMainModel
{
    /**
     * Initialize the model
     */
    public function __construct(array $attributes = [])
    {
        $this->initializeDynamicProperties();
        parent::__construct($attributes);
    }

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();
    }

    /**
     * Initialize all dynamic properties (casts, appends, hidden)
     */
    protected function initializeDynamicProperties()
    {
        $this->hideBaseFields();
        $this->addForeignKeysCasts();
        $this->appendDynamicAttributes();
        $this->hideOriginalForeignKeys();
    }

    /**
     * Always hide 'id' field and merge with child model's $hidden
     */
    protected function hideBaseFields()
    {
        // Always hide 'id'
        $this->hideIfNotExists('id');
    }

    /**
     * Automatically add foreign keys to $casts with Hash cast
     */
    protected function addForeignKeysCasts()
    {
        if (isset($this->foreignKeys) && is_array($this->foreignKeys)) {
            foreach ($this->foreignKeys as $foreignKey) {
                // Only add if not already in casts
                if (!isset($this->casts[$foreignKey])) {
                    $this->casts[$foreignKey] = \App\Casts\Hash::class . ':hash';
                }
            }
        }
    }

    /**
     * Automatically hide original foreign key fields
     * - Fields from $foreignKeys
     * - Fields from $hashableGetterFunctions values
     */
    protected function hideOriginalForeignKeys()
    {
        // Hide foreign keys from $foreignKeys array
        if (isset($this->foreignKeys) && is_array($this->foreignKeys)) {
            foreach ($this->foreignKeys as $foreignKey) {
                $this->hideIfNotExists($foreignKey);
            }
        }

        // Hide fields from $hashableGetterFunctions values
        if (isset($this->hashableGetterFunctions) && is_array($this->hashableGetterFunctions)) {
            foreach ($this->hashableGetterFunctions as $methodName => $fieldName) {
                $this->hideIfNotExists($fieldName);
            }
        }
    }

    /**
     * Hide attribute if it doesn't already exist in $hidden
     *
     * @param string $attribute
     * @return void
     */
    protected function hideIfNotExists($attribute)
    {
        if (!in_array($attribute, $this->hidden)) {
            $this->makeHidden($attribute);
        }
    }

    /**
     * Automatically append dynamic attributes to $appends
     * - xid (always)
     * - Image URL attributes from $images
     * - Hashable getter attributes from $hashableGetterFunctions
     * - Hashable getter attributes from $foreignKeys (auto-generated)
     */
    protected function appendDynamicAttributes()
    {
        // Always append 'xid'
        $this->appendIfNotExists('xid');

        // Append image URL attributes
        if (isset($this->images) && is_array($this->images)) {
            foreach ($this->images as $key => $value) {
                // Handle both formats: ['image'] and ['image' => 'default.png']
                $imageField = is_string($key) ? $key : $value;
                $urlAttribute = $imageField . '_url';
                $this->appendIfNotExists($urlAttribute);
            }
        }

        // Append file URL attributes
        if (isset($this->files) && is_array($this->files)) {
            foreach ($this->files as $key => $value) {
                // Handle both formats: ['file'] and ['file' => 'default.pdf']
                $fileField = is_string($key) ? $key : $value;
                $urlAttribute = $fileField . '_url';
                $this->appendIfNotExists($urlAttribute);
            }
        }

        // Append hashable getter function attributes (x_ prefixed)
        if (isset($this->hashableGetterFunctions) && is_array($this->hashableGetterFunctions)) {
            foreach ($this->hashableGetterFunctions as $methodName => $fieldName) {
                // Extract attribute name from method name (e.g., 'getXCompanyIdAttribute' -> 'x_company_id')
                if (preg_match('/^get(.+)Attribute$/', $methodName, $matches)) {
                    $attributeName = Str::snake($matches[1]);
                    $this->appendIfNotExists($attributeName);
                }
            }
        }

        // Append hashable getter attributes from $foreignKeys (auto-generated)
        if (isset($this->foreignKeys) && is_array($this->foreignKeys)) {
            foreach ($this->foreignKeys as $foreignKey) {
                // Convert 'company_id' to 'x_company_id'
                $attributeName = 'x_' . $foreignKey;
                $this->appendIfNotExists($attributeName);
            }
        }
    }

    /**
     * Append attribute to $appends if it doesn't already exist
     *
     * @param string $attribute
     * @return void
     */
    protected function appendIfNotExists($attribute)
    {
        // Check if already exists in both the original appends and currently appended attributes
        $existsInOriginal = in_array($attribute, $this->appends);
        $existsInMutated = isset($this->mutatorCache[$attribute]);

        // Only append if not already present
        if (!$existsInOriginal && !$existsInMutated) {
            $this->append($attribute);
        }
    }

    /**
     * Override getAttribute to handle dynamic image/file URL attributes
     *
     * @param string $key
     * @return mixed
     */
    public function getAttribute($key)
    {
        // Check if this is a dynamic image URL attribute
        if (isset($this->images) && is_array($this->images)) {
            foreach ($this->images as $imageKey => $imageValue) {
                $imageField = is_string($imageKey) ? $imageKey : $imageValue;
                $urlAttribute = $imageField . '_url';

                if ($key === $urlAttribute) {
                    return $this->generateImageUrl($imageField);
                }
            }
        }

        // Check if this is a dynamic file URL attribute
        if (isset($this->files) && is_array($this->files)) {
            foreach ($this->files as $fileKey => $fileValue) {
                $fileField = is_string($fileKey) ? $fileKey : $fileValue;
                $urlAttribute = $fileField . '_url';

                if ($key === $urlAttribute) {
                    return $this->generateFileUrl($fileField);
                }
            }
        }

        // Check if this is a dynamic hashable getter (x_*_id)
        if (isset($this->foreignKeys) && is_array($this->foreignKeys)) {
            foreach ($this->foreignKeys as $foreignKey) {
                $xAttribute = 'x_' . $foreignKey;

                if ($key === $xAttribute) {
                    $value = parent::getAttribute($foreignKey);
                    return $value ? Hashids::encode($value) : null;
                }
            }
        }

        // Check hashableGetterFunctions
        if (isset($this->hashableGetterFunctions) && is_array($this->hashableGetterFunctions)) {
            foreach ($this->hashableGetterFunctions as $methodName => $fieldName) {
                // Extract attribute name from method: 'getXCompanyIdAttribute' -> 'x_company_id'
                $attrName = Str::snake(str_replace(['get', 'Attribute'], '', $methodName));

                if ($key === $attrName) {
                    $value = parent::getAttribute($fieldName);
                    return $value ? Hashids::encode($value) : null;
                }
            }
        }

        return parent::getAttribute($key);
    }

    function __call($method, $arguments)
    {
        if (isset($this->hashableGetterFunctions) && isset($this->hashableGetterFunctions[$method])) {

            $value = $this->{$this->hashableGetterFunctions[$method]};

            if ($value) {
                $value = Hashids::encode($value);
            }

            return $value;
        }

        if (isset($this->hashableGetterArrayFunctions) && isset($this->hashableGetterArrayFunctions[$method])) {

            $value = $this->{$this->hashableGetterArrayFunctions[$method]};

            if (count($value) > 0) {
                $valueArray = [];

                foreach ($value as $productId) {
                    $valueArray[] = Hashids::encode($productId);
                }

                $value = $valueArray;
            }

            return $value;
        }

        // Handle dynamic hashable getters from $foreignKeys
        if (isset($this->foreignKeys) && is_array($this->foreignKeys)) {
            foreach ($this->foreignKeys as $foreignKey) {
                // Build method name: 'company_id' -> 'getXCompanyIdAttribute'
                $expectedMethod = 'get' . Str::studly('x_' . $foreignKey) . 'Attribute';

                if ($method === $expectedMethod) {
                    $value = $this->{$foreignKey};

                    if ($value) {
                        $value = Hashids::encode($value);
                    }

                    return $value;
                }
            }
        }

        // Handle dynamic image URL getters based on $images property
        if (isset($this->images) && is_array($this->images)) {
            foreach ($this->images as $key => $value) {
                // Handle both formats: ['image'] and ['image' => 'default.png']
                $imageField = is_string($key) ? $key : $value;
                $methodName = 'get' . Str::studly($imageField) . 'UrlAttribute';

                if ($method === $methodName) {
                    return $this->generateImageUrl($imageField);
                }
            }
        }

        // Handle dynamic file URL getters based on $files property
        if (isset($this->files) && is_array($this->files)) {
            foreach ($this->files as $key => $value) {
                // Handle both formats: ['file'] and ['file' => 'default.pdf']
                $fileField = is_string($key) ? $key : $value;
                $methodName = 'get' . Str::studly($fileField) . 'UrlAttribute';

                if ($method === $methodName) {
                    return $this->generateFileUrl($fileField);
                }
            }
        }

        return parent::__call($method, $arguments);
    }

    /**
     * Generate image URL for the given field
     *
     * @param string $fieldName The name of the image field (e.g., 'image', 'profile_image')
     * @return string
     */
    protected function generateImageUrl($fieldName)
    {
        $fieldValue = $this->{$fieldName};

        // If no value, return default image
        if (empty($fieldValue)) {
            // Check if custom default image is defined in $images array
            $defaultImage = $this->getDefaultImageForField($fieldName);
            return asset("images/{$defaultImage}");
        }

        // Get folder path from table name or custom folder path
        $folderPath = $this->getImageFolderPath($fieldName);

        return Common::getFileUrl($folderPath, $fieldValue);
    }

    /**
     * Generate file URL for the given field
     * Unlike images, files return null when no value exists (no default file)
     *
     * @param string $fieldName The name of the file field (e.g., 'bill', 'document')
     * @return string|null
     */
    protected function generateFileUrl($fieldName)
    {
        $fieldValue = $this->{$fieldName};

        // If no value, return null (files don't have defaults)
        if (empty($fieldValue)) {
            return null;
        }

        // Get folder path from table name or custom folder path
        $folderPath = $this->getImageFolderPath($fieldName);

        return Common::getFileUrl($folderPath, $fieldValue);
    }

    /**
     * Get the default image filename for a field
     *
     * @param string $fieldName
     * @return string
     */
    protected function getDefaultImageForField($fieldName)
    {
        // Check if $images has a custom default for this field
        if (isset($this->images) && is_array($this->images)) {
            // Check if the field has a custom default (key => value format)
            if (isset($this->images[$fieldName])) {
                return $this->images[$fieldName];
            }
        }

        // Default to table name
        return "default.png";
        // return "{$this->table}.png";
    }

    /**
     * Get the folder path for storing images
     * Override this method in child models if you need custom folder paths
     *
     * Checks in order:
     * 1. $imageFileFolder - if string, returns that string directly
     *                     - if array, returns array[fieldName] value directly
     * 2. $folderPath - if string, returns Common::getFolderPath(string)
     *                - if array, returns Common::getFolderPath(array[fieldName])
     * 3. Falls back to table name
     *
     * @param string $fieldName
     * @return string
     */
    protected function getImageFolderPath($fieldName)
    {
        // Check imageFileFolder first (higher priority)
        // Returns the folder path directly without calling Common::getFolderPath
        if (isset($this->imageFileFolder)) {
            // If imageFileFolder is a string, return that folder path directly
            if (is_string($this->imageFileFolder)) {
                return $this->imageFileFolder;
            }

            // If imageFileFolder is an array, check for field-specific mapping and return directly
            if (is_array($this->imageFileFolder) && isset($this->imageFileFolder[$fieldName])) {
                return $this->imageFileFolder[$fieldName];
            }
        }

        // If folderPath is a string, all images go to the same folder
        if (isset($this->folderPath) && is_string($this->folderPath)) {
            return Common::getFolderPath($this->folderPath);
        }

        // If folderPath is an array, check for field-specific mapping
        if (isset($this->folderPath) && is_array($this->folderPath) && isset($this->folderPath[$fieldName])) {
            $folderKey = $this->folderPath[$fieldName];
            return Common::getFolderPath($folderKey);
        }

        // Default to table name
        return $this->table;
    }

    public function getXIDAttribute()
    {
        return Hashids::encode($this->id);
    }

    public function getFinalResult()
    {
        return Hashids::encode($this->id);
    }

    public function allPermissionArray($permissionName, $selectionType = 'fields')
    {
        $permssionArray = [];

        return $permssionArray && isset($permssionArray[$permissionName]) && isset($permssionArray[$permissionName][$selectionType]) ? $permssionArray[$permissionName][$selectionType] : null;
    }

    public function isRelationAllowedInQuery($fieldRelation)
    {
        $includeRelation = true;
        $searchRelatoinField = strpos($fieldRelation, ':') !== false ? explode(':', $fieldRelation)[0] : $fieldRelation;

        if (isset($this->permissions) && count($this->permissions) > 0) {
            $user = user();

            foreach ($this->permissions as $permission) {
                $relationNames = $this->allPermissionArray($permission, 'relations');

                if ($relationNames != null && in_array($searchRelatoinField, $relationNames) && !$this->isUserHavePermission($permission)) {
                    $includeRelation = false;
                    break;
                }
            }
        }

        return $includeRelation;
    }

    public function isFieldAllowedInQuery($fieldName)
    {
        $includeField = true;

        if (isset($this->permissions) && count($this->permissions) > 0) {
            $user = user();

            foreach ($this->permissions as $permission) {
                $allFields = $this->allPermissionArray($permission);

                if ($allFields != null && in_array($fieldName, $allFields) && !$this->isUserHavePermission($permission)) {
                    $includeField = false;
                    break;
                }
            }
        }

        return $includeField;
    }

    public function toArray()
    {
        $attributes = parent::toArray();

        if (isset($this->permissions) && count($this->permissions) > 0) {


            foreach ($this->permissions as $permission) {
                $permssionColumns = $this->allPermissionArray($permission);

                if (!$this->isUserHavePermission($permission) && $permssionColumns != null) {
                    $attributes = Arr::except($attributes, $permssionColumns);
                }
            }
        }

        return $attributes;
    }

    public function isUserHavePermission($permission)
    {
        $user = user();
        $userPermissions = $user && $user->role && $user->role->perms ? $user->role->perms->pluck('name')->toArray() : [];

        if ($user && $user->role && $user->role->name == 'admin') {
            return true;
        } else if (in_array($permission, $userPermissions)) {
            return true;
        }

        return false;
    }
}
