<?php

namespace App\Models;

use App\Models\BaseModel;
use App\Scopes\CompanyScope;
use App\Casts\Hash;
use App\Classes\Common;

class Website extends BaseModel
{
    protected $table = 'websites';

    protected $default = ['xid', 'name', 'slug', 'description', 'landing_url', 'is_favourite', 'is_primary', 'theme_name', 'theme_primary_color', 'theme_secondary_color', 'theme_mode', 'sections_count', 'use_same_header_across_all_pages', 'use_same_footer_across_all_pages', 'websiteCategory{id,xid,name}'];

    protected $guarded = ['id', 'created_at', 'updated_at'];

    protected $hidden = ['id', 'company_id', 'website_category_id'];

    protected $appends = ['xid', 'x_company_id', 'x_website_category_id', 'landing_url', 'sections_count'];

    protected $filterable = ['name', 'theme_name', 'theme_mode', 'is_favourite', 'is_primary', 'website_category_id'];

    protected $foreignKeys = ['company_id', 'website_category_id'];

    protected $hashableGetterFunctions = [
        'getXCompanyIdAttribute' => 'company_id',
        'getXWebsiteCategoryIdAttribute' => 'website_category_id',
    ];

    protected $casts = [
        'is_favourite' => 'boolean',
        'is_primary' => 'boolean',
        'use_same_header_across_all_pages' => 'boolean',
        'use_same_footer_across_all_pages' => 'boolean',
    ];

    protected static function boot()
    {
        parent::boot();

        static::addGlobalScope(new CompanyScope);
    }

    /**
     * Get all front languages for this website.
     */
    public function websiteLanguages()
    {
        return $this->hasMany(WebsiteLanguage::class, 'website_id');
    }

    /**
     * Get the default front language for this website.
     */
    public function defaultWebsiteLanguage()
    {
        return $this->hasOne(WebsiteLanguage::class, 'website_id')->where('is_default', true);
    }

    /**
     * Get the website category for this website.
     */
    public function websiteCategory()
    {
        return $this->belongsTo(WebsiteCategory::class, 'website_category_id');
    }

    /**
     * Get all website settings for this website.
     */
    public function websiteSettings()
    {
        return $this->hasMany(WebsiteSetting::class, 'website_id');
    }

    /**
     * Get all front page settings for this website.
     */
    public function websiteBuilder()
    {
        return $this->hasMany(WebsiteBuilder::class, 'website_id');
    }

    /**
     * Get all website legal pages for this website.
     */
    public function websiteLegalPages()
    {
        return $this->hasMany(WebsiteLegalPage::class, 'website_id');
    }

    /**
     * Get theme configuration as array.
     */
    public function getThemeConfigAttribute(): array
    {
        return [
            'name' => $this->theme_name,
            'primary_color' => $this->theme_primary_color,
            'secondary_color' => $this->theme_secondary_color,
            'mode' => $this->theme_mode,
        ];
    }

    /**
     * Get the landing page URL for this website.
     * Uses the global helper landing_home_url() which handles:
     * - Primary website in non-saas mode: returns root URL (/)
     * - Otherwise: returns website-specific URL with prefix
     */
    public function getLandingUrlAttribute(): ?string
    {
        if (!$this->slug) {
            return null;
        }

        return landing_home_url($this);
    }

    /**
     * Get the count of front page sections for this website.
     */
    public function getSectionsCountAttribute(): int
    {
        return $this->websiteBuilder()->count();
    }
}
