<?php

namespace App\Models;

use App\Classes\Common;
use App\Models\BaseModel;
use App\Scopes\CompanyScope;

class WebsiteBuilder extends BaseModel
{
    protected $table = 'website_builder';

    protected $default = ['id', 'xid', 'website_page_id', 'website_language_id', 'image', 'image_url', 'section_key', 'section_layout', 'title', 'subtitle', 'content', 'content_with_urls', 'content_schema', 'sort_order', 'is_white_background', 'padding_top', 'padding_bottom'];

    protected $guarded = ['id', 'created_at', 'updated_at'];

    protected $hidden = [];

    protected $filterable = ['section_key', 'section_layout', 'title', 'website_page_id'];

    protected $images = ['image'];

    protected $files = [];

    protected $foreignKeys = ['company_id', 'website_id', 'website_page_id', 'website_language_id'];

    protected $appends = ['content_with_urls'];

    public $imageFileFolder = "website_builder";

    protected $casts = [
        'content' => 'array',
        'content_schema' => 'array',
        'sample_data' => 'array',
        'is_white_background' => 'boolean',
        'padding_top' => 'integer',
        'padding_bottom' => 'integer',
    ];

    /**
     * Get content with dynamically generated image URLs
     */
    public function getContentWithUrlsAttribute()
    {
        $content = $this->content;
        if (!is_array($content)) {
            return $content;
        }

        // Merge with sample_data (sample_data takes precedence for non-null values)
        // This merges intelligently - for repeater fields, it merges image URLs into each item
        $sampleData = $this->sample_data;
        if (is_array($sampleData)) {
            $content = $this->mergeContentWithSampleData($content, $sampleData);
        }

        // Use the imageFileFolder property directly (website_builder)
        $folderPath = $this->imageFileFolder;

        // Process content and add URLs for image fields
        return $this->processContentUrls($content, $folderPath);
    }

    /**
     * Intelligently merge sample_data into content
     * For repeater fields, merges image URLs into each item rather than replacing the array
     *
     * @param array $content The original content array
     * @param array $sampleData The sample data with image URLs
     * @return array Merged content
     */
    protected function mergeContentWithSampleData(array $content, array $sampleData): array
    {
        foreach ($sampleData as $key => $value) {
            if ($value === null) {
                continue;
            }

            // If content has this key as an indexed array (repeater items)
            // and sample_data has it as an associative array (field => [urls])
            // then merge the URLs into each item
            if (isset($content[$key]) && is_array($content[$key]) && array_is_list($content[$key])
                && is_array($value) && !array_is_list($value)) {
                // This is a repeater field - merge image URLs into each item
                // sample_data structure: ['avatar' => ['url1', 'url2', ...]]
                // content structure: [['name' => '...', 'avatar' => null], ...]
                foreach ($value as $imageFieldName => $imageUrls) {
                    if (is_array($imageUrls) && array_is_list($imageUrls)) {
                        // Merge each URL into the corresponding item
                        foreach ($imageUrls as $index => $url) {
                            if (isset($content[$key][$index]) && is_array($content[$key][$index])) {
                                // Add as _url field for the image
                                $content[$key][$index][$imageFieldName . '_url'] = $url;
                            }
                        }
                    }
                }
            } else {
                // For non-repeater fields, simple replacement
                $content[$key] = $value;
            }
        }

        return $content;
    }

    /**
     * Recursively process content to add image URLs
     */
    protected function processContentUrls($data, $folderPath)
    {
        if (!is_array($data)) {
            return $data;
        }

        $result = [];
        foreach ($data as $key => $value) {
            $result[$key] = $value;

            // If it's an array (could be repeater items), process recursively
            if (is_array($value)) {
                // Check if it's a sequential array (repeater items)
                if (array_is_list($value)) {
                    $result[$key] = array_map(function ($item) use ($folderPath) {
                        return $this->processContentUrls($item, $folderPath);
                    }, $value);
                } else {
                    $result[$key] = $this->processContentUrls($value, $folderPath);
                }
            }
            // If the key doesn't end with _url and has a non-null string value that looks like a filename
            elseif (is_string($value) && !empty($value) && !str_ends_with($key, '_url')) {
                // Check if this looks like a stored image filename (has extension)
                if (preg_match('/\.(jpg|jpeg|png|gif|webp|svg|avif|bmp|ico|tiff|tif)$/i', $value)) {
                    $result[$key . '_url'] = Common::getFileUrl($folderPath, $value);
                }
            }
        }

        return $result;
    }

    protected static function boot()
    {
        parent::boot();

        static::addGlobalScope(new CompanyScope);
    }

    /**
     * Get the website that owns this setting.
     */
    public function website()
    {
        return $this->belongsTo(Website::class, 'website_id');
    }

    /**
     * Get the language for this setting.
     */
    public function websiteLanguage()
    {
        return $this->belongsTo(\App\Models\WebsiteLanguage::class, 'website_language_id');
    }

    /**
     * Get the website page that owns this setting.
     */
    public function websitePage()
    {
        return $this->belongsTo(WebsitePage::class, 'website_page_id');
    }

    /**
     * Get setting by section key
     */
    public static function getBySection(string $sectionKey)
    {
        return static::where('section_key', $sectionKey)->first();
    }

    /**
     * Get all settings
     */
    public static function getAllActive()
    {
        return static::all();
    }

    /**
     * Get all settings as key-value pairs
     */
    public static function getAllAsKeyValue()
    {
        return static::all()->keyBy('section_key');
    }

    /**
     * Get all settings for a specific language and website.
     * If no settings exist for the language/website, copy from default (English) language.
     * @param int|null $langId Optional language ID to filter by
     * @param int|null $websiteId Optional website ID to filter by
     */
    public static function getSettingsForLang($langId = null, $websiteId = null)
    {
        $companyId = company()->id ?? null;

        // Get English language ID for fallback
        $enLang = \App\Models\WebsiteLanguage::where('key', 'en')->first();
        $enLangId = $enLang ? $enLang->id : null;

        // If no langId specified, use English as default
        if (!$langId) {
            $langId = $enLangId;
        }

        // Build query for settings
        $query = static::where('company_id', $companyId)
            ->where('website_language_id', $langId);

        // Filter by website_id if provided
        if ($websiteId) {
            $query->where('website_id', $websiteId);
        }

        $settings = $query->get();

        // If settings exist for this language/website, return them
        if ($settings->count() > 0) {
            return $settings;
        }

        // If no settings exist for this language/website, copy from English language
        // First try to find English settings for the same website
        $defaultQuery = static::where('company_id', $companyId)
            ->where('website_language_id', $enLangId);

        // Use the same website_id for fallback (not null)
        if ($websiteId) {
            $defaultQuery->where('website_id', $websiteId);
        }

        $defaultSettings = $defaultQuery->get();

        // If no English settings found for this website, try without website filter
        if ($defaultSettings->count() === 0 && $websiteId) {
            $defaultQuery = static::where('company_id', $companyId)
                ->where('website_language_id', $enLangId)
                ->whereNull('website_id');
            $defaultSettings = $defaultQuery->get();
        }

        // If still no settings, just return empty collection
        if ($defaultSettings->count() === 0) {
            return $settings;
        }

        // Copy default settings for the new language/website
        $newSettings = collect();
        // Validate website exists before setting website_id
        $validWebsiteId = ($websiteId && \App\Models\Website::where('id', $websiteId)->exists()) ? $websiteId : null;

        foreach ($defaultSettings as $setting) {
            $newSetting = $setting->replicate();
            $newSetting->website_language_id = $langId;
            if ($validWebsiteId) {
                $newSetting->website_id = $validWebsiteId;
            }
            $newSetting->save();
            $newSettings->push($newSetting);
        }

        return $newSettings;
    }

    /**
     * Get or create a setting for a specific section, language, website and website page.
     * @param string $sectionKey The section key
     * @param int|null $langId Optional language ID
     * @param int|null $websiteId Optional website ID
     * @param int|null $websitePageId Optional website page ID
     */
    public static function getOrCreateForLang(string $sectionKey, $langId = null, $websiteId = null, $websitePageId = null)
    {
        $companyId = company()->id ?? null;

        // Get English language ID for fallback
        $enLang = \App\Models\WebsiteLanguage::where('key', 'en')->first();
        $enLangId = $enLang ? $enLang->id : null;

        // If no langId specified, use English as default
        if (!$langId) {
            $langId = $enLangId;
        }

        // Try to find existing setting for this language, website and website page
        $query = static::where('company_id', $companyId)
            ->where('website_language_id', $langId)
            ->where('section_key', $sectionKey);

        if ($websiteId) {
            $query->where('website_id', $websiteId);
        } else {
            $query->whereNull('website_id');
        }

        if ($websitePageId) {
            $query->where('website_page_id', $websitePageId);
        } else {
            $query->whereNull('website_page_id');
        }

        $setting = $query->first();

        if ($setting) {
            return $setting;
        }

        // Try to copy from English language for the same website and website page
        $defaultQuery = static::where('company_id', $companyId)
            ->where('section_key', $sectionKey)
            ->where('website_language_id', $enLangId);

        if ($websiteId) {
            $defaultQuery->where('website_id', $websiteId);
        } else {
            $defaultQuery->whereNull('website_id');
        }

        if ($websitePageId) {
            $defaultQuery->where('website_page_id', $websitePageId);
        } else {
            $defaultQuery->whereNull('website_page_id');
        }

        $defaultSetting = $defaultQuery->first();

        // If no English settings found for this website/page, try without website_page filter
        if (!$defaultSetting && $websitePageId) {
            $fallbackQuery = static::where('company_id', $companyId)
                ->where('section_key', $sectionKey)
                ->where('website_language_id', $enLangId)
                ->whereNull('website_page_id');

            if ($websiteId) {
                $fallbackQuery->where('website_id', $websiteId);
            } else {
                $fallbackQuery->whereNull('website_id');
            }

            $defaultSetting = $fallbackQuery->first();
        }

        // If still no settings found, try without website filter
        if (!$defaultSetting && $websiteId) {
            $defaultSetting = static::where('company_id', $companyId)
                ->where('section_key', $sectionKey)
                ->where('website_language_id', $enLangId)
                ->whereNull('website_id')
                ->whereNull('website_page_id')
                ->first();
        }

        if ($defaultSetting) {
            $newSetting = $defaultSetting->replicate();
            $newSetting->website_language_id = $langId;
            // Only set website_id if website exists
            if ($websiteId && \App\Models\Website::where('id', $websiteId)->exists()) {
                $newSetting->website_id = $websiteId;
            }
            // Set website_page_id if provided
            if ($websitePageId) {
                $newSetting->website_page_id = $websitePageId;
            }
            $newSetting->save();
            return $newSetting;
        }

        return null;
    }
}
