<?php

namespace App\Models;

use App\Models\BaseModel;
use App\Scopes\CompanyScope;

class WebsiteSetting extends BaseModel
{
    protected $table = 'website_settings';

    protected $default = ['xid'];

    protected $guarded = ['id', 'created_at', 'updated_at'];

    protected $hidden = ['id', 'created_at', 'updated_at'];

    protected $appends = ['xid'];

    protected $filterable = ['module_name'];

    protected $foreignKeys = ['company_id', 'website_id', 'website_language_id'];

    protected $casts = [
        'is_global' => 'boolean',
        'status' => 'boolean',
        'show_social_icon' => 'boolean',
        'social_links' => 'array',
    ];

    protected $images = [
        'light_logo' => 'light.png',
        'dark_logo' => 'dark.png',
        'light_logo_small' => 'small_light.png',
        'dark_logo_small' => 'small_dark.png',
        'website_favicon' => 'small_dark.png',
        'seo_og_image' => 'light.png',
    ];

    /**
     * Header logo type options:
     * - logo_with_name: Small logo + website name text
     * - logo_only: Small logo only (no name)
     * - light_dark_with_name: Light/dark logo based on theme + website name
     * - light_dark_only: Light/dark logo only (no name)
     */
    const HEADER_LOGO_TYPES = [
        'logo_with_name' => 'Small Logo + Website Name',
        'logo_only' => 'Small Logo Only',
        'light_dark_with_name' => 'Full Logo (Light/Dark) + Website Name',
        'light_dark_only' => 'Full Logo (Light/Dark) Only',
    ];

    protected static function boot()
    {
        parent::boot();

        static::addGlobalScope(new CompanyScope);
    }

    /**
     * Get the company that owns this setting.
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the website that owns this setting.
     */
    public function website()
    {
        return $this->belongsTo(Website::class, 'website_id');
    }

    /**
     * Get the language for this setting.
     */
    public function websiteLanguage()
    {
        return $this->belongsTo(\App\Models\WebsiteLanguage::class, 'website_language_id');
    }

    /**
     * Scope to filter by module name.
     */
    public function scopeModule($query, string $moduleName)
    {
        return $query->where('module_name', $moduleName);
    }

    /**
     * Scope to get global settings.
     */
    public function scopeGlobal($query)
    {
        return $query->where('is_global', true);
    }

    /**
     * Scope to get active settings.
     */
    public function scopeActive($query)
    {
        return $query->where('status', true);
    }

    /**
     * Get SEO configuration as array.
     */
    public function getSeoConfigAttribute(): array
    {
        return [
            'title' => $this->seo_title,
            'description' => $this->seo_description,
            'keywords' => $this->seo_keywords,
            'author' => $this->seo_author,
            'robots' => $this->seo_robots,
            'og' => [
                'type' => $this->seo_og_type,
                'locale' => $this->seo_og_locale,
                'site_name' => $this->seo_og_site_name,
                'image' => $this->seo_og_image_url,
            ],
            'twitter' => [
                'card' => $this->seo_twitter_card,
                'site' => $this->seo_twitter_site,
                'creator' => $this->seo_twitter_creator,
            ],
        ];
    }

    /**
     * Get website info configuration as array.
     */
    public function getWebsiteConfigAttribute(): array
    {
        return [
            'name' => $this->website_name,
            'tagline' => $this->website_tagline,
            'description' => $this->website_description,
            'phone' => $this->website_phone,
            'email' => $this->website_email,
            'address' => $this->website_address,
            'hours' => $this->website_hours,
            'light_logo' => $this->light_logo_url,
            'dark_logo' => $this->dark_logo_url,
            'light_logo_small' => $this->light_logo_small_url,
            'dark_logo_small' => $this->dark_logo_small_url,
            'favicon' => $this->website_favicon_url,
            'header_logo_type' => $this->header_logo_type ?? 'logo_with_name',
        ];
    }


    /**
     * Get or create Laundry settings for current company.
     * @param int|null $langId Optional language ID to filter by
     * @param int|null $websiteId Optional website ID to filter by
     */
    public static function getSettings($langId = null, $websiteId = null)
    {
        $companyId = company()->id ?? null;

        $query = static::where(function ($q) use ($companyId) {
            $q->where('company_id', $companyId)
                ->orWhere('is_global', true);
        });

        // If langId is provided, filter by it
        if ($langId) {
            $query->where('website_language_id', $langId);
        }

        // If websiteId is provided, filter by it
        if ($websiteId) {
            $query->where('website_id', $websiteId);
        }

        $settings = $query->first();

        if (!$settings) {
            // Create default SVG logo if it doesn't exist
            static::createDefaultLogo();

            $defaultSettings = static::getDefaultSettings($companyId);

            // Override with specified langId if provided
            if ($langId) {
                $defaultSettings['website_language_id'] = $langId;
            }

            // Override with specified websiteId if provided (only if website exists)
            if ($websiteId) {
                $websiteExists = \App\Models\Website::where('id', $websiteId)->exists();
                if ($websiteExists) {
                    $defaultSettings['website_id'] = $websiteId;
                }
            }

            $settings = static::create($defaultSettings);
        }

        return $settings;
    }

    /**
     * Create default SVG logo in public/images folder if it doesn't exist.
     */
    protected static function createDefaultLogo()
    {
        $logoPath = public_path('images/logo.svg');

        if (!file_exists($logoPath)) {
            $svgContent = '<svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" fill="none" stroke="#2563eb" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16 2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143z"/></svg>';

            // Ensure directory exists
            $directory = dirname($logoPath);
            if (!is_dir($directory)) {
                mkdir($directory, 0755, true);
            }

            file_put_contents($logoPath, $svgContent);
        }
    }

    /**
     * Get default settings for Laundry module.
     */
    public static function getDefaultSettings($companyId = null): array
    {
        $enLang = \App\Models\WebsiteLanguage::where('key', 'en')->first();

        return [
            'company_id' => $companyId,
            'website_language_id' => $enLang ? $enLang->id : null,
            'module_name' => 'Laundry',
            'is_global' => false,
            'status' => true,

            // SEO Configuration
            'seo_title' => 'SparkleWash - Premium Laundry Services | Free Pickup & Delivery',
            'seo_description' => 'Professional laundry and dry cleaning services with free pickup & delivery. Eco-friendly products, 24-hour express service, and 99% customer satisfaction.',
            'seo_keywords' => 'laundry service, dry cleaning, wash and fold, pickup delivery laundry',
            'seo_author' => 'SparkleWash',
            'seo_robots' => 'index, follow',
            'seo_og_type' => 'website',
            'seo_og_locale' => 'en_US',
            'seo_og_site_name' => 'SparkleWash',
            'seo_og_image' => null,
            'seo_twitter_card' => 'summary_large_image',
            'seo_twitter_site' => '@sparklewash',
            'seo_twitter_creator' => '@sparklewash',

            // Website Info
            'website_name' => 'SparkleWash',
            'website_tagline' => 'Premium Laundry Services',
            'website_description' => 'We provide professional laundry and dry cleaning services with free pickup and delivery. Experience the difference with our eco-friendly cleaning solutions.',
            'website_phone' => '+1 (555) 123-4567',
            'website_email' => 'hello@sparklewash.com',
            'website_address' => '123 Clean Street, Washville, WV 12345',
            'website_hours' => 'Mon - Sat: 7:00 AM - 9:00 PM | Sun: 8:00 AM - 6:00 PM',
            'website_favicon' => null,

            // Logos
            'light_logo' => null,
            'dark_logo' => null,
            'light_logo_small' => null,
            'dark_logo_small' => null,
            'header_logo_type' => 'logo_with_name',

            // Social Links (using Iconify format: prefix:icon-name)
            'social_links' => [
                ['name' => 'Facebook', 'url' => '#', 'icon' => 'mdi:facebook'],
                ['name' => 'Instagram', 'url' => '#', 'icon' => 'mdi:instagram'],
                ['name' => 'Twitter', 'url' => '#', 'icon' => 'mdi:twitter'],
                ['name' => 'LinkedIn', 'url' => '#', 'icon' => 'mdi:linkedin'],
            ],

            // Show social icons globally (header, footer, contact)
            'show_social_icon' => true,
        ];
    }
}
