<?php

namespace App\Providers;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\ServiceProvider;

class StorageSettingsProvider extends ServiceProvider
{
    /**
     * Static cache for storage settings within request lifecycle
     */
    protected static $storageSettings = [];

    /**
     * Register services.
     *
     * @return void
     */
    public function register()
    {
        try {
            $appType = app_type();

            // Check static cache first to avoid multiple DB queries per request
            $cacheKey = 'storage_setting_' . $appType;

            if (!isset(static::$storageSettings[$cacheKey])) {
                $query = DB::table('settings')
                    ->where('setting_type', 'storage')
                    ->where('status', 1);

                if ($appType == 'saas') {
                    $query->where('is_global', 1);
                } else {
                    $query->where('is_global', 0);
                }

                static::$storageSettings[$cacheKey] = $query->first();
            }

            $storageSetting = static::$storageSettings[$cacheKey];
            // Only configure storage if settings exist
            if ($storageSetting) {
                switch ($storageSetting->name_key) {
                    case 'aws':
                        $credentials = json_decode($storageSetting->credentials, true);

                        // Validate credentials before configuring
                        if (!empty($credentials['key']) && !empty($credentials['secret']) && !empty($credentials['bucket'])) {
                            config([
                                'filesystems.default' => 's3',
                                'filesystems.cloud' => 's3',
                                'filesystems.disks.s3.driver' => 's3',
                                'filesystems.disks.s3.key' => $credentials['key'],
                                'filesystems.disks.s3.secret' => $credentials['secret'],
                                'filesystems.disks.s3.region' => $credentials['region'] ?? 'us-east-1',
                                'filesystems.disks.s3.bucket' => $credentials['bucket'],
                                'filesystems.disks.s3.url' => null,
                                'filesystems.disks.s3.endpoint' => null,
                                'filesystems.disks.s3.use_path_style_endpoint' => false,
                                'filesystems.disks.s3.throw' => true,
                            ]);
                        }
                        break;

                    default:
                        config([
                            'filesystems.default' => 'local',
                        ]);
                        // Unknown storage type, don't set any config
                        break;
                }
            }
        } catch (\Exception $e) {
            // Log error but don't set any storage config
            Log::error('Error configuring storage settings', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Bootstrap services.
     *
     * @return void
     */
    public function boot()
    {
        //
    }

    /**
     * Clear the storage settings cache
     * Call this method when storage settings are updated
     *
     * @return void
     */
    public static function clearCache()
    {
        // Clear static cache
        static::$storageSettings = [];
    }
}
