<?php

namespace App\View\Components;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\View\Component;

class Icon extends Component
{
    public string $icon;
    public string $class;
    public ?string $width;
    public ?string $height;

    /**
     * Create a new component instance.
     */
    public function __construct(
        string $icon,
        string $class = '',
        ?string $width = null,
        ?string $height = null
    ) {
        $this->icon = $icon;
        $this->class = $class;
        $this->width = $width;
        $this->height = $height;
    }

    /**
     * Get the SVG content for the icon.
     */
    public function getSvg(): string
    {
        if (empty($this->icon)) {
            return '';
        }

        // Parse icon name (format: prefix:icon-name)
        $parts = explode(':', $this->icon);
        if (count($parts) !== 2) {
            return '';
        }

        [$prefix, $name] = $parts;

        // Cache key for the icon
        $cacheKey = "iconify_{$prefix}_{$name}";

        // Try to get from cache first (cache for 1 week)
        return Cache::remember($cacheKey, 60 * 60 * 24 * 7, function () use ($prefix, $name) {
            return $this->fetchIconFromApi($prefix, $name);
        });
    }

    /**
     * Fetch icon SVG from Iconify API.
     */
    protected function fetchIconFromApi(string $prefix, string $name): string
    {
        try {
            $url = "https://api.iconify.design/{$prefix}/{$name}.svg";

            $response = Http::timeout(5)->get($url);

            if ($response->successful()) {
                $svg = $response->body();

                // Add class attribute if provided
                if (!empty($this->class)) {
                    $svg = preg_replace(
                        '/<svg([^>]*)>/',
                        '<svg$1 class="' . e($this->class) . '">',
                        $svg,
                        1
                    );
                }

                // Add width attribute if provided
                if (!empty($this->width)) {
                    $svg = preg_replace(
                        '/<svg([^>]*)>/',
                        '<svg$1 width="' . e($this->width) . '">',
                        $svg,
                        1
                    );
                }

                // Add height attribute if provided
                if (!empty($this->height)) {
                    $svg = preg_replace(
                        '/<svg([^>]*)>/',
                        '<svg$1 height="' . e($this->height) . '">',
                        $svg,
                        1
                    );
                }

                return $svg;
            }
        } catch (\Exception $e) {
            // Log error but don't fail
            \Log::warning("Failed to fetch icon: {$prefix}:{$name}", ['error' => $e->getMessage()]);
        }

        // Return empty SVG placeholder on failure
        return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"></svg>';
    }

    /**
     * Get the view / contents that represent the component.
     */
    public function render()
    {
        return view('components.icon');
    }
}
