<?php

use App\Models\WebsiteSetting;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('website_settings', function (Blueprint $table) {
            $table->id();
            $table->bigInteger('company_id')->unsigned()->nullable()->default(null);
            $table->foreign('company_id')->references('id')->on('companies')->onDelete('cascade')->onUpdate('cascade');
            $table->bigInteger('website_id')->unsigned()->nullable()->default(null);
            $table->foreign('website_id')->references('id')->on('websites')->onDelete('cascade')->onUpdate('cascade');
            $table->bigInteger('website_language_id')->unsigned()->nullable();
            $table->foreign('website_language_id')->references('id')->on('website_languages')->onDelete('set null')->onUpdate('cascade');
            $table->string('module_name')->nullable()->comment('Module name e.g., Laundry, Post, etc.');
            $table->boolean('is_global')->default(false);
            $table->boolean('status')->default(true);

            // ============================================================
            // SEO CONFIGURATION
            // ============================================================
            $table->string('seo_title')->nullable();
            $table->text('seo_description')->nullable();
            $table->string('seo_keywords')->nullable();
            $table->string('seo_author')->nullable();
            $table->string('seo_robots')->nullable()->default('index, follow');
            $table->string('seo_og_type')->nullable()->default('website');
            $table->string('seo_og_locale')->nullable()->default('en_US');
            $table->string('seo_og_site_name')->nullable();
            $table->string('seo_og_image')->nullable();
            $table->string('seo_twitter_card')->nullable()->default('summary_large_image');
            $table->string('seo_twitter_site')->nullable();
            $table->string('seo_twitter_creator')->nullable();

            // ============================================================
            // WEBSITE INFO
            // ============================================================
            $table->string('website_name')->nullable();
            $table->string('website_tagline')->nullable();
            $table->text('website_description')->nullable()->comment('Short description for footer');
            $table->string('website_phone')->nullable();
            $table->string('website_email')->nullable();
            $table->text('website_address')->nullable();
            $table->string('website_hours')->nullable();
            $table->string('website_favicon')->nullable();

            // ============================================================
            // LOGO CONFIGURATION
            // ============================================================
            $table->string('light_logo')->nullable()->comment('Logo for light mode');
            $table->string('dark_logo')->nullable()->comment('Logo for dark mode');
            $table->string('light_logo_small')->nullable()->comment('Small logo for light mode (collapsed sidebar)');
            $table->string('dark_logo_small')->nullable()->comment('Small logo for dark mode (collapsed sidebar)');
            $table->string('header_logo_type')->default('logo_with_name')->comment('Header/Footer logo display: logo_with_name, logo_only, light_dark_with_name, light_dark_only');

            // ============================================================
            // SOCIAL LINKS (JSON)
            // ============================================================
            $table->json('social_links')->nullable()->comment('Array of social links: [{name, url, icon}]');
            $table->boolean('show_social_icon')->default(true)->comment('Show social icons in header, footer, and contact sections');

            $table->timestamps();

            // Indexes
            $table->index('module_name');
            $table->index('website_id');
            $table->index(['company_id', 'website_id', 'website_language_id']);
        });

        // Create default website settings for each website and language
        $websites = DB::table('websites')->get();
        $languages = DB::table('website_languages')->get();

        foreach ($websites as $website) {
            $websiteName = $website->name;
            $websiteSlug = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $websiteName));
            $twitterHandle = '@' . $websiteSlug;
            $websiteEmail = 'hello@' . $websiteSlug . '.com';

            foreach ($languages as $lang) {
                $defaultSettings = WebsiteSetting::getDefaultSettings($website->company_id);
                $defaultSettings['website_id'] = $website->id;
                $defaultSettings['website_language_id'] = $lang->id;

                // Replace default values with actual website name
                $defaultSettings['website_name'] = $websiteName;
                $defaultSettings['website_tagline'] = $websiteName . ' Website';
                $defaultSettings['website_email'] = $websiteEmail;
                $defaultSettings['seo_title'] = $websiteName;
                $defaultSettings['seo_description'] = 'Welcome to ' . $websiteName . '. We provide quality services and products to our customers.';
                $defaultSettings['seo_keywords'] = strtolower($websiteName) . ', ' . $websiteSlug . ', website, services';
                $defaultSettings['seo_author'] = $websiteName;
                $defaultSettings['seo_og_site_name'] = $websiteName;
                $defaultSettings['seo_twitter_site'] = $twitterHandle;
                $defaultSettings['seo_twitter_creator'] = $twitterHandle;

                WebsiteSetting::create($defaultSettings);
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('website_settings');
    }
};
