<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('website_builder', function (Blueprint $table) {
            $table->id();
            $table->bigInteger('company_id')->unsigned()->nullable()->default(null);
            $table->foreign('company_id')->references('id')->on('companies')->onDelete('cascade')->onUpdate('cascade');
            $table->bigInteger('website_id')->unsigned()->nullable()->default(null);
            $table->foreign('website_id')->references('id')->on('websites')->onDelete('cascade')->onUpdate('cascade');
            $table->bigInteger('website_language_id')->unsigned()->nullable();
            $table->foreign('website_language_id')->references('id')->on('website_languages')->onDelete('set null')->onUpdate('cascade');
            $table->bigInteger('website_page_id')->unsigned()->nullable()->default(null);
            $table->foreign('website_page_id')->references('id')->on('website_pages')->onDelete('cascade')->onUpdate('cascade');
            $table->string('section_key', 100); // e.g., 'hero', 'features', 'services', 'process', 'pricing', 'testimonials', 'faq', 'cta', 'newsletter'
            $table->string('section_layout', 100)->nullable()->default(null); // Component layout to use for this section (e.g., 'hero-section1', 'features', 'services')
            $table->string('title')->nullable()->default(null);
            $table->text('subtitle')->nullable()->default(null);
            $table->json('content')->nullable(); // Flexible JSON storage for section-specific data (items, icons, buttons, images, etc.)
            $table->json('content_schema')->nullable(); // Schema defining field types for dynamic form generation
            $table->json('sample_data')->nullable()->default(null); // Sample data for seeder to populate content examples
            $table->integer('sort_order')->default(0); // Display order of sections
            $table->boolean('is_white_background')->default(false);
            $table->integer('padding_top')->unsigned()->nullable()->default(null); // Custom padding top in pixels (null = use section default)
            $table->integer('padding_bottom')->unsigned()->nullable()->default(null); // Custom padding bottom in pixels (null = use section default)
            $table->timestamps();

            // Indexes
            $table->index('website_id');
            $table->index('website_page_id');

            // Ensure unique section_key per company, website_page and language
            $table->unique(['company_id', 'website_page_id', 'website_language_id', 'section_key'], 'website_builder_unique');
        });

        // Create landing page entries for all existing websites
        $this->createLandingPagesForExistingBusinesses();
    }

    /**
     * Create landing page entries for all existing websites
     * This creates: website_pages and website_page_translations only
     * User will add website_builder sections manually
     */
    protected function createLandingPagesForExistingBusinesses(): void
    {
        $websites = DB::table('websites')->get();

        foreach ($websites as $website) {
            // Get all front languages for this company
            $frontLanguages = DB::table('website_languages')
                ->where('company_id', $website->company_id)
                ->get();

            // Create home page in website_pages
            $homePageId = DB::table('website_pages')->insertGetId([
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'page_key' => 'home',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Create translations for the home page for each front language
            foreach ($frontLanguages as $lang) {
                DB::table('website_page_translations')->insert([
                    'website_page_id' => $homePageId,
                    'website_language_id' => $lang->id,
                    'name' => 'Home',
                    'show_in_header' => true,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('website_builder');
    }
};
