<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('website_legal_pages', function (Blueprint $table) {
            $table->id();
            $table->bigInteger('company_id')->unsigned()->nullable()->default(null);
            $table->foreign('company_id')->references('id')->on('companies')->onDelete('cascade')->onUpdate('cascade');
            $table->bigInteger('website_id')->unsigned()->nullable()->default(null);
            $table->foreign('website_id')->references('id')->on('websites')->onDelete('cascade')->onUpdate('cascade');
            $table->bigInteger('website_language_id')->unsigned()->nullable();
            $table->foreign('website_language_id')->references('id')->on('website_languages')->onDelete('set null')->onUpdate('cascade');
            $table->string('title');
            $table->string('slug');
            $table->text('content');
            $table->string('meta_title')->nullable();
            $table->text('meta_description')->nullable();
            $table->string('meta_keywords')->nullable();
            $table->boolean('show_in_footer')->default(true);
            $table->integer('display_order')->default(0);
            $table->string('status', 20)->default('active');          // ["active","inactive"]
            $table->timestamps();

            // Indexes
            $table->index('website_id');
            $table->index('slug');

            // Unique slug per website and language
            $table->unique(['company_id', 'website_id', 'website_language_id', 'slug'], 'website_legal_pages_unique_slug');
        });

        // Seed default website legal pages for existing websites
        $this->seedWebsiteLegalPages();
    }

    /**
     * Seed website legal pages for existing websites with English language
     */
    private function seedWebsiteLegalPages(): void
    {
        // Get first company
        $company = DB::table('companies')->first();
        if (!$company) {
            return;
        }

        // Get first business
        $website = DB::table('websites')->where('company_id', $company->id)->first();
        if (!$website) {
            return;
        }

        // Get English language
        $enLang = DB::table('website_languages')->where('key', 'en')->first();
        $langId = $enLang?->id;

        $pages = $this->getWebsiteLegalPagesData();

        foreach ($pages as $page) {
            DB::table('website_legal_pages')->insert([
                'company_id' => $company->id,
                'website_id' => $website->id,
                'website_language_id' => $langId,
                'title' => $page['title'],
                'slug' => $page['slug'],
                'content' => $page['content'],
                'meta_title' => $page['meta_title'],
                'meta_description' => $page['meta_description'],
                'meta_keywords' => $page['meta_keywords'],
                'show_in_footer' => $page['show_in_footer'] ?? true,
                'display_order' => $page['display_order'],
                'status' => $page['status'],
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }

    /**
     * Get website legal pages data
     */
    private function getWebsiteLegalPagesData(): array
    {
        $lastUpdated = date('F d, Y');

        return [
            // Privacy Policy
            [
                'title' => 'Privacy Policy',
                'slug' => 'privacy-policy',
                'content' => '<div class="privacy-policy">
    <h1>Privacy Policy</h1>
    <p class="last-updated">Last updated: ' . $lastUpdated . '</p>

    <section>
        <h2>1. Information We Collect</h2>
        <p>We collect information that you provide directly to us when using our services, including:</p>
        <ul>
            <li>Personal identification information (name, email address, phone number)</li>
            <li>Delivery address and location information</li>
            <li>Payment information (processed securely through third-party payment processors)</li>
            <li>Order history and preferences</li>
            <li>Communication preferences</li>
        </ul>
    </section>

    <section>
        <h2>2. How We Use Your Information</h2>
        <p>We use the information we collect to:</p>
        <ul>
            <li>Process and fulfill your orders</li>
            <li>Schedule pickup and delivery services</li>
            <li>Send order confirmations and updates</li>
            <li>Process payments and prevent fraud</li>
            <li>Improve our services and customer experience</li>
            <li>Send promotional communications (with your consent)</li>
            <li>Respond to your inquiries and provide customer support</li>
        </ul>
    </section>

    <section>
        <h2>3. Information Sharing and Disclosure</h2>
        <p>We do not sell, trade, or rent your personal information to third parties. We may share your information only in the following circumstances:</p>
        <ul>
            <li><strong>Service Providers:</strong> We work with trusted third-party service providers who assist us in operating our business</li>
            <li><strong>Legal Requirements:</strong> When required by law or to protect our rights</li>
            <li><strong>Business Transfers:</strong> In connection with any merger, sale of company assets, or acquisition</li>
        </ul>
    </section>

    <section>
        <h2>4. Data Security</h2>
        <p>We implement appropriate technical and organizational security measures to protect your personal information against unauthorized access, alteration, disclosure, or destruction. However, no method of transmission over the Internet is 100% secure.</p>
    </section>

    <section>
        <h2>5. Your Rights and Choices</h2>
        <p>You have the right to:</p>
        <ul>
            <li>Access, update, or delete your personal information</li>
            <li>Opt-out of marketing communications</li>
            <li>Request a copy of your data</li>
            <li>Restrict or object to certain data processing</li>
        </ul>
        <p>To exercise these rights, please contact us.</p>
    </section>

    <section>
        <h2>6. Cookies and Tracking Technologies</h2>
        <p>We use cookies and similar tracking technologies to enhance your experience on our website. For more information, please see our <a href="cookie-policy">Cookie Policy</a>.</p>
    </section>

    <section>
        <h2>7. Children\'s Privacy</h2>
        <p>Our services are not intended for children under 13 years of age. We do not knowingly collect personal information from children under 13.</p>
    </section>

    <section>
        <h2>8. Changes to This Privacy Policy</h2>
        <p>We may update this Privacy Policy from time to time. We will notify you of any changes by posting the new Privacy Policy on this page and updating the "Last updated" date.</p>
    </section>

    <section>
        <h2>9. Contact Us</h2>
        <p>If you have any questions about this Privacy Policy, please contact us through our website.</p>
    </section>
</div>',
                'meta_title' => 'Privacy Policy',
                'meta_description' => 'Learn how we collect, use, and protect your personal information when you use our services. Read our comprehensive privacy policy.',
                'meta_keywords' => 'privacy policy, data protection, personal information, customer data security',
                'show_in_footer' => true,
                'display_order' => 1,
                'status' => 'active',
            ],

            // Terms of Service
            [
                'title' => 'Terms of Service',
                'slug' => 'terms-of-service',
                'content' => '<div class="terms-of-service">
    <h1>Terms of Service</h1>
    <p class="last-updated">Last updated: ' . $lastUpdated . '</p>

    <section>
        <h2>1. Acceptance of Terms</h2>
        <p>By accessing and using our services, you accept and agree to be bound by these Terms of Service. If you do not agree to these terms, please do not use our services.</p>
    </section>

    <section>
        <h2>2. Service Description</h2>
        <p>We provide professional services including:</p>
        <ul>
            <li>Various service offerings</li>
            <li>Pickup and delivery services</li>
            <li>Special care options</li>
        </ul>
        <p>Service availability may vary by location.</p>
    </section>

    <section>
        <h2>3. User Responsibilities</h2>
        <p>As a user of our services, you agree to:</p>
        <ul>
            <li>Provide accurate and complete information when placing orders</li>
            <li>Be available during scheduled pickup and delivery times</li>
            <li>Pay for services in accordance with our pricing</li>
        </ul>
    </section>

    <section>
        <h2>4. Pricing and Payment</h2>
        <p>All prices are subject to change without notice. Payment is due upon delivery unless otherwise arranged. We accept major credit cards, debit cards, and digital payment methods. Late payments may incur additional fees.</p>
    </section>

    <section>
        <h2>5. Pickup and Delivery</h2>
        <ul>
            <li>Pickup and delivery times are estimates and may vary due to traffic or other circumstances</li>
            <li>You must provide a safe and accessible location for pickup and delivery</li>
            <li>If you are not available during the scheduled time, additional fees may apply for rescheduling</li>
        </ul>
    </section>

    <section>
        <h2>6. Cancellation and Refund Policy</h2>
        <ul>
            <li>Orders can be cancelled up to 2 hours before scheduled pickup without charge</li>
            <li>Cancellations within 2 hours of pickup may incur a cancellation fee</li>
            <li>Refunds are issued at our discretion for service failures</li>
            <li>No refunds for services already rendered</li>
        </ul>
    </section>

    <section>
        <h2>7. Intellectual Property</h2>
        <p>All content on our website and app, including text, graphics, logos, and software, is the property of our company and protected by copyright laws. You may not reproduce, distribute, or create derivative works without our permission.</p>
    </section>

    <section>
        <h2>8. Privacy</h2>
        <p>Your use of our services is also governed by our <a href="privacy-policy">Privacy Policy</a>. Please review it to understand how we collect and use your information.</p>
    </section>

    <section>
        <h2>9. Limitation of Liability</h2>
        <p>To the maximum extent permitted by law, we shall not be liable for any indirect, incidental, special, consequential, or punitive damages resulting from your use of our services.</p>
    </section>

    <section>
        <h2>10. Modifications to Terms</h2>
        <p>We reserve the right to modify these Terms of Service at any time. Changes will be effective immediately upon posting. Your continued use of our services constitutes acceptance of the modified terms.</p>
    </section>

    <section>
        <h2>11. Governing Law</h2>
        <p>These Terms of Service shall be governed by and construed in accordance with the laws of the jurisdiction in which we operate, without regard to its conflict of law provisions.</p>
    </section>

    <section>
        <h2>12. Contact Information</h2>
        <p>For questions about these Terms of Service, please contact us through our website.</p>
    </section>
</div>',
                'meta_title' => 'Terms of Service',
                'meta_description' => 'Read our terms of service to understand the rules and regulations governing the use of our professional services.',
                'meta_keywords' => 'terms of service, terms and conditions, user agreement, service agreement',
                'show_in_footer' => true,
                'display_order' => 2,
                'status' => 'active',
            ],

            // Cookie Policy
            [
                'title' => 'Cookie Policy',
                'slug' => 'cookie-policy',
                'content' => '<div class="cookie-policy">
    <h1>Cookie Policy</h1>
    <p class="last-updated">Last updated: ' . $lastUpdated . '</p>

    <section>
        <h2>1. What Are Cookies</h2>
        <p>Cookies are small text files that are stored on your device when you visit our website. They help us provide you with a better experience by remembering your preferences and understanding how you use our site.</p>
    </section>

    <section>
        <h2>2. How We Use Cookies</h2>
        <p>We use cookies for various purposes, including:</p>
        <ul>
            <li>Keeping you signed in to your account</li>
            <li>Remembering your preferences and settings</li>
            <li>Analyzing how you use our website to improve our services</li>
            <li>Personalizing content and advertisements</li>
            <li>Measuring the effectiveness of our marketing campaigns</li>
            <li>Preventing fraud and enhancing security</li>
        </ul>
    </section>

    <section>
        <h2>3. Types of Cookies We Use</h2>

        <h3>3.1 Essential Cookies</h3>
        <p>These cookies are necessary for the website to function properly. They enable core functionality such as security, network management, and accessibility. You cannot opt-out of these cookies.</p>

        <h3>3.2 Performance Cookies</h3>
        <p>These cookies collect information about how you use our website, such as which pages you visit most often. This data helps us improve the performance and user experience of our site.</p>

        <h3>3.3 Functionality Cookies</h3>
        <p>These cookies allow our website to remember choices you make (such as your username, language, or region) and provide enhanced, personalized features.</p>

        <h3>3.4 Targeting/Advertising Cookies</h3>
        <p>These cookies are used to deliver advertisements more relevant to you and your interests. They also help limit the number of times you see an advertisement and measure the effectiveness of advertising campaigns.</p>
    </section>

    <section>
        <h2>4. Third-Party Cookies</h2>
        <p>We may use third-party service providers who are allowed to place cookies on your device for the purposes described above. These providers include analytics and advertising services.</p>
    </section>

    <section>
        <h2>5. How Long Do Cookies Last</h2>
        <p>Cookies may be either "session" cookies or "persistent" cookies:</p>
        <ul>
            <li><strong>Session Cookies:</strong> These are temporary and are deleted when you close your browser</li>
            <li><strong>Persistent Cookies:</strong> These remain on your device until they expire or you delete them</li>
        </ul>
    </section>

    <section>
        <h2>6. Managing Cookies</h2>
        <p>You have the right to decide whether to accept or reject cookies. You can exercise your cookie preferences through your browser settings. Most web browsers allow you to:</p>
        <ul>
            <li>Block all cookies</li>
            <li>Only accept first-party cookies</li>
            <li>Delete cookies when you close your browser</li>
            <li>Notify you when a cookie is being set</li>
        </ul>
        <p>Note that blocking all cookies may impact your ability to use certain features of our website.</p>
    </section>

    <section>
        <h2>7. Do Not Track Signals</h2>
        <p>Some browsers have a "Do Not Track" feature that lets you tell websites you do not want to have your online activities tracked. We currently do not respond to Do Not Track signals.</p>
    </section>

    <section>
        <h2>8. Changes to This Cookie Policy</h2>
        <p>We may update this Cookie Policy from time to time to reflect changes in technology, legislation, or our business practices. We will notify you of any significant changes by posting the updated policy on this page.</p>
    </section>

    <section>
        <h2>9. More Information</h2>
        <p>For more information about how we handle your personal data, please see our <a href="privacy-policy">Privacy Policy</a>.</p>
        <p>If you have questions about our use of cookies, please contact us through our website.</p>
    </section>
</div>',
                'meta_title' => 'Cookie Policy',
                'meta_description' => 'Learn about how we use cookies on our website, what types of cookies we use, and how you can manage your cookie preferences.',
                'meta_keywords' => 'cookie policy, cookies, tracking, website cookies, browser cookies, privacy settings',
                'show_in_footer' => true,
                'display_order' => 3,
                'status' => 'active',
            ],
        ];
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('website_legal_pages');
    }
};
