<?php

namespace Database\Seeders;

use App\Classes\PermsSeed;
use App\Models\Company;
use App\Models\Permission;
use App\Models\Role;
use App\Scopes\CompanyScope;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class RolesTableSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */

    public function run()
    {
        Model::unguard();

        DB::table('permissions')->delete();
        DB::table('roles')->delete();

        DB::statement('ALTER TABLE `permissions` AUTO_INCREMENT = 1');
        DB::statement('ALTER TABLE roles AUTO_INCREMENT = 1');

        PermsSeed::seedMainPermissions();

        $company = Company::with(['currency' => function ($query) {
            return $query->withoutGlobalScope(CompanyScope::class);
        }, 'subscriptionPlan'])->where('is_global', 0)->first();

        $adminRole = new Role();
        $adminRole->company_id = $company->id;
        $adminRole->name = 'admin';
        $adminRole->display_name = 'Admin';
        $adminRole->description = 'Admin is allowed to manage everything of the app.';
        $adminRole->save();

        // Website Manager Role
        $websiteManagerRole = new Role();
        $websiteManagerRole->company_id = $company->id;
        $websiteManagerRole->name = 'website_manager';
        $websiteManagerRole->display_name = 'Website Manager';
        $websiteManagerRole->description = 'Website Manager can manage all website-related features including pages, content, and settings.';
        $websiteManagerRole->save();

        // Assign all website-related permissions to Website Manager
        $websiteManagerPermissions = Permission::whereIn('name', [
            'websites_view',
            'websites_create',
            'websites_edit',
            'websites_delete',
            'website_builder_view',
            'website_builder_create',
            'website_builder_edit',
            'website_builder_delete',
            'website_pages_view',
            'website_pages_create',
            'website_pages_edit',
            'website_pages_delete',
            'website_categories_view',
            'website_categories_create',
            'website_categories_edit',
            'website_categories_delete',
            'website_languages_view',
            'website_languages_create',
            'website_languages_edit',
            'website_languages_delete',
            'website_legal_pages_view',
            'website_legal_pages_create',
            'website_legal_pages_edit',
            'website_legal_pages_delete',
            'website_settings',
        ])->get();
        $websiteManagerRole->perms()->attach($websiteManagerPermissions);

        // Content Editor Role
        $contentEditorRole = new Role();
        $contentEditorRole->company_id = $company->id;
        $contentEditorRole->name = 'content_editor';
        $contentEditorRole->display_name = 'Content Editor';
        $contentEditorRole->description = 'Content Editor can edit website content and pages but has limited administrative access.';
        $contentEditorRole->save();

        // Assign limited permissions to Content Editor (view + edit only, no delete)
        $contentEditorPermissions = Permission::whereIn('name', [
            'websites_view',
            'website_pages_view',
            'website_pages_create',
            'website_pages_edit',
            'website_pages_delete',
            'website_categories_view',
            'website_languages_view',
            'website_legal_pages_view',
            'website_legal_pages_create',
            'website_legal_pages_edit',
            'website_legal_pages_delete',
            'website_settings',
        ])->get();
        $contentEditorRole->perms()->attach($contentEditorPermissions);
    }
}
