<?php

namespace Database\Seeders;

use App\Classes\ThemeSections;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class WebsiteBuilderSeeder extends Seeder
{
    protected $headerLayouts = ['header', 'header-split', 'header-topbar', 'header-pill'];
    protected $heroLayouts = ['hero', 'hero-split', 'hero-centered', 'hero-asymmetric', 'hero-fullscreen', 'hero-pattern', 'hero-card'];
    protected $featuresLayouts = ['features', 'features-grid', 'features-cards', 'features-hexagon', 'features-overlap', 'features-split'];
    protected $servicesLayouts = ['services', 'services-cards', 'services-bento', 'services-floating', 'services-gradient', 'services-horizontal', 'services-icons', 'services-minimal', 'services-orbit', 'services-timeline', 'services-zigzag'];
    protected $howItWorksLayouts = ['how_it_works', 'how-it-works-cards', 'how-it-works-compact', 'how-it-works-horizontal', 'how-it-works-minimal', 'how-it-works-numbered', 'how-it-works-split', 'how-it-works-stacked', 'how-it-works-timeline', 'how-it-works-zigzag'];
    protected $pricingLayouts = ['pricing', 'pricing-comparison', 'pricing-creative', 'pricing-gradient', 'pricing-minimal', 'pricing-modern'];
    protected $teamLayouts = ['team', 'team-carousel', 'team-grid-2', 'team-hexagon', 'team-minimal', 'team-overlap', 'team-split', 'team-wave'];
    protected $appDownloadLayouts = ['app_download', 'app_download_minimal', 'app_download_glassmorphism', 'app_download_centered'];

    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        Model::unguard();

        DB::table('website_builder')->delete();
        DB::statement('ALTER TABLE website_builder AUTO_INCREMENT = 1');

        $this->command->info("Starting website builder seeding...");

        $websites = DB::table('websites')
            ->leftJoin('website_categories', 'websites.website_category_id', '=', 'website_categories.id')
            ->select('websites.*', 'website_categories.name as category_name')
            ->get();

        $totalSections = 0;

        foreach ($websites as $websiteIndex => $website) {
            $languages = DB::table('website_languages')
                ->where('website_id', $website->id)
                ->where('enabled', true)
                ->get();

            if ($languages->isEmpty()) {
                $this->command->warn("No languages found for website ID: {$website->id} - Skipping");
                continue;
            }

            $pages = DB::table('website_pages')
                ->where('website_id', $website->id)
                ->get();

            if ($pages->isEmpty()) {
                $this->command->warn("No pages found for website ID: {$website->id} - Skipping");
                continue;
            }

            $this->command->info("Processing website: {$website->name} (Category: " . ($website->category_name ?? 'N/A') . ")");

            foreach ($languages as $language) {
                $this->command->info("  Language: {$language->name} ({$language->key})");

                // Get header layout for this website
                $headerLayout = $this->headerLayouts[$websiteIndex % count($this->headerLayouts)];

                // Create sections for each page (including header and footer for each page)
                foreach ($pages as $page) {
                    // Determine hero layout for this page (if it has a hero section)
                    $heroLayout = null;
                    if ($page->page_key === 'home') {
                        $heroLayout = $this->heroLayouts[$websiteIndex % count($this->heroLayouts)];
                    }

                    // Create header and footer for this page
                    $headerFooterSections = $this->createGlobalSections($website, $language, $page->id, $headerLayout, $heroLayout);
                    $totalSections += count($headerFooterSections);

                    // Create page-specific content sections (pass header layout for padding calculation)
                    $pageSections = $this->createPageSections($website, $language, $page, $websiteIndex, $headerLayout);
                    $totalSections += count($pageSections);

                    $this->command->info("    Page '{$page->page_key}': Created " . (count($headerFooterSections) + count($pageSections)) . " sections (header, footer, content)");
                }
            }
        }

        $this->command->info("\n" . str_repeat('=', 70));
        $this->command->info("✓ Website Builder seeding completed!");
        $this->command->info("Total sections created: {$totalSections}");
        $this->command->info(str_repeat('=', 70));
    }

    /**
     * Get section data from ThemeSections by section_key
     */
    protected function getSectionData($sectionKey)
    {
        foreach (ThemeSections::$sections as $section) {
            if ($section['section_key'] === $sectionKey) {
                return $section;
            }
        }
        return null;
    }

    /**
     * Create global header and footer sections
     */
    protected function createGlobalSections($website, $language, $pageId, $headerLayout, $heroLayout = null): array
    {
        $sections = [];

        // Get header section data from ThemeSections
        $headerData = $this->getSectionData('header');
        $headerContent = $headerData['default_content'] ?? [];

        // Determine header settings based on hero layout
        $lightHeroLayouts = ['hero', 'hero-card', 'hero-fullscreen', 'hero-with-form'];

        if ($heroLayout && in_array($heroLayout, $lightHeroLayouts)) {
            // Light hero layouts: header should be transparent with light text
            $headerContent['transparent'] = true;
            $headerContent['text_color_mode'] = 'light';
        } else {
            // Other hero layouts or no hero: header should not be transparent with dark text
            $headerContent['transparent'] = false;
            $headerContent['text_color_mode'] = 'dark';
        }

        // Add CTA buttons to header (with translations)
        $headerContent['cta_buttons'] = [
            [
                'text' => $this->translate('Pricing', $language->key),
                'url' => '/admin/login',
                'style' => 'outline'
            ],
            [
                'text' => $this->translate('Login', $language->key),
                'url' => '/admin/login',
                'style' => 'outline'
            ]
        ];

        // Translate "More" text in header
        $headerContent['more_text'] = $this->translate('More', $language->key);

        // Header Section
        $sections[] = [
            'company_id' => $website->company_id,
            'website_id' => $website->id,
            'website_language_id' => $language->id,
            'website_page_id' => $pageId,
            'section_key' => 'header',
            'section_layout' => $headerLayout,
            'title' => $headerData['default_title'] ?? null,
            'subtitle' => $headerData['default_subtitle'] ?? null,
            'content' => json_encode($headerContent),
            'content_schema' => json_encode($headerData['content_schema'] ?? []),
            'sample_data' => null,
            'sort_order' => $headerData['sort_order'] ?? 0,
            'is_white_background' => $headerData['is_white_background'] ?? false,
            'created_at' => now(),
            'updated_at' => now(),
        ];

        // Get footer section data from ThemeSections
        $footerData = $this->getSectionData('footer');

        // Footer Section
        $sections[] = [
            'company_id' => $website->company_id,
            'website_id' => $website->id,
            'website_language_id' => $language->id,
            'website_page_id' => $pageId,
            'section_key' => 'footer',
            'section_layout' => 'footer',
            'title' => $footerData['default_title'] ?? null,
            'subtitle' => $footerData['default_subtitle'] ?? null,
            'content' => json_encode($this->translateContent($footerData['default_content'] ?? [], $language->key)),
            'content_schema' => json_encode($footerData['content_schema'] ?? []),
            'sample_data' => null,
            'sort_order' => $footerData['sort_order'] ?? 9999,
            'is_white_background' => $footerData['is_white_background'] ?? false,
            'created_at' => now(),
            'updated_at' => now(),
        ];

        // Insert sections
        foreach ($sections as $section) {
            DB::table('website_builder')->insert($section);
        }

        return $sections;
    }

    /**
     * Create sections for a specific page
     */
    protected function createPageSections($website, $language, $page, $websiteIndex, $headerLayout = 'header'): array
    {
        $sections = [];

        switch ($page->page_key) {
            case 'home':
                $sections = $this->getHomepageSections($website, $language, $page, $websiteIndex, $headerLayout);
                break;
            case 'services':
                $sections = $this->getServicesSections($website, $language, $page, $websiteIndex);
                break;
            case 'pricing':
                $sections = $this->getPricingSections($website, $language, $page, $websiteIndex);
                break;
            case 'about':
            case 'about-us':
                $sections = $this->getAboutSections($website, $language, $page, $websiteIndex);
                break;
            case 'contact':
                $sections = $this->getContactSections($website, $language, $page, $websiteIndex);
                break;
            case 'how-it-works':
                $sections = $this->getHowItWorksSections($website, $language, $page, $websiteIndex);
                break;
            default:
                $sections = $this->getGenericSections($website, $language, $page, $websiteIndex);
                break;
        }

        // Insert all sections
        foreach ($sections as $section) {
            DB::table('website_builder')->insert($section);
        }

        return $sections;
    }

    /**
     * Get homepage sections with varied layouts
     */
    protected function getHomepageSections($website, $language, $page, $websiteIndex, $headerLayout = 'header'): array
    {
        $categoryName = $website->category_name ?? 'Business';

        // Rotate through layouts based on website index
        $heroLayout = $this->heroLayouts[$websiteIndex % count($this->heroLayouts)];
        $featuresLayout = $this->featuresLayouts[$websiteIndex % count($this->featuresLayouts)];
        $servicesLayout = $this->servicesLayouts[$websiteIndex % count($this->servicesLayouts)];

        // Calculate hero padding_top based on header layout
        // header or header-pill: 120px
        // header-split or header-topbar: 10px
        $heroPaddingTop = in_array($headerLayout, ['header', 'header-pill']) ? 120 : 10;

        // Get section data from ThemeSections
        $heroData = $this->getSectionData('hero');
        $featuresData = $this->getSectionData('features');
        $servicesData = $this->getSectionData('services');
        $pricingData = $this->getSectionData('pricing');
        $testimonialsData = $this->getSectionData('testimonials');
        $ctaData = $this->getSectionData('cta');
        $statsData = $this->getSectionData('stats');
        $galleryData = $this->getSectionData('gallery');
        $partnersData = $this->getSectionData('partners');
        $appDownloadData = $this->getSectionData('app_download');

        // Determine which optional sections to include based on category
        $includeAppDownload = in_array($categoryName, ['E-Commerce', 'Restaurant & Food', 'Fitness & Gym', 'Travel & Tourism']);
        $includeGallery = in_array($categoryName, ['Salon & Spa', 'Photography & Media', 'Event Management', 'Real Estate', 'Restaurant & Food']);
        $includePartners = in_array($categoryName, ['Consulting & Professional Services', 'Technology & IT', 'Healthcare & Medical']);

        $sections = [
            // Hero Section
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'hero',
                'section_layout' => $heroLayout,
                'title' => $this->getHeroTitle($categoryName, $language->key),
                'subtitle' => $this->getHeroSubtitle($categoryName, $language->key),
                'content' => json_encode(array_merge(
                    $this->translateContent($heroData['default_content'] ?? [], $language->key),
                    [
                        'description' => $this->getHeroDescription($categoryName, $language->key),
                        'hero_image' => null,
                        'buttons' => [
                            [
                                'text' => $this->translate('View Pricing', $language->key),
                                'url' => '/admin/login',
                                'icon' => 'lucide:tag',
                                'is_transparent' => false
                            ],
                            [
                                'text' => $this->translate('Login', $language->key),
                                'url' => '/admin/login',
                                'icon' => 'lucide:log-in',
                                'is_transparent' => true
                            ]
                        ]
                    ]
                )),
                'content_schema' => json_encode($heroData['content_schema'] ?? []),
                'sample_data' => json_encode([
                    'hero_image' => $this->getHeroImageUrl($categoryName),
                ]),
                'sort_order' => 1,
                'is_white_background' => $heroData['is_white_background'] ?? false,
                'padding_top' => $heroPaddingTop,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            // Features Section
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'features',
                'section_layout' => $featuresLayout,
                'title' => $this->translate('Why Choose Us', $language->key),
                'subtitle' => $this->translate('Features that set us apart', $language->key),
                'content' => json_encode(array_merge(
                    $this->translateContent($featuresData['default_content'] ?? [], $language->key),
                    [
                        'items' => $this->getFeatureItems($categoryName, $language->key),
                    ]
                )),
                'content_schema' => json_encode($featuresData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => 2,
                'is_white_background' => $featuresData['is_white_background'] ?? true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            // Stats Section (for all businesses)
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'stats',
                'section_layout' => 'stats',
                'title' => $this->translate('Our Impact', $language->key),
                'subtitle' => $this->translate('Numbers that speak for themselves', $language->key),
                'content' => json_encode($this->translateContent($statsData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($statsData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => 3,
                'is_white_background' => $statsData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            // Services Section
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'services',
                'section_layout' => $servicesLayout,
                'title' => $this->translate('Our Services', $language->key),
                'subtitle' => $this->translate('Comprehensive solutions tailored to your needs', $language->key),
                'content' => json_encode(array_merge(
                    $this->translateContent($servicesData['default_content'] ?? [], $language->key),
                    [
                        'items' => $this->getServiceItems($categoryName, $language->key),
                    ]
                )),
                'content_schema' => json_encode($servicesData['content_schema'] ?? []),
                'sample_data' => json_encode([
                    'service_images' => ['service1.jpg', 'service2.jpg', 'service3.jpg'],
                ]),
                'sort_order' => 4,
                'is_white_background' => $servicesData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        // Conditionally add Gallery Section (for visual businesses)
        if ($includeGallery) {
            $sections[] = [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'gallery',
                'section_layout' => 'gallery',
                'title' => $this->translate('Our Portfolio', $language->key),
                'subtitle' => $this->translate('Showcasing our finest work', $language->key),
                'content' => json_encode($this->translateContent($galleryData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($galleryData['content_schema'] ?? []),
                'sample_data' => json_encode([
                    'items' => $this->getGalleryImages($categoryName),
                ]),
                'sort_order' => 5,
                'is_white_background' => $galleryData['is_white_background'] ?? true,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Continue with Pricing Section
        $sections[] = [
            'company_id' => $website->company_id,
            'website_id' => $website->id,
            'website_language_id' => $language->id,
            'website_page_id' => $page->id,
            'section_key' => 'pricing',
            'section_layout' => $this->pricingLayouts[$websiteIndex % count($this->pricingLayouts)],
            'title' => $this->translate('Simple & Transparent Pricing', $language->key),
            'subtitle' => $this->translate('Choose the perfect plan for your needs', $language->key),
            'content' => json_encode($this->translateContent($pricingData['default_content'] ?? [], $language->key)),
            'content_schema' => json_encode($pricingData['content_schema'] ?? []),
            'sample_data' => null,
            'sort_order' => $includeGallery ? 6 : 5,
            'is_white_background' => $pricingData['is_white_background'] ?? false,
            'created_at' => now(),
            'updated_at' => now(),
        ];

        // Testimonials Section
        $sections[] = [
            'company_id' => $website->company_id,
            'website_id' => $website->id,
            'website_language_id' => $language->id,
            'website_page_id' => $page->id,
            'section_key' => 'testimonials',
            'section_layout' => 'testimonials',
            'title' => $this->translate('What Our Customers Say', $language->key),
            'subtitle' => $this->translate("Don't just take our word for it", $language->key),
            'content' => json_encode(array_merge(
                $testimonialsData['default_content'] ?? [],
                [
                    'items' => [
                        [
                            'name' => 'Sarah Johnson',
                            'role' => $this->translate('Business Owner', $language->key),
                            'avatar' => null,
                            'rating' => 5,
                            'text' => $this->translate('Exceptional service! They exceeded all my expectations and delivered outstanding results.', $language->key)
                        ],
                        [
                            'name' => 'Michael Chen',
                            'role' => $this->translate('Marketing Director', $language->key),
                            'avatar' => null,
                            'rating' => 5,
                            'text' => $this->translate('Professional, reliable, and high-quality work. Highly recommended for anyone looking for excellence.', $language->key)
                        ],
                        [
                            'name' => 'Emily Rodriguez',
                            'role' => $this->translate('Entrepreneur', $language->key),
                            'avatar' => null,
                            'rating' => 5,
                            'text' => $this->translate('The best decision I made was choosing this service. Amazing quality and customer support!', $language->key)
                        ],
                    ],
                ]
            )),
            'content_schema' => json_encode($testimonialsData['content_schema'] ?? []),
            'sample_data' => json_encode([
                'avatars' => ['avatar1.jpg', 'avatar2.jpg', 'avatar3.jpg'],
            ]),
            'sort_order' => $includeGallery ? 7 : 6,
            'is_white_background' => $testimonialsData['is_white_background'] ?? true,
            'created_at' => now(),
            'updated_at' => now(),
        ];

        // Conditionally add Partners Section (for enterprise businesses)
        if ($includePartners) {
            $sections[] = [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'partners',
                'section_layout' => 'partners',
                'title' => $this->translate('Trusted Partners', $language->key),
                'subtitle' => $this->translate('Working with industry leaders', $language->key),
                'content' => json_encode($this->translateContent($partnersData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($partnersData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => $includeGallery ? 8 : 7,
                'is_white_background' => $partnersData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Conditionally add App Download Section (for mobile app businesses)
        if ($includeAppDownload) {
            $sections[] = [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'app_download',
                'section_layout' => $this->appDownloadLayouts[$websiteIndex % count($this->appDownloadLayouts)],
                'title' => $this->translate('Download Our App', $language->key),
                'subtitle' => $this->translate('Get started in seconds', $language->key),
                'content' => json_encode($this->translateContent($appDownloadData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($appDownloadData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => ($includeGallery ? 8 : 7) + ($includePartners ? 1 : 0),
                'is_white_background' => $appDownloadData['is_white_background'] ?? true,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // CTA Section (always last)
        $lastSortOrder = ($includeGallery ? 8 : 7) + ($includePartners ? 1 : 0) + ($includeAppDownload ? 1 : 0);
        $sections[] = [
            'company_id' => $website->company_id,
            'website_id' => $website->id,
            'website_language_id' => $language->id,
            'website_page_id' => $page->id,
            'section_key' => 'cta',
            'section_layout' => 'cta',
            'title' => $this->translate('Ready to Get Started?', $language->key),
            'subtitle' => $this->translate('Join thousands of satisfied customers today', $language->key),
            'content' => json_encode($this->translateContent($ctaData['default_content'] ?? [], $language->key)),
            'content_schema' => json_encode($ctaData['content_schema'] ?? []),
            'sample_data' => json_encode([
                'background_image' => 'cta-background.jpg',
            ]),
            'sort_order' => $lastSortOrder + 1,
            'is_white_background' => $ctaData['is_white_background'] ?? false,
            'created_at' => now(),
            'updated_at' => now(),
        ];

        return $sections;
    }

    /**
     * Get services page sections
     */
    protected function getServicesSections($website, $language, $page, $websiteIndex): array
    {
        $categoryName = $website->category_name ?? 'Business';
        $servicesLayout = $this->servicesLayouts[$websiteIndex % count($this->servicesLayouts)];
        $howItWorksLayout = $this->howItWorksLayouts[$websiteIndex % count($this->howItWorksLayouts)];

        // Get section data from ThemeSections
        $servicesData = $this->getSectionData('services');
        $howItWorksData = $this->getSectionData('how_it_works');

        return [
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'services',
                'section_layout' => $servicesLayout,
                'title' => $servicesData['default_title'] ?? 'Our Professional Services',
                'subtitle' => $servicesData['default_subtitle'] ?? 'Comprehensive solutions designed to meet your unique needs',
                'content' => json_encode(array_merge(
                    $this->translateContent($servicesData['default_content'] ?? [], $language->key),
                    [
                        'items' => $this->getServiceItems($categoryName, $language->key),
                    ]
                )),
                'content_schema' => json_encode($servicesData['content_schema'] ?? []),
                'sample_data' => json_encode([
                    'service_images' => ['service1.jpg', 'service2.jpg', 'service3.jpg', 'service4.jpg'],
                ]),
                'sort_order' => 1,
                'is_white_background' => $servicesData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'how_it_works',
                'section_layout' => $howItWorksLayout,
                'title' => $howItWorksData['default_title'] ?? 'How It Works',
                'subtitle' => $howItWorksData['default_subtitle'] ?? 'Simple process, outstanding results',
                'content' => json_encode(array_merge(
                    $this->translateContent($howItWorksData['default_content'] ?? [], $language->key),
                    [
                        'items' => [
                            ['step' => '01', 'title' => $this->translate('Consultation', $language->key), 'description' => $this->translate('We discuss your needs and requirements', $language->key), 'icon' => 'lucide:phone'],
                            ['step' => '02', 'title' => $this->translate('Planning', $language->key), 'description' => $this->translate('We create a customized plan for you', $language->key), 'icon' => 'lucide:clipboard'],
                            ['step' => '03', 'title' => $this->translate('Execution', $language->key), 'description' => $this->translate('Our experts deliver quality service', $language->key), 'icon' => 'lucide:zap'],
                            ['step' => '04', 'title' => $this->translate('Delivery', $language->key), 'description' => $this->translate('Enjoy your completed project', $language->key), 'icon' => 'lucide:check']
                        ],
                    ]
                )),
                'content_schema' => json_encode($howItWorksData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => 2,
                'is_white_background' => $howItWorksData['is_white_background'] ?? true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];
    }

    /**
     * Get pricing page sections
     */
    protected function getPricingSections($website, $language, $page, $websiteIndex): array
    {
        $pricingLayout = $this->pricingLayouts[$websiteIndex % count($this->pricingLayouts)];

        // Get section data from ThemeSections
        $pricingData = $this->getSectionData('pricing');
        $faqData = $this->getSectionData('faq');

        return [
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'pricing',
                'section_layout' => $pricingLayout,
                'title' => $pricingData['default_title'] ?? 'Choose Your Perfect Plan',
                'subtitle' => $pricingData['default_subtitle'] ?? 'Flexible pricing designed to fit your needs',
                'content' => json_encode($this->translateContent($pricingData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($pricingData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => 1,
                'is_white_background' => $pricingData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'faq',
                'section_layout' => 'faq',
                'title' => $faqData['default_title'] ?? 'Frequently Asked Questions',
                'subtitle' => $faqData['default_subtitle'] ?? 'Find answers to common questions about pricing',
                'content' => json_encode($this->translateContent($faqData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($faqData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => 2,
                'is_white_background' => $faqData['is_white_background'] ?? true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];
    }

    /**
     * Get about page sections
     */
    protected function getAboutSections($website, $language, $page, $websiteIndex): array
    {
        $teamLayout = $this->teamLayouts[$websiteIndex % count($this->teamLayouts)];

        // Get section data from ThemeSections
        $aboutData = $this->getSectionData('about');
        $teamData = $this->getSectionData('team');

        return [
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'about',
                'section_layout' => 'about',
                'title' => $aboutData['default_title'] ?? 'About Our Company',
                'subtitle' => $aboutData['default_subtitle'] ?? 'A legacy of quality and trust',
                'content' => json_encode($this->translateContent($aboutData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($aboutData['content_schema'] ?? []),
                'sample_data' => json_encode([
                    'image' => 'about-company.jpg',
                ]),
                'sort_order' => 1,
                'is_white_background' => $aboutData['is_white_background'] ?? true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'team',
                'section_layout' => $teamLayout,
                'title' => $teamData['default_title'] ?? 'Meet Our Team',
                'subtitle' => $teamData['default_subtitle'] ?? 'The dedicated professionals behind our success',
                'content' => json_encode($this->translateContent($teamData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($teamData['content_schema'] ?? []),
                'sample_data' => json_encode([
                    'team_images' => ['team1.jpg', 'team2.jpg', 'team3.jpg'],
                ]),
                'sort_order' => 2,
                'is_white_background' => $teamData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];
    }

    /**
     * Get contact page sections
     */
    protected function getContactSections($website, $language, $page, $websiteIndex): array
    {
        // Get section data from ThemeSections
        $contactData = $this->getSectionData('contact');

        return [
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'contact',
                'section_layout' => 'contact',
                'title' => $contactData['default_title'] ?? 'Get In Touch',
                'subtitle' => $contactData['default_subtitle'] ?? 'We\'d love to hear from you',
                'content' => json_encode($this->translateContent($contactData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($contactData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => 1,
                'is_white_background' => $contactData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];
    }

    /**
     * Get how it works page sections
     */
    protected function getHowItWorksSections($website, $language, $page, $websiteIndex): array
    {
        $howItWorksLayout = $this->howItWorksLayouts[$websiteIndex % count($this->howItWorksLayouts)];

        // Get section data from ThemeSections
        $howItWorksData = $this->getSectionData('how_it_works');

        return [
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'how_it_works',
                'section_layout' => $howItWorksLayout,
                'title' => $howItWorksData['default_title'] ?? 'Simple 4-Step Process',
                'subtitle' => $howItWorksData['default_subtitle'] ?? 'How we deliver exceptional results',
                'content' => json_encode($this->translateContent($howItWorksData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($howItWorksData['content_schema'] ?? []),
                'sample_data' => null,
                'sort_order' => 1,
                'is_white_background' => $howItWorksData['is_white_background'] ?? true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];
    }

    /**
     * Get generic sections for other pages
     */
    protected function getGenericSections($website, $language, $page, $websiteIndex): array
    {
        // Get section data from ThemeSections
        $ctaData = $this->getSectionData('cta');

        return [
            [
                'company_id' => $website->company_id,
                'website_id' => $website->id,
                'website_language_id' => $language->id,
                'website_page_id' => $page->id,
                'section_key' => 'cta',
                'section_layout' => 'cta',
                'title' => $ctaData['default_title'] ?? 'Ready to Get Started?',
                'subtitle' => $ctaData['default_subtitle'] ?? 'Contact us today for a consultation',
                'content' => json_encode($this->translateContent($ctaData['default_content'] ?? [], $language->key)),
                'content_schema' => json_encode($ctaData['content_schema'] ?? []),
                'sample_data' => json_encode([
                    'background_image' => 'cta-background.jpg',
                ]),
                'sort_order' => 1,
                'is_white_background' => $ctaData['is_white_background'] ?? false,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];
    }

    /**
     * Recursively translate all text content in an array
     */
    protected function translateContent(mixed $content, string $langKey = 'en'): mixed
    {
        if (is_string($content)) {
            return $this->translate($content, $langKey);
        }

        if (is_array($content)) {
            $translated = [];
            foreach ($content as $key => $value) {
                $translated[$key] = $this->translateContent($value, $langKey);
            }
            return $translated;
        }

        return $content;
    }

    /**
     * Translate common strings - comprehensive translations for all section content
     */
    protected function translate(string $text, string $langKey = 'en'): string
    {
        $translations = [
            'en' => [
                // Buttons & CTAs
                'View Pricing' => 'View Pricing',
                'Learn More' => 'Learn More',
                'Get Started' => 'Get Started',
                'Contact Us' => 'Contact Us',
                'Schedule Pickup' => 'Schedule Pickup',
                'Book Now' => 'Book Now',
                'Call Us Now' => 'Call Us Now',
                'Download Now' => 'Download Now',

                // Section Titles
                'Why Choose Us' => 'Why Choose Us',
                'Features that set us apart' => 'Features that set us apart',
                'Our Impact' => 'Our Impact',
                'Numbers that speak for themselves' => 'Numbers that speak for themselves',
                'Our Services' => 'Our Services',
                'Comprehensive solutions tailored to your needs' => 'Comprehensive solutions tailored to your needs',
                'Simple & Transparent Pricing' => 'Simple & Transparent Pricing',
                'Choose the perfect plan for your needs' => 'Choose the perfect plan for your needs',
                'What Our Customers Say' => 'What Our Customers Say',
                "Don't just take our word for it" => "Don't just take our word for it",
                'Ready to Get Started?' => 'Ready to Get Started?',
                'Join thousands of satisfied customers today' => 'Join thousands of satisfied customers today',
                'Trusted Partners' => 'Trusted Partners',
                'Working with industry leaders' => 'Working with industry leaders',
                'Download Our App' => 'Download Our App',
                'Get started in seconds' => 'Get started in seconds',
                'Our Portfolio' => 'Our Portfolio',
                'Showcasing our finest work' => 'Showcasing our finest work',

                // Feature Items
                'Eco-Friendly' => 'Eco-Friendly',
                'Fast Service' => 'Fast Service',
                'Fully Insured' => 'Fully Insured',
                'Free Delivery' => 'Free Delivery',
                'Quality Service' => 'Quality Service',
                'Expert Team' => 'Expert Team',
                'Timely Delivery' => 'Timely Delivery',
                'Customer Care' => 'Customer Care',
                '100% biodegradable detergents for safe cleaning.' => '100% biodegradable detergents for safe cleaning.',
                'Quick turnaround with express options available.' => 'Quick turnaround with express options available.',
                'Your garments are protected and insured.' => 'Your garments are protected and insured.',
                'Complimentary pickup and delivery service.' => 'Complimentary pickup and delivery service.',
                'Exceptional quality in everything we do.' => 'Exceptional quality in everything we do.',
                'Experienced professionals at your service.' => 'Experienced professionals at your service.',
                'On-time, every time guarantee.' => 'On-time, every time guarantee.',
                'Dedicated to your satisfaction.' => 'Dedicated to your satisfaction.',

                // Service Items
                'Professional Service' => 'Professional Service',
                'High-quality service delivered by experts.' => 'High-quality service delivered by experts.',
                'Contact for pricing' => 'Contact for pricing',
                'Experienced professionals' => 'Experienced professionals',
                'Quality guaranteed' => 'Quality guaranteed',
                'Fast Turnaround' => 'Fast Turnaround',
                'Quick and efficient service delivery.' => 'Quick and efficient service delivery.',
                'Flexible pricing' => 'Flexible pricing',
                'Express options available' => 'Express options available',
                'On-time delivery' => 'On-time delivery',
                'Reliable & Trusted' => 'Reliable & Trusted',
                'Dependable service you can count on.' => 'Dependable service you can count on.',
                'Competitive rates' => 'Competitive rates',
                'Satisfaction guaranteed' => 'Satisfaction guaranteed',
                'Insured and protected' => 'Insured and protected',

                // Testimonials
                'Business Owner' => 'Business Owner',
                'Marketing Director' => 'Marketing Director',
                'Entrepreneur' => 'Entrepreneur',
                'Exceptional service! They exceeded all my expectations and delivered outstanding results.' => 'Exceptional service! They exceeded all my expectations and delivered outstanding results.',
                'Professional, reliable, and high-quality work. Highly recommended for anyone looking for excellence.' => 'Professional, reliable, and high-quality work. Highly recommended for anyone looking for excellence.',
                'The best decision I made was choosing this service. Amazing quality and customer support!' => 'The best decision I made was choosing this service. Amazing quality and customer support!',

                // Badges
                'Our Services' => 'Our Services',
                'Pricing Plans' => 'Pricing Plans',
                'How It Works' => 'How It Works',
                'Testimonials' => 'Testimonials',
                'FAQ' => 'FAQ',
                'Contact Us' => 'Contact Us',
                'Gallery' => 'Gallery',
                'Our Partners' => 'Our Partners',
                'Mobile App' => 'Mobile App',
                'Statistics' => 'Statistics',
                'About Us' => 'About Us',
                'Our Team' => 'Our Team',

                // Header/Footer
                'Pricing' => 'Pricing',
                'More' => 'More',
                'Login' => 'Login',

                // Restaurant & Food Features
                'Expert Chefs' => 'Expert Chefs',
                'Skilled culinary professionals.' => 'Skilled culinary professionals.',
                'Fresh Ingredients' => 'Fresh Ingredients',
                'Locally-sourced, organic produce.' => 'Locally-sourced, organic produce.',
                'Diverse Menu' => 'Diverse Menu',
                'Wide variety of delicious options.' => 'Wide variety of delicious options.',
                'Cozy Ambiance' => 'Cozy Ambiance',
                'Welcoming and comfortable atmosphere.' => 'Welcoming and comfortable atmosphere.',

                // Hero Section Content
                '#1 Rated Laundry Service in Town' => '#1 Rated Laundry Service in Town',
                'Experience premium laundry services with free pickup & delivery. We treat your clothes with care using eco-friendly products.' => 'Experience premium laundry services with free pickup & delivery. We treat your clothes with care using eco-friendly products.',
                'Free Pickup' => 'Free Pickup',
                '& Delivery' => '& Delivery',
                'Express' => 'Express',
                '24h Service' => '24h Service',
                'Happy Customers' => 'Happy Customers',
                'Orders Completed' => 'Orders Completed',
                'Satisfaction Rate' => 'Satisfaction Rate',
                'Express Delivery' => 'Express Delivery',

                // Pricing Section
                'Most Popular' => 'Most Popular',
                'per month' => 'per month',
                'Basic' => 'Basic',
                'Family' => 'Family',
                'Business' => 'Business',
                'Perfect for individuals' => 'Perfect for individuals',
                'Best for families' => 'Best for families',
                'For businesses & hotels' => 'For businesses & hotels',

                // How It Works
                'Consultation' => 'Consultation',
                'We discuss your needs and requirements' => 'We discuss your needs and requirements',
                'Planning' => 'Planning',
                'We create a customized plan for you' => 'We create a customized plan for you',
                'Execution' => 'Execution',
                'Our experts deliver quality service' => 'Our experts deliver quality service',
                'Delivery' => 'Delivery',
                'Enjoy your completed project' => 'Enjoy your completed project',

                // Stats values (numbers - usually not translated but kept for consistency)
                '15K+' => '15K+',
                '50K+' => '50K+',
                '99%' => '99%',
                '24h' => '24h',
                '15000+' => '15000+',
                '50000+' => '50000+',

                // Services Section
                'Wash & Fold' => 'Wash & Fold',
                'Professional washing, drying, and folding service for your everyday clothes.' => 'Professional washing, drying, and folding service for your everyday clothes.',
                'From $2.50/lb' => 'From $2.50/lb',
                'Same-day service available' => 'Same-day service available',
                'Fabric softener included' => 'Fabric softener included',
                'Dry Cleaning' => 'Dry Cleaning',
                'Expert care for your delicate and special garments that need extra attention.' => 'Expert care for your delicate and special garments that need extra attention.',
                'From $8.99/item' => 'From $8.99/item',
                'Delicate fabrics' => 'Delicate fabrics',

                // Features Section
                'On-Time Delivery' => 'On-Time Delivery',
                'Guaranteed delivery within promised timeframe.' => 'Guaranteed delivery within promised timeframe.',
                'Care & Quality' => 'Care & Quality',
                'Every item treated with attention.' => 'Every item treated with attention.',
                '100% biodegradable detergents.' => '100% biodegradable detergents.',
                'Your garments are protected.' => 'Your garments are protected.',

                // Badges and descriptions
                'Our Achievements' => 'Our Achievements',
                'Professional Laundry Services' => 'Professional Laundry Services',
                "From everyday laundry to delicate dry cleaning, we've got all your garment care needs covered." => "From everyday laundry to delicate dry cleaning, we've got all your garment care needs covered.",

                // Pricing Section Features
                '15 lbs of laundry' => '15 lbs of laundry',
                'Wash & fold service' => 'Wash & fold service',
                'Free pickup & delivery' => 'Free pickup & delivery',
                '48-hour turnaround' => '48-hour turnaround',
                'Standard detergents' => 'Standard detergents',
                'Premium eco detergents' => 'Premium eco detergents',
                'Same-day turnaround' => 'Same-day turnaround',
                '50 lbs of laundry' => '50 lbs of laundry',
                'Wash, fold & iron' => 'Wash, fold & iron',
                '24-hour turnaround' => '24-hour turnaround',
                '2 dry clean items/month' => '2 dry clean items/month',
                'Unlimited laundry' => 'Unlimited laundry',
                'All services included' => 'All services included',
                'Daily pickup available' => 'Daily pickup available',
                'Dedicated account manager' => 'Dedicated account manager',
                'Priority support' => 'Priority support',
                'Custom branding' => 'Custom branding',

                // Additional Pricing Content
                'Pricing Plans' => 'Pricing Plans',
                'Choose Your Perfect Plan' => 'Choose Your Perfect Plan',
                'Flexible plans designed to fit your lifestyle.' => 'Flexible plans designed to fit your lifestyle.',

                // Footer Section
                'Professional laundry and dry cleaning services with free pickup & delivery.' => 'Professional laundry and dry cleaning services with free pickup & delivery.',
                'Professional event planning and management services to bring your vision to life. From intimate gatherings to large-scale events, we handle every detail with expertise and creativity.' => 'Professional event planning and management services to bring your vision to life. From intimate gatherings to large-scale events, we handle every detail with expertise and creativity.',
                'Our Services' => 'Our Services',
                'Company' => 'Company',
                'Support' => 'Support',
                'Wash & Fold' => 'Wash & Fold',
                'Dry Cleaning' => 'Dry Cleaning',
                'Ironing' => 'Ironing',
                'Alterations' => 'Alterations',
                'About Us' => 'About Us',
                'Our Team' => 'Our Team',
                'Careers' => 'Careers',
                'Contact' => 'Contact',
                'Contact Us' => 'Contact Us',
                'FAQ' => 'FAQ',
                'Help Center' => 'Help Center',
                'Service Areas' => 'Service Areas',
                '© %year% %business%. All rights reserved.' => '© %year% %business%. All rights reserved.',
                'Services' => 'Services',
                'How It Works' => 'How It Works',
                'Pricing' => 'Pricing',
                'Reviews' => 'Reviews',
                'Resources' => 'Resources',
                'Blog' => 'Blog',
                'News' => 'News',
                'Events' => 'Events',
                'Legal' => 'Legal',
                'Privacy Policy' => 'Privacy Policy',
                'Terms of Service' => 'Terms of Service',
                'Cookie Policy' => 'Cookie Policy',
            ],
            'ar' => [
                // Buttons & CTAs
                'View Pricing' => 'عرض الأسعار',
                'Learn More' => 'معرفة المزيد',
                'Get Started' => 'ابدأ الآن',
                'Contact Us' => 'اتصل بنا',
                'Schedule Pickup' => 'جدولة الاستلام',
                'Book Now' => 'احجز الآن',
                'Call Us Now' => 'اتصل بنا الآن',
                'Download Now' => 'حمل الآن',

                // Section Titles
                'Why Choose Us' => 'لماذا تختارنا',
                'Features that set us apart' => 'الميزات التي تميزنا',
                'Our Impact' => 'تأثيرنا',
                'Numbers that speak for themselves' => 'أرقام تتحدث عن نفسها',
                'Our Services' => 'خدماتنا',
                'Comprehensive solutions tailored to your needs' => 'حلول شاملة مصممة خصيصًا لاحتياجاتك',
                'Simple & Transparent Pricing' => 'أسعار بسيطة وشفافة',
                'Choose the perfect plan for your needs' => 'اختر الخطة المثالية لاحتياجاتك',
                'What Our Customers Say' => 'ماذا يقول عملاؤنا',
                "Don't just take our word for it" => 'لا تأخذ كلمتنا فقط',
                'Ready to Get Started?' => 'هل أنت مستعد للبدء؟',
                'Join thousands of satisfied customers today' => 'انضم إلى آلاف العملاء الراضين اليوم',
                'Trusted Partners' => 'شركاء موثوقون',
                'Working with industry leaders' => 'العمل مع قادة الصناعة',
                'Download Our App' => 'حمل تطبيقنا',
                'Get started in seconds' => 'ابدأ في ثوانٍ',
                'Our Portfolio' => 'معرض أعمالنا',
                'Showcasing our finest work' => 'عرض أفضل أعمالنا',

                // Feature Items
                'Eco-Friendly' => 'صديق للبيئة',
                'Fast Service' => 'خدمة سريعة',
                'Fully Insured' => 'مؤمن بالكامل',
                'Free Delivery' => 'توصيل مجاني',
                'Quality Service' => 'خدمة عالية الجودة',
                'Expert Team' => 'فريق خبير',
                'Timely Delivery' => 'توصيل في الوقت المحدد',
                'Customer Care' => 'رعاية العملاء',
                '100% biodegradable detergents for safe cleaning.' => 'منظفات قابلة للتحلل 100٪ للتنظيف الآمن.',
                'Quick turnaround with express options available.' => 'تسليم سريع مع خيارات تعجيل متاحة.',
                'Your garments are protected and insured.' => 'ملابسك محمية ومؤمنة.',
                'Complimentary pickup and delivery service.' => 'خدمة استلام وتوصيل مجانية.',
                'Exceptional quality in everything we do.' => 'جودة استثنائية في كل ما نقوم به.',
                'Experienced professionals at your service.' => 'محترفون ذوو خبرة في خدمتك.',
                'On-time, every time guarantee.' => 'ضمان التسليم في الوقت المحدد في كل مرة.',
                'Dedicated to your satisfaction.' => 'مكرسون لإرضائك.',

                // Service Items
                'Professional Service' => 'خدمة مهنية',
                'High-quality service delivered by experts.' => 'خدمة عالية الجودة يقدمها خبراء.',
                'Contact for pricing' => 'اتصل للحصول على الأسعار',
                'Experienced professionals' => 'محترفون ذوو خبرة',
                'Quality guaranteed' => 'الجودة مضمونة',
                'Fast Turnaround' => 'تسليم سريع',
                'Quick and efficient service delivery.' => 'تقديم خدمة سريعة وفعالة.',
                'Flexible pricing' => 'أسعار مرنة',
                'Express options available' => 'خيارات تعجيل متاحة',
                'On-time delivery' => 'التسليم في الوقت المحدد',
                'Reliable & Trusted' => 'موثوق ومعتمد',
                'Dependable service you can count on.' => 'خدمة يمكنك الاعتماد عليها.',
                'Competitive rates' => 'أسعار تنافسية',
                'Satisfaction guaranteed' => 'الرضا مضمون',
                'Insured and protected' => 'مؤمن ومحمي',

                // Testimonials
                'Business Owner' => 'صاحب عمل',
                'Marketing Director' => 'مدير تسويق',
                'Entrepreneur' => 'رائد أعمال',
                'Exceptional service! They exceeded all my expectations and delivered outstanding results.' => 'خدمة استثنائية! تجاوزوا كل توقعاتي وقدموا نتائج رائعة.',
                'Professional, reliable, and high-quality work. Highly recommended for anyone looking for excellence.' => 'عمل احترافي وموثوق وعالي الجودة. موصى به بشدة لأي شخص يبحث عن التميز.',
                'The best decision I made was choosing this service. Amazing quality and customer support!' => 'أفضل قرار اتخذته هو اختيار هذه الخدمة. جودة مذهلة ودعم عملاء رائع!',

                // Badges
                'Our Services' => 'خدماتنا',
                'Pricing Plans' => 'خطط الأسعار',
                'How It Works' => 'كيف يعمل',
                'Testimonials' => 'الشهادات',
                'FAQ' => 'الأسئلة الشائعة',
                'Contact Us' => 'اتصل بنا',
                'Gallery' => 'معرض الصور',
                'Our Partners' => 'شركاؤنا',
                'Mobile App' => 'تطبيق الجوال',
                'Statistics' => 'الإحصائيات',
                'About Us' => 'من نحن',
                'Our Team' => 'فريقنا',

                // Header/Footer
                'Pricing' => 'الأسعار',
                'More' => 'المزيد',
                'Login' => 'تسجيل الدخول',

                // Restaurant & Food Features
                'Expert Chefs' => 'طهاة خبراء',
                'Skilled culinary professionals.' => 'محترفون في الطهي ذوو مهارة.',
                'Fresh Ingredients' => 'مكونات طازجة',
                'Locally-sourced, organic produce.' => 'منتجات عضوية من مصادر محلية.',
                'Diverse Menu' => 'قائمة متنوعة',
                'Wide variety of delicious options.' => 'مجموعة واسعة من الخيارات اللذيذة.',
                'Cozy Ambiance' => 'أجواء مريحة',
                'Welcoming and comfortable atmosphere.' => 'أجواء ترحيبية ومريحة.',

                // Hero Section Content
                '#1 Rated Laundry Service in Town' => 'خدمة الغسيل رقم 1 في المدينة',
                'Experience premium laundry services with free pickup & delivery. We treat your clothes with care using eco-friendly products.' => 'استمتع بخدمات غسيل متميزة مع التوصيل المجاني. نعامل ملابسك بعناية باستخدام منتجات صديقة للبيئة.',
                'Free Pickup' => 'استلام مجاني',
                '& Delivery' => 'وتوصيل',
                'Express' => 'خدمة سريعة',
                '24h Service' => 'خدمة 24 ساعة',
                'Happy Customers' => 'عملاء سعداء',
                'Orders Completed' => 'طلبات مكتملة',
                'Satisfaction Rate' => 'معدل الرضا',
                'Express Delivery' => 'توصيل سريع',

                // Pricing Section
                'Most Popular' => 'الأكثر شعبية',
                'per month' => 'شهرياً',
                'Basic' => 'أساسي',
                'Family' => 'عائلي',
                'Business' => 'أعمال',
                'Perfect for individuals' => 'مثالي للأفراد',
                'Best for families' => 'الأفضل للعائلات',
                'For businesses & hotels' => 'للشركات والفنادق',

                // How It Works
                'Consultation' => 'استشارة',
                'We discuss your needs and requirements' => 'نناقش احتياجاتك ومتطلباتك',
                'Planning' => 'تخطيط',
                'We create a customized plan for you' => 'نضع خطة مخصصة لك',
                'Execution' => 'تنفيذ',
                'Our experts deliver quality service' => 'خبراؤنا يقدمون خدمة عالية الجودة',
                'Delivery' => 'تسليم',
                'Enjoy your completed project' => 'استمتع بمشروعك المكتمل',

                // Stats values (numbers - usually kept same but maintaining translations)
                '15K+' => '15K+',
                '50K+' => '50K+',
                '99%' => '99%',
                '24h' => '24h',
                '15000+' => '15000+',
                '50000+' => '50000+',

                // Services Section
                'Wash & Fold' => 'غسيل وطي',
                'Professional washing, drying, and folding service for your everyday clothes.' => 'خدمة غسيل وتجفيف وطي احترافية لملابسك اليومية.',
                'From $2.50/lb' => 'من 2.50$ للرطل',
                'Same-day service available' => 'خدمة في نفس اليوم متاحة',
                'Fabric softener included' => 'منعم الأقمشة مضمن',
                'Dry Cleaning' => 'تنظيف جاف',
                'Expert care for your delicate and special garments that need extra attention.' => 'عناية خبيرة بملابسك الحساسة والخاصة التي تحتاج إلى اهتمام إضافي.',
                'From $8.99/item' => 'من 8.99$ للقطعة',
                'Delicate fabrics' => 'أقمشة حساسة',

                // Features Section
                'On-Time Delivery' => 'تسليم في الوقت المحدد',
                'Guaranteed delivery within promised timeframe.' => 'تسليم مضمون خلال الإطار الزمني الموعود.',
                'Care & Quality' => 'عناية وجودة',
                'Every item treated with attention.' => 'كل قطعة تعامل باهتمام.',
                '100% biodegradable detergents.' => 'منظفات قابلة للتحلل 100%.',
                'Your garments are protected.' => 'ملابسك محمية.',

                // Badges and descriptions
                'Our Achievements' => 'إنجازاتنا',
                'Professional Laundry Services' => 'خدمات غسيل احترافية',
                "From everyday laundry to delicate dry cleaning, we've got all your garment care needs covered." => 'من الغسيل اليومي إلى التنظيف الجاف الدقيق، لدينا جميع احتياجات العناية بملابسك.',

                // Pricing Section Features
                '15 lbs of laundry' => '15 رطل من الغسيل',
                'Wash & fold service' => 'خدمة غسيل وطي',
                'Free pickup & delivery' => 'استلام وتوصيل مجاني',
                '48-hour turnaround' => 'تسليم خلال 48 ساعة',
                'Standard detergents' => 'منظفات قياسية',
                'Premium eco detergents' => 'منظفات صديقة للبيئة متميزة',
                'Same-day turnaround' => 'تسليم في نفس اليوم',
                '50 lbs of laundry' => '50 رطل من الغسيل',
                'Wash, fold & iron' => 'غسيل وطي وكي',
                '24-hour turnaround' => 'تسليم خلال 24 ساعة',
                '2 dry clean items/month' => 'قطعتين تنظيف جاف شهرياً',
                'Unlimited laundry' => 'غسيل غير محدود',
                'All services included' => 'جميع الخدمات مضمنة',
                'Daily pickup available' => 'استلام يومي متاح',
                'Dedicated account manager' => 'مدير حساب مخصص',
                'Priority support' => 'دعم ذو أولوية',
                'Custom branding' => 'علامة تجارية مخصصة',

                // Additional Pricing Content
                'Pricing Plans' => 'خطط الأسعار',
                'Choose Your Perfect Plan' => 'اختر خطتك المثالية',
                'Flexible plans designed to fit your lifestyle.' => 'خطط مرنة مصممة لتناسب نمط حياتك.',

                // Footer Section
                'Professional laundry and dry cleaning services with free pickup & delivery.' => 'خدمات غسيل وتنظيف جاف احترافية مع استلام وتوصيل مجاني.',
                'Professional event planning and management services to bring your vision to life. From intimate gatherings to large-scale events, we handle every detail with expertise and creativity.' => 'خدمات تخطيط وإدارة الفعاليات الاحترافية لتحقيق رؤيتك. من التجمعات الحميمة إلى الفعاليات واسعة النطاق، نتعامل مع كل التفاصيل بخبرة وإبداع.',
                'Our Services' => 'خدماتنا',
                'Company' => 'الشركة',
                'Support' => 'الدعم',
                'Wash & Fold' => 'غسيل وطي',
                'Dry Cleaning' => 'تنظيف جاف',
                'Ironing' => 'كي',
                'Alterations' => 'تعديلات',
                'About Us' => 'من نحن',
                'Our Team' => 'فريقنا',
                'Careers' => 'الوظائف',
                'Contact' => 'اتصل بنا',
                'Contact Us' => 'اتصل بنا',
                'FAQ' => 'الأسئلة الشائعة',
                'Help Center' => 'مركز المساعدة',
                'Service Areas' => 'مناطق الخدمة',
                '© %year% %business%. All rights reserved.' => '© %year% %business%. جميع الحقوق محفوظة.',
                'Services' => 'الخدمات',
                'How It Works' => 'كيف يعمل',
                'Pricing' => 'الأسعار',
                'Reviews' => 'المراجعات',
                'Resources' => 'الموارد',
                'Blog' => 'المدونة',
                'News' => 'الأخبار',
                'Events' => 'الفعاليات',
                'Legal' => 'قانوني',
                'Privacy Policy' => 'سياسة الخصوصية',
                'Terms of Service' => 'شروط الخدمة',
                'Cookie Policy' => 'سياسة ملفات تعريف الارتباط',
            ],
        ];

        return $translations[$langKey][$text] ?? ($translations['en'][$text] ?? $text);
    }

    // Helper methods for category-specific content with translations
    protected function getHeroTitle($categoryName, $langKey = 'en'): string
    {
        $titles = [
            'en' => [
                'Laundry & Dry Cleaning' => 'Fresh & Clean',
                'Restaurant & Food' => 'Delicious Dining',
                'Salon & Spa' => 'Beauty & Wellness',
                'Healthcare & Medical' => 'Compassionate Care',
                'Fitness & Gym' => 'Transform Your Body',
                'Real Estate' => 'Dream Homes',
                'E-Commerce' => 'Shop Smart',
                'Education & Training' => 'Learn & Grow',
                'Travel & Tourism' => 'Explore the World',
                'Automotive' => 'Expert Auto Care',
                'Photography & Media' => 'Capture Moments',
                'Event Management' => 'Unforgettable Events',
                'Consulting & Professional Services' => 'Strategic Solutions',
                'Technology & IT' => 'Tech Innovation',
                'Retail & Shopping' => 'Quality Products',
            ],
            'ar' => [
                'Laundry & Dry Cleaning' => 'نظيف ومنعش',
                'Restaurant & Food' => 'طعام شهي',
                'Salon & Spa' => 'الجمال والعافية',
                'Healthcare & Medical' => 'رعاية متعاطفة',
                'Fitness & Gym' => 'حول جسمك',
                'Real Estate' => 'منازل الأحلام',
                'E-Commerce' => 'تسوق ذكي',
                'Education & Training' => 'تعلم وانمو',
                'Travel & Tourism' => 'استكشف العالم',
                'Automotive' => 'خدمة سيارات خبيرة',
                'Photography & Media' => 'التقط اللحظات',
                'Event Management' => 'فعاليات لا تُنسى',
                'Consulting & Professional Services' => 'حلول استراتيجية',
                'Technology & IT' => 'ابتكار تقني',
                'Retail & Shopping' => 'منتجات عالية الجودة',
            ],
        ];

        return $titles[$langKey][$categoryName] ?? ($titles['en'][$categoryName] ?? 'Excellence Delivered');
    }

    protected function getHeroSubtitle($categoryName, $langKey = 'en'): string
    {
        $subtitles = [
            'en' => [
                'Laundry & Dry Cleaning' => 'Premium Service',
                'Restaurant & Food' => 'Fresh Ingredients',
                'Salon & Spa' => 'Expert Styling',
                'Healthcare & Medical' => 'Professional Care',
                'Fitness & Gym' => 'Expert Trainers',
                'Real Estate' => 'Expert Guidance',
                'E-Commerce' => 'Fast Delivery',
                'Education & Training' => 'Expert Instructors',
                'Travel & Tourism' => 'Curated Experiences',
                'Automotive' => 'Certified Technicians',
                'Photography & Media' => 'Professional Photography',
                'Event Management' => 'Flawless Execution',
                'Consulting & Professional Services' => 'Proven Methodologies',
                'Technology & IT' => 'Modern Solutions',
                'Retail & Shopping' => 'Personalized Shopping',
            ],
            'ar' => [
                'Laundry & Dry Cleaning' => 'خدمة متميزة',
                'Restaurant & Food' => 'مكونات طازجة',
                'Salon & Spa' => 'تصفيف خبير',
                'Healthcare & Medical' => 'رعاية مهنية',
                'Fitness & Gym' => 'مدربون خبراء',
                'Real Estate' => 'إرشاد خبير',
                'E-Commerce' => 'توصيل سريع',
                'Education & Training' => 'مدربون خبراء',
                'Travel & Tourism' => 'تجارب منسقة',
                'Automotive' => 'فنيون معتمدون',
                'Photography & Media' => 'تصوير احترافي',
                'Event Management' => 'تنفيذ لا تشوبه شائبة',
                'Consulting & Professional Services' => 'منهجيات مثبتة',
                'Technology & IT' => 'حلول حديثة',
                'Retail & Shopping' => 'تسوق شخصي',
            ],
        ];

        return $subtitles[$langKey][$categoryName] ?? ($subtitles['en'][$categoryName] ?? 'Quality Service');
    }

    protected function getHeroDescription($categoryName, $langKey = 'en'): string
    {
        $descriptions = [
            'en' => [
                'Laundry & Dry Cleaning' => 'Experience premium laundry services with free pickup & delivery. We treat your clothes with care using eco-friendly products.',
                'Restaurant & Food' => 'Savor exceptional cuisine crafted from fresh, locally-sourced ingredients in a welcoming atmosphere.',
                'Salon & Spa' => 'Indulge in premium beauty and wellness services delivered by expert stylists and therapists.',
                'Healthcare & Medical' => 'Receive compassionate, quality healthcare from experienced professionals dedicated to your wellbeing.',
                'Fitness & Gym' => 'Achieve your fitness goals with modern equipment, expert trainers, and motivating group classes.',
                'Real Estate' => 'Discover your perfect property with comprehensive listings and personalized guidance from experienced agents.',
                'E-Commerce' => 'Shop quality products with secure payments and fast delivery. Enjoy a seamless online shopping experience.',
                'Education & Training' => 'Unlock your potential with expert instructors, comprehensive courses, and flexible learning options.',
                'Travel & Tourism' => 'Create unforgettable memories with our curated travel experiences and professional planning services.',
                'Automotive' => 'Keep your vehicle running smoothly with professional maintenance and repairs from certified technicians.',
                'Photography & Media' => 'Capture your special moments with professional photography and media services for every occasion.',
                'Event Management' => 'Bring your vision to life with professional event planning and management from start to finish.',
                'Consulting & Professional Services' => 'Transform your business with strategic consulting and expert solutions tailored to your needs.',
                'Technology & IT' => 'Power your digital transformation with innovative technology solutions and reliable support.',
                'Retail & Shopping' => 'Discover quality products and enjoy personalized service in your favorite shopping destination.',
            ],
            'ar' => [
                'Laundry & Dry Cleaning' => 'استمتع بخدمات غسيل متميزة مع خدمة التوصيل المجانية. نعامل ملابسك بعناية باستخدام منتجات صديقة للبيئة.',
                'Restaurant & Food' => 'استمتع بمأكولات استثنائية مصنوعة من مكونات طازجة ومحلية في أجواء ترحيبية.',
                'Salon & Spa' => 'انغمس في خدمات الجمال والعافية المتميزة التي يقدمها خبراء التصفيف والمعالجون.',
                'Healthcare & Medical' => 'احصل على رعاية صحية عالية الجودة من متخصصين ذوي خبرة مكرسين لرفاهيتك.',
                'Fitness & Gym' => 'حقق أهداف اللياقة البدنية مع معدات حديثة ومدربين خبراء وفصول جماعية محفزة.',
                'Real Estate' => 'اكتشف عقارك المثالي مع قوائم شاملة وإرشادات شخصية من وكلاء ذوي خبرة.',
                'E-Commerce' => 'تسوق منتجات عالية الجودة مع مدفوعات آمنة وتوصيل سريع. استمتع بتجربة تسوق سلسة عبر الإنترنت.',
                'Education & Training' => 'أطلق العنان لإمكاناتك مع مدربين خبراء ودورات شاملة وخيارات تعلم مرنة.',
                'Travel & Tourism' => 'اصنع ذكريات لا تُنسى مع تجارب السفر المنسقة وخدمات التخطيط المهنية.',
                'Automotive' => 'حافظ على سيارتك تعمل بسلاسة مع الصيانة والإصلاحات المهنية من فنيين معتمدين.',
                'Photography & Media' => 'التقط لحظاتك الخاصة مع خدمات التصوير والإعلام المهنية لكل مناسبة.',
                'Event Management' => 'حقق رؤيتك مع التخطيط والإدارة المهنية للفعاليات من البداية إلى النهاية.',
                'Consulting & Professional Services' => 'حول عملك بالاستشارات الاستراتيجية والحلول الخبيرة المصممة لاحتياجاتك.',
                'Technology & IT' => 'عزز تحولك الرقمي بحلول تقنية مبتكرة ودعم موثوق.',
                'Retail & Shopping' => 'اكتشف منتجات عالية الجودة واستمتع بخدمة شخصية في وجهة التسوق المفضلة لديك.',
            ],
        ];

        return $descriptions[$langKey][$categoryName] ?? ($descriptions['en'][$categoryName] ?? 'We provide exceptional services tailored to meet your unique needs with professionalism and quality.');
    }

    protected function getHeroImageUrl($categoryName): string
    {
        $imageUrls = [
            'Laundry & Dry Cleaning' => 'https://images.unsplash.com/photo-1517677208171-0bc6725a3e60?q=80&w=1920&auto=format&fit=crop',
            'Restaurant & Food' => 'https://images.unsplash.com/photo-1517248135467-4c7edcad34c4?q=80&w=1920&auto=format&fit=crop',
            'Salon & Spa' => 'https://images.unsplash.com/photo-1560066984-138dadb4c035?q=80&w=1920&auto=format&fit=crop',
            'Healthcare & Medical' => 'https://images.unsplash.com/photo-1519494026892-80bbd2d6fd0d?q=80&w=1920&auto=format&fit=crop',
            'Fitness & Gym' => 'https://images.unsplash.com/photo-1534438327276-14e5300c3a48?q=80&w=1920&auto=format&fit=crop',
            'Real Estate' => 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?q=80&w=1920&auto=format&fit=crop',
            'E-Commerce' => 'https://images.unsplash.com/photo-1441986300917-64674bd600d8?q=80&w=1920&auto=format&fit=crop',
            'Education & Training' => 'https://images.unsplash.com/photo-1427504494785-3a9ca7044f45?q=80&w=1920&auto=format&fit=crop',
            'Travel & Tourism' => 'https://images.unsplash.com/photo-1488646953014-85cb44e25828?q=80&w=1920&auto=format&fit=crop',
            'Automotive' => 'https://images.unsplash.com/photo-1486262715619-67b85e0b08d3?q=80&w=1920&auto=format&fit=crop',
            'Photography & Media' => 'https://images.unsplash.com/photo-1542038784456-1ea8e935640e?q=80&w=1920&auto=format&fit=crop',
            'Event Management' => 'https://images.unsplash.com/photo-1511578314322-379afb476865?q=80&w=1920&auto=format&fit=crop',
            'Consulting & Professional Services' => 'https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?q=80&w=1920&auto=format&fit=crop',
            'Technology & IT' => 'https://images.unsplash.com/photo-1518770660439-4636190af475?q=80&w=1920&auto=format&fit=crop',
            'Retail & Shopping' => 'https://images.unsplash.com/photo-1441984904996-e0b6ba687e04?q=80&w=1920&auto=format&fit=crop',
        ];

        return $imageUrls[$categoryName] ?? 'https://images.unsplash.com/photo-1497366216548-37526070297c?q=80&w=1920&auto=format&fit=crop';
    }

    protected function getGalleryImages($categoryName): array
    {
        $galleryImages = [
            'Salon & Spa' => [
                ['image' => 'https://images.unsplash.com/photo-1560066984-138dadb4c035?w=800&q=80', 'alt' => 'Modern salon interior'],
                ['image' => 'https://images.unsplash.com/photo-1519415510236-718bdfcd89c8?w=800&q=80', 'alt' => 'Professional hair styling'],
                ['image' => 'https://images.unsplash.com/photo-1582095133179-bfd08e2fc6b3?w=800&q=80', 'alt' => 'Spa treatment room'],
                ['image' => 'https://images.unsplash.com/photo-1516975080664-ed2fc6a32937?w=800&q=80', 'alt' => 'Relaxing massage'],
                ['image' => 'https://images.unsplash.com/photo-1604654894610-df63bc536371?w=800&q=80', 'alt' => 'Manicure service'],
                ['image' => 'https://images.unsplash.com/photo-1522337660859-02fbefca4702?w=800&q=80', 'alt' => 'Beauty products'],
            ],
            'Photography & Media' => [
                ['image' => 'https://images.unsplash.com/photo-1542038784456-1ea8e935640e?w=800&q=80', 'alt' => 'Professional camera equipment'],
                ['image' => 'https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=800&q=80', 'alt' => 'Photography studio'],
                ['image' => 'https://images.unsplash.com/photo-1519741497674-611481863552?w=800&q=80', 'alt' => 'Wedding photography'],
                ['image' => 'https://images.unsplash.com/photo-1554415707-6e8cfc93fe23?w=800&q=80', 'alt' => 'Portrait session'],
                ['image' => 'https://images.unsplash.com/photo-1531746790731-6c087fecd65a?w=800&q=80', 'alt' => 'Commercial photography'],
                ['image' => 'https://images.unsplash.com/photo-1452587925148-ce544e77e70d?w=800&q=80', 'alt' => 'Video production'],
            ],
            'Event Management' => [
                ['image' => 'https://images.unsplash.com/photo-1511578314322-379afb476865?w=800&q=80', 'alt' => 'Wedding event setup'],
                ['image' => 'https://images.unsplash.com/photo-1540575467063-178a50c2df87?w=800&q=80', 'alt' => 'Corporate event hall'],
                ['image' => 'https://images.unsplash.com/photo-1464366400600-7168b8af9bc3?w=800&q=80', 'alt' => 'Elegant table settings'],
                ['image' => 'https://images.unsplash.com/photo-1505236858219-8359eb29e329?w=800&q=80', 'alt' => 'Conference setup'],
                ['image' => 'https://images.unsplash.com/photo-1478147427282-58a87a120781?w=800&q=80', 'alt' => 'Outdoor event venue'],
                ['image' => 'https://images.unsplash.com/photo-1530103862676-de8c9debad1d?w=800&q=80', 'alt' => 'Birthday celebration'],
            ],
            'Real Estate' => [
                ['image' => 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=800&q=80', 'alt' => 'Modern house exterior'],
                ['image' => 'https://images.unsplash.com/photo-1512917774080-9991f1c4c750?w=800&q=80', 'alt' => 'Luxury home interior'],
                ['image' => 'https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?w=800&q=80', 'alt' => 'Contemporary kitchen'],
                ['image' => 'https://images.unsplash.com/photo-1600607687939-ce8a6c25118c?w=800&q=80', 'alt' => 'Master bedroom'],
                ['image' => 'https://images.unsplash.com/photo-1600585154340-be6161a56a0c?w=800&q=80', 'alt' => 'Spacious living room'],
                ['image' => 'https://images.unsplash.com/photo-1600585154526-990dced4db0d?w=800&q=80', 'alt' => 'Modern bathroom'],
            ],
            'Restaurant & Food' => [
                ['image' => 'https://images.unsplash.com/photo-1517248135467-4c7edcad34c4?w=800&q=80', 'alt' => 'Restaurant interior'],
                ['image' => 'https://images.unsplash.com/photo-1414235077428-338989a2e8c0?w=800&q=80', 'alt' => 'Gourmet dish plating'],
                ['image' => 'https://images.unsplash.com/photo-1559339352-11d035aa65de?w=800&q=80', 'alt' => 'Fresh ingredients'],
                ['image' => 'https://images.unsplash.com/photo-1504674900247-0877df9cc836?w=800&q=80', 'alt' => 'Chef preparing food'],
                ['image' => 'https://images.unsplash.com/photo-1555939594-58d7cb561ad1?w=800&q=80', 'alt' => 'Breakfast spread'],
                ['image' => 'https://images.unsplash.com/photo-1551218808-94e220e084d2?w=800&q=80', 'alt' => 'Dinner setting'],
            ],
        ];

        return $galleryImages[$categoryName] ?? [
            ['image' => 'https://images.unsplash.com/photo-1497366216548-37526070297c?w=800&q=80', 'alt' => 'Business showcase 1'],
            ['image' => 'https://images.unsplash.com/photo-1497366811353-6870744d04b2?w=800&q=80', 'alt' => 'Business showcase 2'],
            ['image' => 'https://images.unsplash.com/photo-1497366754035-f200968a6e72?w=800&q=80', 'alt' => 'Business showcase 3'],
            ['image' => 'https://images.unsplash.com/photo-1497215728101-856f4ea42174?w=800&q=80', 'alt' => 'Business showcase 4'],
            ['image' => 'https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?w=800&q=80', 'alt' => 'Business showcase 5'],
            ['image' => 'https://images.unsplash.com/photo-1486312338219-ce68d2c6f44d?w=800&q=80', 'alt' => 'Business showcase 6'],
        ];
    }

    protected function getFeatureItems($categoryName, $langKey = 'en'): array
    {
        $features = [
            'Laundry & Dry Cleaning' => [
                ['icon' => 'lucide:leaf', 'title' => $this->translate('Eco-Friendly', $langKey), 'description' => $this->translate('100% biodegradable detergents for safe cleaning.', $langKey)],
                ['icon' => 'lucide:clock', 'title' => $this->translate('Fast Service', $langKey), 'description' => $this->translate('Quick turnaround with express options available.', $langKey)],
                ['icon' => 'lucide:shield-check', 'title' => $this->translate('Fully Insured', $langKey), 'description' => $this->translate('Your garments are protected and insured.', $langKey)],
                ['icon' => 'lucide:truck', 'title' => $this->translate('Free Delivery', $langKey), 'description' => $this->translate('Complimentary pickup and delivery service.', $langKey)],
            ],
            'Restaurant & Food' => [
                ['icon' => 'lucide:chef-hat', 'title' => $this->translate('Expert Chefs', $langKey), 'description' => $this->translate('Skilled culinary professionals.', $langKey)],
                ['icon' => 'lucide:leaf', 'title' => $this->translate('Fresh Ingredients', $langKey), 'description' => $this->translate('Locally-sourced, organic produce.', $langKey)],
                ['icon' => 'lucide:utensils', 'title' => $this->translate('Diverse Menu', $langKey), 'description' => $this->translate('Wide variety of delicious options.', $langKey)],
                ['icon' => 'lucide:home', 'title' => $this->translate('Cozy Ambiance', $langKey), 'description' => $this->translate('Welcoming and comfortable atmosphere.', $langKey)],
            ],
        ];

        return $features[$categoryName] ?? [
            ['icon' => 'lucide:check', 'title' => $this->translate('Quality Service', $langKey), 'description' => $this->translate('Exceptional quality in everything we do.', $langKey)],
            ['icon' => 'lucide:users', 'title' => $this->translate('Expert Team', $langKey), 'description' => $this->translate('Experienced professionals at your service.', $langKey)],
            ['icon' => 'lucide:clock', 'title' => $this->translate('Timely Delivery', $langKey), 'description' => $this->translate('On-time, every time guarantee.', $langKey)],
            ['icon' => 'lucide:heart', 'title' => $this->translate('Customer Care', $langKey), 'description' => $this->translate('Dedicated to your satisfaction.', $langKey)],
        ];
    }

    protected function getServiceItems($categoryName, $langKey = 'en'): array
    {
        $services = [
            'Laundry & Dry Cleaning' => [
                [
                    'icon' => 'lucide:shirt',
                    'title' => 'Wash & Fold',
                    'description' => 'Professional washing, drying, and folding for everyday clothes.',
                    'price' => 'From $2.50/lb',
                    'features' => [
                        ['text' => 'Same-day service available'],
                        ['text' => 'Eco-friendly detergents'],
                    ],
                    'button_text' => $this->translate('Learn More', $langKey),
                    'button_url' => '#contact',
                    'button_icon' => 'lucide:arrow-right',
                ],
                [
                    'icon' => 'lucide:sparkles',
                    'title' => 'Dry Cleaning',
                    'description' => 'Expert care for delicate and special garments.',
                    'price' => 'From $8.99/item',
                    'features' => [
                        ['text' => 'Suits & formal wear'],
                        ['text' => 'Stain treatment included'],
                    ],
                    'button_text' => $this->translate('Learn More', $langKey),
                    'button_url' => '#contact',
                    'button_icon' => 'lucide:arrow-right',
                ],
                [
                    'icon' => 'lucide:wind',
                    'title' => 'Press & Iron',
                    'description' => 'Professional pressing for a crisp, polished look.',
                    'price' => 'From $3.50/item',
                    'features' => [
                        ['text' => 'Steam pressing'],
                        ['text' => 'Ready to wear'],
                    ],
                    'button_text' => $this->translate('Learn More', $langKey),
                    'button_url' => '#contact',
                    'button_icon' => 'lucide:arrow-right',
                ],
            ],
        ];

        return $services[$categoryName] ?? [
            [
                'icon' => 'lucide:star',
                'title' => $this->translate('Professional Service', $langKey),
                'description' => $this->translate('High-quality service delivered by experts.', $langKey),
                'price' => $this->translate('Contact for pricing', $langKey),
                'features' => [
                    ['text' => $this->translate('Experienced professionals', $langKey)],
                    ['text' => $this->translate('Quality guaranteed', $langKey)],
                ],
                'button_text' => $this->translate('Learn More', $langKey),
                'button_url' => '#contact',
                'button_icon' => 'lucide:arrow-right',
            ],
            [
                'icon' => 'lucide:zap',
                'title' => $this->translate('Fast Turnaround', $langKey),
                'description' => $this->translate('Quick and efficient service delivery.', $langKey),
                'price' => $this->translate('Flexible pricing', $langKey),
                'features' => [
                    ['text' => $this->translate('Express options available', $langKey)],
                    ['text' => $this->translate('On-time delivery', $langKey)],
                ],
                'button_text' => $this->translate('Learn More', $langKey),
                'button_url' => '#contact',
                'button_icon' => 'lucide:arrow-right',
            ],
            [
                'icon' => 'lucide:shield',
                'title' => $this->translate('Reliable & Trusted', $langKey),
                'description' => $this->translate('Dependable service you can count on.', $langKey),
                'price' => $this->translate('Competitive rates', $langKey),
                'features' => [
                    ['text' => $this->translate('Satisfaction guaranteed', $langKey)],
                    ['text' => $this->translate('Insured and protected', $langKey)],
                ],
                'button_text' => $this->translate('Learn More', $langKey),
                'button_url' => '#contact',
                'button_icon' => 'lucide:arrow-right',
            ],
        ];
    }
}
