<?php

namespace Database\Seeders;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class WebsiteSettingSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        Model::unguard();

        DB::table('website_settings')->delete();

        DB::statement('ALTER TABLE website_settings AUTO_INCREMENT = 1');

        // Get all websites with their categories
        $websites = DB::table('websites')
            ->leftJoin('website_categories', 'websites.website_category_id', '=', 'website_categories.id')
            ->select('websites.*', 'website_categories.name as category_name')
            ->get();

        $totalWebsites = $websites->count();
        $totalSettings = 0;

        $this->command->info("Processing {$totalWebsites} websites...");

        // Create website settings for each website and its languages
        foreach ($websites as $website) {
            // Get languages for this website only
            $websiteLanguages = DB::table('website_languages')
                ->where('website_id', $website->id)
                ->where('enabled', true)
                ->get();

            foreach ($websiteLanguages as $lang) {
                // Get category-specific content with language support
                $categoryContent = $this->getCategoryContent($website->category_name, $lang->key);

                // Build settings array
                $settings = [
                    'company_id' => $website->company_id,
                    'website_id' => $website->id,
                    'website_language_id' => $lang->id,
                    'module_name' => $categoryContent['module_name'],
                    'is_global' => false,
                    'status' => true,

                    // Website name and identity
                    'website_name' => $website->name,
                    'website_tagline' => $categoryContent['tagline'],
                    'website_description' => $categoryContent['description'],
                    'website_phone' => '+1 (555) 123-4567',
                    'website_email' => 'hello@' . strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $website->name)) . '.com',
                    'website_address' => $categoryContent['address'],
                    'website_hours' => $categoryContent['hours'],
                    'website_favicon' => null,

                    // SEO Configuration
                    'seo_title' => $website->name . ' - ' . $categoryContent['seo_title'],
                    'seo_description' => $categoryContent['seo_description'],
                    'seo_keywords' => $categoryContent['seo_keywords'],
                    'seo_author' => $website->name,
                    'seo_robots' => 'index, follow',
                    'seo_og_type' => 'website',
                    'seo_og_locale' => 'en_US',
                    'seo_og_site_name' => $website->name,
                    'seo_og_image' => null,
                    'seo_twitter_card' => 'summary_large_image',
                    'seo_twitter_site' => '@' . strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $website->name)),
                    'seo_twitter_creator' => '@' . strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $website->name)),

                    // Logos
                    'light_logo' => null,
                    'dark_logo' => null,
                    'light_logo_small' => null,
                    'dark_logo_small' => null,
                    'header_logo_type' => 'logo_with_name',

                    // Social Links (using Iconify format: prefix:icon-name)
                    'social_links' => json_encode([
                        ['name' => 'Facebook', 'url' => '#', 'icon' => 'mdi:facebook'],
                        ['name' => 'Instagram', 'url' => '#', 'icon' => 'mdi:instagram'],
                        ['name' => 'Twitter', 'url' => '#', 'icon' => 'mdi:twitter'],
                        ['name' => 'LinkedIn', 'url' => '#', 'icon' => 'mdi:linkedin'],
                    ]),

                    // Show social icons globally (header, footer, contact)
                    'show_social_icon' => true,

                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                DB::table('website_settings')->insert($settings);
                $totalSettings++;
            }
        }

        $this->command->info("✓ Successfully created {$totalSettings} website settings for {$totalWebsites} websites");
    }

    /**
     * Get category-specific content for descriptions and SEO with language support
     */
    private function getCategoryContent($categoryName, $langKey = 'en'): array
    {
        $categoryContent = [
            'Laundry & Dry Cleaning' => [
                'module_name' => 'Laundry',
                'tagline' => $this->translateText('Premium Laundry & Dry Cleaning Services', $langKey),
                'description' => $this->translateText('Professional laundry and dry cleaning services with free pickup and delivery. Experience the difference with our eco-friendly cleaning solutions and expert garment care.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Professional Laundry & Dry Cleaning Services | Free Pickup & Delivery',
                'seo_description' => 'Expert laundry and dry cleaning services with convenient pickup & delivery. Eco-friendly products, 24-hour express service, and professional garment care you can trust.',
                'seo_keywords' => 'laundry service, dry cleaning, wash and fold, pickup delivery laundry, eco-friendly cleaning, garment care, express laundry',
            ],
            'Restaurant & Food' => [
                'module_name' => 'Restaurant',
                'tagline' => $this->translateText('Delicious Dining Experience', $langKey),
                'description' => $this->translateText('Experience exceptional cuisine crafted with fresh, locally-sourced ingredients. From intimate dinners to family gatherings, we serve memorable meals in a welcoming atmosphere.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Fine Dining Restaurant | Fresh Local Cuisine & Exceptional Service',
                'seo_description' => 'Discover culinary excellence with our chef-prepared dishes using fresh, local ingredients. Enjoy a memorable dining experience with exceptional service and ambiance.',
                'seo_keywords' => 'restaurant, fine dining, local cuisine, fresh food, chef prepared meals, dining experience, food delivery, reservations',
            ],
            'Salon & Spa' => [
                'module_name' => 'Salon',
                'tagline' => $this->translateText('Beauty & Wellness Sanctuary', $langKey),
                'description' => $this->translateText('Transform your look and rejuvenate your spirit with our premium salon and spa services. Expert stylists, luxurious treatments, and personalized care for your ultimate relaxation.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Luxury Salon & Spa | Hair, Beauty & Wellness Services',
                'seo_description' => 'Premium salon and spa offering expert hair styling, beauty treatments, and wellness services. Book your appointment today for a transformative beauty experience.',
                'seo_keywords' => 'salon, spa, hair styling, beauty treatments, wellness, massage, facials, manicure, pedicure, beauty services',
            ],
            'Healthcare & Medical' => [
                'module_name' => 'Healthcare',
                'tagline' => $this->translateText('Comprehensive Healthcare Solutions', $langKey),
                'description' => $this->translateText('Providing compassionate, quality healthcare services with experienced medical professionals. Your health and well-being are our top priorities with personalized treatment plans.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Quality Healthcare Services | Experienced Medical Professionals',
                'seo_description' => 'Trusted healthcare services with compassionate care from experienced medical professionals. Offering comprehensive treatment, diagnostics, and wellness programs.',
                'seo_keywords' => 'healthcare, medical services, doctors, medical care, health clinic, diagnostics, patient care, wellness programs',
            ],
            'Fitness & Gym' => [
                'module_name' => 'Fitness',
                'tagline' => $this->translateText('Your Fitness Journey Starts Here', $langKey),
                'description' => $this->translateText('State-of-the-art fitness facility with expert trainers, modern equipment, and motivating group classes. Achieve your fitness goals in a supportive and energizing environment.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Modern Fitness Center & Gym | Personal Training & Group Classes',
                'seo_description' => 'Transform your fitness journey with our modern gym facility, expert personal trainers, and energizing group classes. Join today and achieve your health goals.',
                'seo_keywords' => 'gym, fitness center, personal training, group classes, workout, exercise, fitness programs, health club, strength training',
            ],
            'Real Estate' => [
                'module_name' => 'RealEstate',
                'tagline' => $this->translateText('Find Your Dream Property', $langKey),
                'description' => $this->translateText('Discover your perfect home or investment property with our expert real estate services. Comprehensive listings, market insights, and personalized guidance throughout your property journey.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Real Estate Services | Homes, Properties & Investment Opportunities',
                'seo_description' => 'Find your dream home or investment property with expert real estate services. Browse listings, get market insights, and receive personalized guidance from experienced agents.',
                'seo_keywords' => 'real estate, property for sale, homes for sale, buy property, sell property, real estate agent, property listings, investment property',
            ],
            'E-Commerce' => [
                'module_name' => 'Ecommerce',
                'tagline' => $this->translateText('Shop Online with Confidence', $langKey),
                'description' => $this->translateText('Your trusted online shopping destination offering quality products, secure payments, and fast delivery. Browse our extensive catalog and enjoy a seamless shopping experience.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Online Shopping | Quality Products with Fast Delivery',
                'seo_description' => 'Shop online for quality products with secure payments and fast delivery. Browse our extensive catalog and enjoy hassle-free returns and excellent customer service.',
                'seo_keywords' => 'online shopping, e-commerce, buy online, online store, products, fast delivery, secure shopping, online retail',
            ],
            'Education & Training' => [
                'module_name' => 'Education',
                'tagline' => $this->translateText('Empowering Minds, Building Futures', $langKey),
                'description' => $this->translateText('Quality education and professional training programs designed to unlock your potential. Expert instructors, comprehensive curriculum, and flexible learning options for your success.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Education & Training Programs | Expert Instructors & Flexible Learning',
                'seo_description' => 'Advance your knowledge with quality education and professional training programs. Expert instructors, comprehensive courses, and flexible learning options to achieve your goals.',
                'seo_keywords' => 'education, training programs, courses, online learning, professional development, certification, tutoring, skills training',
            ],
            'Travel & Tourism' => [
                'module_name' => 'Travel',
                'tagline' => $this->translateText('Your Gateway to Amazing Adventures', $langKey),
                'description' => $this->translateText('Create unforgettable memories with our curated travel experiences and tourism services. From exotic destinations to local tours, we make your travel dreams come true.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Travel & Tourism Services | Unforgettable Vacation Experiences',
                'seo_description' => 'Plan your perfect vacation with our travel and tourism services. Explore amazing destinations, book tours, and create lasting memories with expert travel planning.',
                'seo_keywords' => 'travel, tourism, vacation packages, tours, destinations, travel planning, holiday booking, adventure travel, travel agency',
            ],
            'Automotive' => [
                'module_name' => 'Automotive',
                'tagline' => $this->translateText('Expert Auto Care & Services', $langKey),
                'description' => $this->translateText('Professional automotive services and maintenance you can trust. From routine maintenance to complex repairs, our certified technicians keep your vehicle running smoothly.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Professional Auto Repair & Maintenance Services | Certified Technicians',
                'seo_description' => 'Trusted automotive services with certified technicians. Offering comprehensive car maintenance, repairs, and diagnostics to keep your vehicle in top condition.',
                'seo_keywords' => 'auto repair, car maintenance, automotive services, vehicle service, car repair, oil change, brake service, engine repair',
            ],
            'Photography & Media' => [
                'module_name' => 'Photography',
                'tagline' => $this->translateText('Capturing Moments, Creating Memories', $langKey),
                'description' => $this->translateText('Professional photography and media services to capture your special moments. Creative vision, technical expertise, and stunning results for every occasion.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Professional Photography & Media Services | Creative Visual Solutions',
                'seo_description' => 'Expert photography and media services for weddings, events, portraits, and commercial projects. Creative professionals delivering stunning visual content.',
                'seo_keywords' => 'photography, photographer, media services, wedding photography, portrait photography, event photography, commercial photography, video production',
            ],
            'Event Management' => [
                'module_name' => 'Events',
                'tagline' => $this->translateText('Creating Unforgettable Events', $langKey),
                'description' => $this->translateText('Professional event planning and management services to bring your vision to life. From intimate gatherings to large-scale events, we handle every detail with expertise and creativity.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Event Planning & Management Services | Memorable Experiences',
                'seo_description' => 'Expert event management services for weddings, corporate events, and special occasions. Professional planning, coordination, and execution for unforgettable experiences.',
                'seo_keywords' => 'event management, event planning, wedding planning, corporate events, event coordinator, party planning, event services',
            ],
            'Consulting & Professional Services' => [
                'module_name' => 'Consulting',
                'tagline' => $this->translateText('Expert Advice, Strategic Solutions', $langKey),
                'description' => $this->translateText('Strategic consulting and professional services to help your business thrive. Expert guidance, proven methodologies, and customized solutions for your unique challenges.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Business Consulting & Professional Services | Strategic Solutions',
                'seo_description' => 'Professional consulting services offering strategic guidance and expert solutions. Transform your business with proven methodologies and customized strategies.',
                'seo_keywords' => 'consulting, business consulting, professional services, strategic planning, business advisor, management consulting, business solutions',
            ],
            'Technology & IT' => [
                'module_name' => 'Technology',
                'tagline' => $this->translateText('Innovative Technology Solutions', $langKey),
                'description' => $this->translateText('Cutting-edge technology and IT services to power your digital transformation. Expert solutions, reliable support, and innovative approaches for your business success.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'IT Services & Technology Solutions | Digital Transformation Experts',
                'seo_description' => 'Professional IT services and technology solutions for businesses. From software development to IT support, we deliver innovative solutions for your digital needs.',
                'seo_keywords' => 'IT services, technology solutions, software development, IT support, digital transformation, tech consulting, cloud services',
            ],
            'Retail & Shopping' => [
                'module_name' => 'Retail',
                'tagline' => $this->translateText('Quality Products, Exceptional Service', $langKey),
                'description' => $this->translateText('Your favorite shopping destination offering quality products and personalized service. Discover curated collections, exclusive deals, and a shopping experience you\'ll love.', $langKey),
                'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
                'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
                'seo_title' => 'Retail Store | Quality Products & Personalized Shopping Experience',
                'seo_description' => 'Shop quality products with exceptional customer service. Browse our curated collections, enjoy exclusive deals, and experience personalized shopping.',
                'seo_keywords' => 'retail store, shopping, quality products, retail shopping, store, merchandise, shopping center, retail outlet',
            ],
        ];

        // Return category-specific content or default content
        return $categoryContent[$categoryName] ?? [
            'module_name' => 'Business',
            'tagline' => $this->translateText('Quality Services You Can Trust', $langKey),
            'description' => $this->translateText('We provide exceptional services tailored to meet your needs. Experience professionalism, quality, and reliability with every interaction.', $langKey),
            'address' => $this->translateText('123 Main Street, City, State 12345', $langKey),
            'hours' => $this->translateText('Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed', $langKey),
            'seo_title' => 'Professional Services | Quality & Reliability',
            'seo_description' => 'Discover professional services designed to exceed your expectations. Quality workmanship, reliable service, and customer satisfaction guaranteed.',
            'seo_keywords' => 'professional services, quality service, reliable business, customer satisfaction, business services',
        ];
    }

    /**
     * Translate text based on language key
     */
    private function translateText(string $text, string $langKey = 'en'): string
    {
        $translations = [
            'ar' => [
                // Taglines
                'Premium Laundry & Dry Cleaning Services' => 'خدمات غسيل وتنظيف جاف متميزة',
                'Delicious Dining Experience' => 'تجربة طعام لذيذة',
                'Beauty & Wellness Sanctuary' => 'ملاذ الجمال والعافية',
                'Comprehensive Healthcare Solutions' => 'حلول رعاية صحية شاملة',
                'Your Fitness Journey Starts Here' => 'رحلتك الصحية تبدأ هنا',
                'Find Your Dream Property' => 'اعثر على عقارك المثالي',
                'Shop Online with Confidence' => 'تسوق عبر الإنترنت بثقة',
                'Empowering Minds, Building Futures' => 'تمكين العقول، بناء المستقبل',
                'Your Gateway to Amazing Adventures' => 'بوابتك لمغامرات مذهلة',
                'Expert Auto Care & Services' => 'خدمات رعاية سيارات متخصصة',
                'Capturing Moments, Creating Memories' => 'التقاط اللحظات، صنع الذكريات',
                'Creating Unforgettable Events' => 'خلق فعاليات لا تُنسى',
                'Expert Advice, Strategic Solutions' => 'نصائح خبراء، حلول استراتيجية',
                'Innovative Technology Solutions' => 'حلول تقنية مبتكرة',
                'Quality Products, Exceptional Service' => 'منتجات عالية الجودة، خدمة استثنائية',
                'Quality Services You Can Trust' => 'خدمات عالية الجودة يمكنك الوثوق بها',

                // Descriptions
                'Professional laundry and dry cleaning services with free pickup and delivery. Experience the difference with our eco-friendly cleaning solutions and expert garment care.' => 'خدمات غسيل وتنظيف جاف احترافية مع التوصيل والاستلام المجاني. اختبر الفرق مع حلول التنظيف الصديقة للبيئة ورعاية الملابس الخبيرة.',
                'Experience exceptional cuisine crafted with fresh, locally-sourced ingredients. From intimate dinners to family gatherings, we serve memorable meals in a welcoming atmosphere.' => 'استمتع بمأكولات استثنائية مصنوعة من مكونات طازجة محلية المصدر. من العشاء الحميم إلى التجمعات العائلية، نقدم وجبات لا تُنسى في جو مرحب.',
                'Transform your look and rejuvenate your spirit with our premium salon and spa services. Expert stylists, luxurious treatments, and personalized care for your ultimate relaxation.' => 'حوّل مظهرك وجدد روحك مع خدمات الصالون والسبا المتميزة لدينا. خبراء تصفيف، علاجات فاخرة، ورعاية شخصية للاسترخاء المطلق.',
                'Providing compassionate, quality healthcare services with experienced medical professionals. Your health and well-being are our top priorities with personalized treatment plans.' => 'تقديم خدمات رعاية صحية عالية الجودة مع متخصصين طبيين ذوي خبرة. صحتك وعافيتك هي أولوياتنا مع خطط علاج شخصية.',
                'State-of-the-art fitness facility with expert trainers, modern equipment, and motivating group classes. Achieve your fitness goals in a supportive and energizing environment.' => 'منشأة لياقة بدنية حديثة مع مدربين خبراء ومعدات حديثة وصفوف جماعية محفزة. حقق أهدافك الصحية في بيئة داعمة ومنشطة.',
                'Discover your perfect home or investment property with our expert real estate services. Comprehensive listings, market insights, and personalized guidance throughout your property journey.' => 'اكتشف منزلك المثالي أو عقارك الاستثماري مع خدماتنا العقارية الخبيرة. قوائم شاملة، رؤى السوق، وإرشادات شخصية طوال رحلة عقارك.',
                'Your trusted online shopping destination offering quality products, secure payments, and fast delivery. Browse our extensive catalog and enjoy a seamless shopping experience.' => 'وجهتك الموثوقة للتسوق عبر الإنترنت التي تقدم منتجات عالية الجودة ومدفوعات آمنة وتوصيل سريع. تصفح كتالوجنا الواسع واستمتع بتجربة تسوق سلسة.',
                'Quality education and professional training programs designed to unlock your potential. Expert instructors, comprehensive curriculum, and flexible learning options for your success.' => 'برامج تعليم وتدريب مهني عالية الجودة مصممة لإطلاق إمكاناتك. مدربون خبراء، منهج شامل، وخيارات تعلم مرنة لنجاحك.',
                'Create unforgettable memories with our curated travel experiences and tourism services. From exotic destinations to local tours, we make your travel dreams come true.' => 'اصنع ذكريات لا تُنسى مع تجارب السفر والسياحة المنسقة لدينا. من الوجهات الغريبة إلى الجولات المحلية، نحقق أحلام سفرك.',
                'Professional automotive services and maintenance you can trust. From routine maintenance to complex repairs, our certified technicians keep your vehicle running smoothly.' => 'خدمات وصيانة سيارات احترافية يمكنك الوثوق بها. من الصيانة الروتينية إلى الإصلاحات المعقدة، يحافظ فنيونا المعتمدون على سيارتك تعمل بسلاسة.',
                'Professional photography and media services to capture your special moments. Creative vision, technical expertise, and stunning results for every occasion.' => 'خدمات تصوير ووسائط احترافية لالتقاط لحظاتك الخاصة. رؤية إبداعية، خبرة فنية، ونتائج مذهلة لكل مناسبة.',
                'Professional event planning and management services to bring your vision to life. From intimate gatherings to large-scale events, we handle every detail with expertise and creativity.' => 'خدمات تخطيط وإدارة الفعاليات الاحترافية لتحقيق رؤيتك. من التجمعات الحميمة إلى الفعاليات واسعة النطاق، نتعامل مع كل التفاصيل بخبرة وإبداع.',
                'Strategic consulting and professional services to help your business thrive. Expert guidance, proven methodologies, and customized solutions for your unique challenges.' => 'خدمات استشارية استراتيجية ومهنية لمساعدة عملك على الازدهار. إرشادات خبيرة، منهجيات مثبتة، وحلول مخصصة لتحدياتك الفريدة.',
                'Cutting-edge technology and IT services to power your digital transformation. Expert solutions, reliable support, and innovative approaches for your business success.' => 'تقنية وخدمات تكنولوجيا معلومات متطورة لتشغيل تحولك الرقمي. حلول خبيرة، دعم موثوق، ونهج مبتكرة لنجاح عملك.',
                'Your favorite shopping destination offering quality products and personalized service. Discover curated collections, exclusive deals, and a shopping experience you\'ll love.' => 'وجهة التسوق المفضلة لديك التي تقدم منتجات عالية الجودة وخدمة شخصية. اكتشف المجموعات المنسقة والعروض الحصرية وتجربة تسوق ستحبها.',
                'We provide exceptional services tailored to meet your needs. Experience professionalism, quality, and reliability with every interaction.' => 'نقدم خدمات استثنائية مصممة خصيصاً لتلبية احتياجاتك. اختبر الاحترافية والجودة والموثوقية مع كل تفاعل.',

                // Address
                '123 Main Street, City, State 12345' => '١٢٣ شارع الرئيسي، المدينة، الولاية ١٢٣٤٥',

                // Hours
                'Mon - Sat: 9:00 AM - 6:00 PM | Sun: Closed' => 'الاثنين - السبت: 9:00 صباحاً - 6:00 مساءً | الأحد: مغلق',
            ],
        ];

        // Return translated text if available, otherwise return original
        return $translations[$langKey][$text] ?? $text;
    }

}
