import { nextTick } from 'vue';
import { createI18n } from 'vue-i18n';

// Track which languages have been fully loaded
const loadedLanguages = new Set();

// Essential translation groups needed for initial render (login page, common UI)
const ESSENTIAL_GROUPS = ['common', 'menu', 'staff_member', 'messages'];

// LocalStorage keys
const STORAGE_KEY_TRANSLATIONS = 'app_translations';
const STORAGE_KEY_LAST_UPDATED = 'app_translations_last_updated';

// Flag to prevent multiple background loads
let isBackgroundLoadingInProgress = false;

export function setupI18n(options = { locale: 'en', warnHtmlMessage: false }) {
    const i18n = createI18n(options);
    setI18nLanguage(i18n, options.locale);
    return i18n;
}

export function setI18nLanguage(i18n, locale) {
    if (i18n.mode === 'legacy') {
        i18n.global.locale = locale;
    } else {
        i18n.global.locale.value = locale;
    }
    document.querySelector('html').setAttribute('lang', locale);
}

/**
 * Get cached translations from localStorage
 */
function getCachedTranslations() {
    try {
        const cached = localStorage.getItem(STORAGE_KEY_TRANSLATIONS);
        return cached ? JSON.parse(cached) : null;
    } catch (e) {
        console.warn('Failed to parse cached translations:', e);
        return null;
    }
}

/**
 * Save translations to localStorage
 */
function saveCachedTranslations(translations, lastUpdated) {
    try {
        localStorage.setItem(STORAGE_KEY_TRANSLATIONS, JSON.stringify(translations));
        if (lastUpdated) {
            localStorage.setItem(STORAGE_KEY_LAST_UPDATED, lastUpdated);
        }
    } catch (e) {
        console.warn('Failed to cache translations:', e);
    }
}

/**
 * Get cached last updated timestamp
 */
function getCachedLastUpdated() {
    return localStorage.getItem(STORAGE_KEY_LAST_UPDATED);
}

/**
 * Merge new translations into existing ones
 */
function mergeTranslations(existing, newTranslations) {
    const merged = { ...existing };
    Object.keys(newTranslations).forEach(locale => {
        if (!merged[locale]) {
            merged[locale] = {};
        }
        Object.keys(newTranslations[locale]).forEach(group => {
            merged[locale][group] = {
                ...(merged[locale][group] || {}),
                ...newTranslations[locale][group]
            };
        });
    });
    return merged;
}

/**
 * Parse API response into messages format
 */
function parseTranslationsResponse(data) {
    const messages = {};
    data.forEach((lang) => {
        messages[lang.key] = {};
        lang.translations.forEach((translation) => {
            if (messages[lang.key][translation.group] === undefined) {
                messages[lang.key][translation.group] = {};
            }
            messages[lang.key][translation.group][translation.key] = translation.value;
        });
    });
    return messages;
}

/**
 * Apply translations to i18n instance
 */
function applyTranslationsToI18n(i18n, messages, locale) {
    Object.keys(messages).forEach(langKey => {
        const existingMessages = i18n.global.getLocaleMessage(langKey) || {};
        const mergedMessages = {
            ...existingMessages,
            ...messages[langKey]
        };
        i18n.global.setLocaleMessage(langKey, mergedMessages);
    });

    // Ensure the current locale is set
    setI18nLanguage(i18n, locale);
}

/**
 * Check if essential groups exist in cached translations for a locale
 */
function hasEssentialGroupsInCache(cached, locale) {
    if (!cached || !cached[locale]) return false;
    return ESSENTIAL_GROUPS.every(group => cached[locale][group]);
}

/**
 * Load essential translations for fast initial render
 * Only loads common, menu, staff_member, messages groups
 */
export async function loadEssentialTranslations(i18n, locale) {
    // Check if we have cached translations with essential groups
    const cached = getCachedTranslations();

    if (cached && hasEssentialGroupsInCache(cached, locale)) {
        // Apply cached translations immediately for fast render
        applyTranslationsToI18n(i18n, { [locale]: cached[locale] }, locale);
        return nextTick();
    }

    // Fetch essential groups only from server
    try {
        const res = await axiosBase.get(`lang-trans?lang=${locale}&groups=${ESSENTIAL_GROUPS.join(',')}`);
        const messages = parseTranslationsResponse(res.data.data);

        // Merge with cached and save
        const updatedCache = mergeTranslations(cached || {}, messages);
        saveCachedTranslations(updatedCache, res.data.last_updated);

        // Apply to i18n
        applyTranslationsToI18n(i18n, messages, locale);
    } catch (error) {
        console.error('Failed to load essential translations:', error);
        // If we have any cache, use it as fallback
        if (cached && cached[locale]) {
            applyTranslationsToI18n(i18n, { [locale]: cached[locale] }, locale);
        }
    }

    return nextTick();
}

/**
 * Load all translations in background (single request)
 * This fetches everything and updates the cache
 */
export async function loadAllTranslationsInBackground(i18n, locale) {
    // Prevent multiple simultaneous background loads
    if (isBackgroundLoadingInProgress) {
        return;
    }

    isBackgroundLoadingInProgress = true;

    try {
        // Single request to fetch all translations for all languages
        const res = await axiosBase.get('lang-trans');
        const messages = parseTranslationsResponse(res.data.data);

        // Save to cache
        saveCachedTranslations(messages, res.data.last_updated);

        // Apply all languages to i18n
        Object.keys(messages).forEach(langKey => {
            const existingMessages = i18n.global.getLocaleMessage(langKey) || {};
            i18n.global.setLocaleMessage(langKey, {
                ...existingMessages,
                ...messages[langKey]
            });
            loadedLanguages.add(langKey);
        });
    } catch (error) {
        console.error('Failed to load translations in background:', error);
    } finally {
        isBackgroundLoadingInProgress = false;
    }
}

/**
 * Clear translation cache (useful for debugging or after language changes)
 */
export function clearTranslationCache() {
    localStorage.removeItem(STORAGE_KEY_TRANSLATIONS);
    localStorage.removeItem(STORAGE_KEY_LAST_UPDATED);
    loadedLanguages.clear();
}

// ============================================
// Main API functions
// ============================================

/**
 * Load initial translations - loads only essential groups for fast initial render
 * Use this at app startup
 */
export async function loadInitialTranslations(i18n, locale) {
    return loadEssentialTranslations(i18n, locale);
}

/**
 * Load locale messages - loads all translations for a specific locale
 * Used when switching languages to ensure all translations are available
 */
export async function loadLocaleMessages(i18n, locale) {
    // Check cache first
    const cached = getCachedTranslations();

    // If we have full translations in cache, use them
    if (cached && cached[locale] && loadedLanguages.has(locale)) {
        applyTranslationsToI18n(i18n, { [locale]: cached[locale] }, locale);
        return nextTick();
    }

    // Fetch all translations for this specific language
    try {
        const res = await axiosBase.get(`lang-trans?lang=${locale}`);
        const messages = parseTranslationsResponse(res.data.data);

        // Merge with cached and save
        const updatedCache = mergeTranslations(cached || {}, messages);
        saveCachedTranslations(updatedCache, res.data.last_updated);

        // Apply to i18n
        applyTranslationsToI18n(i18n, messages, locale);
        loadedLanguages.add(locale);
    } catch (error) {
        console.error('Failed to load translations for locale:', locale, error);
        // Fallback to cache if available
        if (cached && cached[locale]) {
            applyTranslationsToI18n(i18n, { [locale]: cached[locale] }, locale);
        }
    }

    return nextTick();
}

/**
 * Load remaining languages in the background (call this after app is mounted)
 * Now makes only ONE request to fetch all translations
 */
export async function loadRemainingLanguages(i18n) {
    const locale = i18n.global.locale.value || i18n.global.locale;
    await loadAllTranslationsInBackground(i18n, locale);
}

export function setLangsLocaleMessage(res, i18n, locale) {
    const messages = parseTranslationsResponse(res.data.data);
    applyTranslationsToI18n(i18n, messages, locale);
    return nextTick();
}
