// Country codes with dial codes, names, primary language codes, and native language names
export const countryCodes = [
    { code: "US", dialCode: "+1", name: "United States", langCode: "en", langName: "English", isRtl: false },
    { code: "GB", dialCode: "+44", name: "United Kingdom", langCode: "en", langName: "English", isRtl: false },
    { code: "IN", dialCode: "+91", name: "India", langCode: "hi", langName: "हिंदी", isRtl: false },
    { code: "AU", dialCode: "+61", name: "Australia", langCode: "en", langName: "English", isRtl: false },
    { code: "CA", dialCode: "+1", name: "Canada", langCode: "en", langName: "English", isRtl: false },
    { code: "DE", dialCode: "+49", name: "Germany", langCode: "de", langName: "Deutsch", isRtl: false },
    { code: "FR", dialCode: "+33", name: "France", langCode: "fr", langName: "Français", isRtl: false },
    { code: "IT", dialCode: "+39", name: "Italy", langCode: "it", langName: "Italiano", isRtl: false },
    { code: "ES", dialCode: "+34", name: "Spain", langCode: "es", langName: "Español", isRtl: false },
    { code: "NL", dialCode: "+31", name: "Netherlands", langCode: "nl", langName: "Nederlands", isRtl: false },
    { code: "BE", dialCode: "+32", name: "Belgium", langCode: "nl", langName: "Nederlands", isRtl: false },
    { code: "CH", dialCode: "+41", name: "Switzerland", langCode: "de", langName: "Deutsch", isRtl: false },
    { code: "AT", dialCode: "+43", name: "Austria", langCode: "de", langName: "Deutsch", isRtl: false },
    { code: "SE", dialCode: "+46", name: "Sweden", langCode: "sv", langName: "Svenska", isRtl: false },
    { code: "NO", dialCode: "+47", name: "Norway", langCode: "no", langName: "Norsk", isRtl: false },
    { code: "DK", dialCode: "+45", name: "Denmark", langCode: "da", langName: "Dansk", isRtl: false },
    { code: "FI", dialCode: "+358", name: "Finland", langCode: "fi", langName: "Suomi", isRtl: false },
    { code: "PL", dialCode: "+48", name: "Poland", langCode: "pl", langName: "Polski", isRtl: false },
    { code: "PT", dialCode: "+351", name: "Portugal", langCode: "pt", langName: "Português", isRtl: false },
    { code: "IE", dialCode: "+353", name: "Ireland", langCode: "en", langName: "English", isRtl: false },
    { code: "NZ", dialCode: "+64", name: "New Zealand", langCode: "en", langName: "English", isRtl: false },
    { code: "SG", dialCode: "+65", name: "Singapore", langCode: "en", langName: "English", isRtl: false },
    { code: "HK", dialCode: "+852", name: "Hong Kong", langCode: "zh", langName: "中文", isRtl: false },
    { code: "JP", dialCode: "+81", name: "Japan", langCode: "ja", langName: "日本語", isRtl: false },
    { code: "KR", dialCode: "+82", name: "South Korea", langCode: "ko", langName: "한국어", isRtl: false },
    { code: "CN", dialCode: "+86", name: "China", langCode: "zh", langName: "中文", isRtl: false },
    { code: "TW", dialCode: "+886", name: "Taiwan", langCode: "zh", langName: "中文", isRtl: false },
    { code: "TH", dialCode: "+66", name: "Thailand", langCode: "th", langName: "ไทย", isRtl: false },
    { code: "MY", dialCode: "+60", name: "Malaysia", langCode: "ms", langName: "Bahasa Melayu", isRtl: false },
    { code: "ID", dialCode: "+62", name: "Indonesia", langCode: "id", langName: "Bahasa Indonesia", isRtl: false },
    { code: "PH", dialCode: "+63", name: "Philippines", langCode: "tl", langName: "Tagalog", isRtl: false },
    { code: "VN", dialCode: "+84", name: "Vietnam", langCode: "vi", langName: "Tiếng Việt", isRtl: false },
    { code: "AE", dialCode: "+971", name: "UAE", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "SA", dialCode: "+966", name: "Saudi Arabia", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "QA", dialCode: "+974", name: "Qatar", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "KW", dialCode: "+965", name: "Kuwait", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "BH", dialCode: "+973", name: "Bahrain", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "OM", dialCode: "+968", name: "Oman", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "EG", dialCode: "+20", name: "Egypt", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "ZA", dialCode: "+27", name: "South Africa", langCode: "en", langName: "English", isRtl: false },
    { code: "NG", dialCode: "+234", name: "Nigeria", langCode: "en", langName: "English", isRtl: false },
    { code: "KE", dialCode: "+254", name: "Kenya", langCode: "sw", langName: "Kiswahili", isRtl: false },
    { code: "GH", dialCode: "+233", name: "Ghana", langCode: "en", langName: "English", isRtl: false },
    { code: "BR", dialCode: "+55", name: "Brazil", langCode: "pt", langName: "Português", isRtl: false },
    { code: "MX", dialCode: "+52", name: "Mexico", langCode: "es", langName: "Español", isRtl: false },
    { code: "AR", dialCode: "+54", name: "Argentina", langCode: "es", langName: "Español", isRtl: false },
    { code: "CL", dialCode: "+56", name: "Chile", langCode: "es", langName: "Español", isRtl: false },
    { code: "CO", dialCode: "+57", name: "Colombia", langCode: "es", langName: "Español", isRtl: false },
    { code: "PE", dialCode: "+51", name: "Peru", langCode: "es", langName: "Español", isRtl: false },
    { code: "RU", dialCode: "+7", name: "Russia", langCode: "ru", langName: "Русский", isRtl: false },
    { code: "UA", dialCode: "+380", name: "Ukraine", langCode: "uk", langName: "Українська", isRtl: false },
    { code: "TR", dialCode: "+90", name: "Turkey", langCode: "tr", langName: "Türkçe", isRtl: false },
    { code: "IL", dialCode: "+972", name: "Israel", langCode: "he", langName: "עברית", isRtl: true },
    { code: "PK", dialCode: "+92", name: "Pakistan", langCode: "ur", langName: "اردو", isRtl: true },
    { code: "BD", dialCode: "+880", name: "Bangladesh", langCode: "bn", langName: "বাংলা", isRtl: false },
    { code: "LK", dialCode: "+94", name: "Sri Lanka", langCode: "si", langName: "සිංහල", isRtl: false },
    { code: "NP", dialCode: "+977", name: "Nepal", langCode: "ne", langName: "नेपाली", isRtl: false },
    { code: "GR", dialCode: "+30", name: "Greece", langCode: "el", langName: "Ελληνικά", isRtl: false },
    { code: "CZ", dialCode: "+420", name: "Czech Republic", langCode: "cs", langName: "Čeština", isRtl: false },
    { code: "HU", dialCode: "+36", name: "Hungary", langCode: "hu", langName: "Magyar", isRtl: false },
    { code: "RO", dialCode: "+40", name: "Romania", langCode: "ro", langName: "Română", isRtl: false },
    { code: "BG", dialCode: "+359", name: "Bulgaria", langCode: "bg", langName: "Български", isRtl: false },
    { code: "HR", dialCode: "+385", name: "Croatia", langCode: "hr", langName: "Hrvatski", isRtl: false },
    { code: "SK", dialCode: "+421", name: "Slovakia", langCode: "sk", langName: "Slovenčina", isRtl: false },
    { code: "SI", dialCode: "+386", name: "Slovenia", langCode: "sl", langName: "Slovenščina", isRtl: false },
    { code: "RS", dialCode: "+381", name: "Serbia", langCode: "sr", langName: "Српски", isRtl: false },
    { code: "LT", dialCode: "+370", name: "Lithuania", langCode: "lt", langName: "Lietuvių", isRtl: false },
    { code: "LV", dialCode: "+371", name: "Latvia", langCode: "lv", langName: "Latviešu", isRtl: false },
    { code: "EE", dialCode: "+372", name: "Estonia", langCode: "et", langName: "Eesti", isRtl: false },
    { code: "IR", dialCode: "+98", name: "Iran", langCode: "fa", langName: "فارسی", isRtl: true },
    { code: "IQ", dialCode: "+964", name: "Iraq", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "AF", dialCode: "+93", name: "Afghanistan", langCode: "ps", langName: "پښتو", isRtl: true },
    { code: "MA", dialCode: "+212", name: "Morocco", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "DZ", dialCode: "+213", name: "Algeria", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "TN", dialCode: "+216", name: "Tunisia", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "LY", dialCode: "+218", name: "Libya", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "JO", dialCode: "+962", name: "Jordan", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "LB", dialCode: "+961", name: "Lebanon", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "SY", dialCode: "+963", name: "Syria", langCode: "ar", langName: "العربية", isRtl: true },
    { code: "YE", dialCode: "+967", name: "Yemen", langCode: "ar", langName: "العربية", isRtl: true },
];

// Convert country code to flag emoji
export const getFlagEmoji = (countryCode) => {
    if (!countryCode) return "";
    const codePoints = countryCode
        .toUpperCase()
        .split("")
        .map((char) => 127397 + char.charCodeAt());
    return String.fromCodePoint(...codePoints);
};

// Timezone to country mapping for auto-detection
export const timezoneToCountry = {
    "Asia/Kolkata": "IN",
    "Asia/Calcutta": "IN",
    "America/New_York": "US",
    "America/Los_Angeles": "US",
    "America/Chicago": "US",
    "America/Denver": "US",
    "Europe/London": "GB",
    "Europe/Paris": "FR",
    "Europe/Berlin": "DE",
    "Europe/Rome": "IT",
    "Europe/Madrid": "ES",
    "Europe/Amsterdam": "NL",
    "Europe/Brussels": "BE",
    "Europe/Zurich": "CH",
    "Europe/Vienna": "AT",
    "Europe/Stockholm": "SE",
    "Europe/Oslo": "NO",
    "Europe/Copenhagen": "DK",
    "Europe/Helsinki": "FI",
    "Europe/Warsaw": "PL",
    "Europe/Lisbon": "PT",
    "Europe/Dublin": "IE",
    "Australia/Sydney": "AU",
    "Australia/Melbourne": "AU",
    "Australia/Perth": "AU",
    "Pacific/Auckland": "NZ",
    "Asia/Singapore": "SG",
    "Asia/Hong_Kong": "HK",
    "Asia/Tokyo": "JP",
    "Asia/Seoul": "KR",
    "Asia/Shanghai": "CN",
    "Asia/Taipei": "TW",
    "Asia/Bangkok": "TH",
    "Asia/Kuala_Lumpur": "MY",
    "Asia/Jakarta": "ID",
    "Asia/Manila": "PH",
    "Asia/Ho_Chi_Minh": "VN",
    "Asia/Dubai": "AE",
    "Asia/Riyadh": "SA",
    "Asia/Qatar": "QA",
    "Asia/Kuwait": "KW",
    "Asia/Bahrain": "BH",
    "Asia/Muscat": "OM",
    "Africa/Cairo": "EG",
    "Africa/Johannesburg": "ZA",
    "Africa/Lagos": "NG",
    "Africa/Nairobi": "KE",
    "Africa/Accra": "GH",
    "America/Sao_Paulo": "BR",
    "America/Mexico_City": "MX",
    "America/Argentina/Buenos_Aires": "AR",
    "America/Santiago": "CL",
    "America/Bogota": "CO",
    "America/Lima": "PE",
    "Europe/Moscow": "RU",
    "Europe/Kiev": "UA",
    "Europe/Istanbul": "TR",
    "Asia/Jerusalem": "IL",
    "Asia/Karachi": "PK",
    "Asia/Dhaka": "BD",
    "Asia/Colombo": "LK",
    "Asia/Kathmandu": "NP",
    "America/Toronto": "CA",
    "America/Vancouver": "CA",
};

// Detect user's country from browser locale/timezone
export const detectUserCountry = () => {
    try {
        const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
        if (timezone && timezoneToCountry[timezone]) {
            return timezoneToCountry[timezone];
        }

        // Fallback: try to get from browser language
        const language = navigator.language || navigator.userLanguage;
        if (language) {
            const parts = language.split("-");
            if (parts.length > 1) {
                const countryCode = parts[1].toUpperCase();
                const country = countryCodes.find((c) => c.code === countryCode);
                if (country) {
                    return countryCode;
                }
            }
        }
    } catch (e) {
        // Fallback to US if detection fails
    }
    return "US";
};

// Get dial code from country code
export const getDialCodeFromCountry = (countryCode) => {
    const country = countryCodes.find((c) => c.code === countryCode);
    return country ? country.dialCode : "+1";
};

// Get language code from country code
export const getLangCodeFromCountry = (countryCode) => {
    const country = countryCodes.find((c) => c.code === countryCode);
    return country ? country.langCode : "en";
};

// Get full language info from country code
export const getLanguageInfoFromCountry = (countryCode) => {
    const country = countryCodes.find((c) => c.code === countryCode);
    if (country) {
        return {
            name: country.langName,
            key: country.langCode,
            flag_code: country.code,
            is_rtl: country.isRtl,
        };
    }
    return {
        name: "English",
        key: "en",
        flag_code: "US",
        is_rtl: false,
    };
};

// Get default language entry based on user's browser locale/location
export const getLocaleBasedDefaultLanguage = () => {
    const userCountry = detectUserCountry();
    const langInfo = getLanguageInfoFromCountry(userCountry);
    return {
        name: langInfo.name,
        key: langInfo.key,
        flag_code: langInfo.flag_code,
        is_rtl: langInfo.is_rtl,
        enabled: true,
        is_default: true,
    };
};

// Filter option for ant-design select
export const filterCountryOption = (input, option) => {
    const searchTerm = input.toLowerCase();
    const label = option.label || "";
    const filterLabel = option.filterLabel || "";
    return (
        label.toLowerCase().includes(searchTerm) ||
        filterLabel.toLowerCase().includes(searchTerm)
    );
};
