/**
 * Modules Menu Store (Pinia)
 *
 * Manages module menu items, loading states, and dynamic module loading.
 */

import { defineStore } from 'pinia';
import { ref } from 'vue';
import { loadModuleBundle } from '../utils/moduleLoader';

export const useModulesMenuStore = defineStore('modulesMenu', () => {
    const menus = ref([]);
    const loading = ref(false);
    const error = ref(null);
    const loadedModules = ref(new Set());
    const hasFetchedMenus = ref(false);

    /**
     * Fetch module menu from server API.
     * @param {boolean} force - Force refetch even if already fetched
     */
    const fetchMenus = async (force = false) => {
        // Don't fetch if already fetched (unless forced)
        if (!force && hasFetchedMenus.value) {
            return;
        }

        // Don't fetch if already loading
        if (loading.value) {
            return;
        }

        loading.value = true;
        error.value = null;

        try {
            // Use global axiosAdmin instance
            const axios = window.axiosAdmin;

            if (!axios) {
                throw new Error('axiosAdmin not initialized');
            }

            const response = await axios.get('modules/menu');

            if (response.success) {
                menus.value = response.data;
                hasFetchedMenus.value = true;
            } else {
                throw new Error('Failed to load menus');
            }
        } catch (err) {
            error.value = err.message || 'Unknown error';
        } finally {
            loading.value = false;
        }
    };

    /**
     * Load a module bundle dynamically and register its routes.
     *
     * @param {Object} menuItem - Menu item with assets
     * @returns {Promise<void>}
     */
    const loadModule = async (menuItem) => {
        const moduleName = menuItem.module;

        // Already loaded?
        if (loadedModules.value.has(moduleName)) {
            return;
        }

        // Check status
        if (menuItem.status !== 'enabled') {
            throw new Error(`Module "${moduleName}" is not enabled (status: ${menuItem.status})`);
        }

        // Load bundle (injects scripts/css and waits for registration)
        await loadModuleBundle(moduleName, menuItem.assets);

        // Get module API from registry
        const moduleApi = window.__registeredModules[moduleName]?.api;

        if (!moduleApi) {
            throw new Error(`Module "${moduleName}" did not register properly`);
        }

        // Add routes to router as children of /admin route
        if (moduleApi.routes && Array.isArray(moduleApi.routes)) {
            const router = window.__appRouter;

            moduleApi.routes.forEach(route => {
                // Convert absolute paths to relative paths for nested routing
                const routeCopy = { ...route };

                // If route starts with /admin/, convert to relative path
                if (routeCopy.path.startsWith('/admin/')) {
                    routeCopy.path = routeCopy.path.replace(/^\/admin\//, '');
                }

                // Ensure route has required meta properties
                routeCopy.meta = {
                    requireAuth: true, // Module routes require authentication by default
                    menuParent: moduleName,
                    ...route.meta, // Allow route to override defaults
                };

                // Add as child of 'admin' parent route
                router.addRoute('admin', routeCopy);
            });
        }

        // Call module init hook
        if (typeof moduleApi.init === 'function') {
            moduleApi.init(window.__app, window.__appRouter);
        }

        // Mark as loaded
        loadedModules.value.add(moduleName);
    };

    /**
     * Clear menu cache on server.
     */
    const clearCache = async () => {
        try {
            const axios = window.axiosAdmin;

            if (!axios) {
                throw new Error('axiosAdmin not initialized');
            }

            await axios.post('modules/menu/clear-cache');
            await fetchMenus(true);
        } catch (err) {
        }
    };

    return {
        menus,
        loading,
        error,
        loadedModules,
        hasFetchedMenus,
        fetchMenus,
        loadModule,
        clearCache,
    };
});
