/**
 * Module Loader Utilities
 *
 * Provides functions to dynamically load module JS/CSS assets and wait for registration.
 */

/**
 * Load a script tag into the document.
 *
 * @param {string} src - Script URL
 * @returns {Promise<void>}
 */
export function loadScript(src) {
    return new Promise((resolve, reject) => {
        // Check if already loaded
        if (document.querySelector(`script[src="${src}"]`)) {
            return resolve();
        }

        const script = document.createElement('script');
        // Modules are built as IIFE, not ES modules, so no type='module'
        script.src = src;
        script.onload = () => {
            resolve();
        };
        script.onerror = () => {
            reject(new Error(`Failed to load script: ${src}`));
        };

        document.head.appendChild(script);
    });
}

/**
 * Load a CSS link into the document.
 *
 * @param {string} href - CSS URL
 * @returns {Promise<void>}
 */
export function loadCss(href) {
    return new Promise((resolve, reject) => {
        // Check if already loaded
        if (document.querySelector(`link[href="${href}"]`)) {
            return resolve();
        }

        const link = document.createElement('link');
        link.rel = 'stylesheet';
        link.href = href;
        link.onload = () => {
            resolve();
        };
        link.onerror = () => {
            reject(new Error(`Failed to load CSS: ${href}`));
        };

        document.head.appendChild(link);
    });
}

/**
 * Load a module bundle (JS + CSS) and wait for registration.
 *
 * @param {string} moduleName - Module name
 * @param {Object} assets - { js: [...], css: [...] }
 * @param {number} timeout - Timeout in ms (default 10s)
 * @returns {Promise<Object>} Module API
 */
export async function loadModuleBundle(moduleName, assets, timeout = 10000) {
    try {

        // Only load assets manually when not using Vite dev server
        if (!window.config?.vite_hot_file_exists) {
            // Load CSS first (non-blocking)
            if (assets.css && assets.css.length > 0) {
                await Promise.all(assets.css.map(href => loadCss(href)));
            }

            // Load JS scripts in order
            if (assets.js && assets.js.length > 0) {
                for (const src of assets.js) {
                    await loadScript(src);
                }
            }
        }

        // Wait for module to call window.registerModule
        const moduleApi = await window.waitForModuleRegistration(moduleName, timeout);

        return moduleApi;

    } catch (error) {
        throw error;
    }
}
