<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Default awards
        $defaultItems = [
            [
                'title' => 'Best Service Award',
                'organization' => 'Industry Excellence',
                'year' => '2024',
                'description' => 'Recognized for outstanding customer service',
                'icon' => 'heroicons:trophy',
            ],
            [
                'title' => 'Innovation Leader',
                'organization' => 'Tech Awards',
                'year' => '2024',
                'description' => 'Leading innovation in our industry',
                'icon' => 'heroicons:light-bulb',
            ],
            [
                'title' => 'Customer Choice',
                'organization' => 'Consumer Reports',
                'year' => '2023',
                'description' => 'Voted #1 by customers nationwide',
                'icon' => 'heroicons:heart',
            ],
            [
                'title' => 'Green Business',
                'organization' => 'Environmental Board',
                'year' => '2023',
                'description' => 'Commitment to sustainability',
                'icon' => 'heroicons:globe-alt',
            ],
        ];

        $this->items = !empty($items) ? $items : $defaultItems;

        $this->section = array_merge([
            'title' => 'Awards & Recognition',
            'subtitle' => 'Our achievements speak for themselves',
            'badge' => 'Awards',
        ], $section);
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="awards" class="section {{ $isWhiteBackground ? 'bg-white dark:bg-gray-800' : 'bg-gray-50 dark:bg-gray-900' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <!-- Section Header -->
        <div class="text-center mb-16 scroll-animate" data-animation="fade-up">
            @if(!empty($section['badge']))
            <span class="badge badge-primary mb-4">
                {{ $section['badge'] }}
            </span>
            @endif

            <h2 class="section-title">
                {{ $section['title'] }}
            </h2>

            @if(!empty($section['subtitle']))
            <p class="section-subtitle">
                {{ $section['subtitle'] }}
            </p>
            @endif
        </div>

        <!-- Awards Grid -->
        @if(count($items) > 0)
        <div class="grid sm:grid-cols-2 lg:grid-cols-4 gap-6">
            @foreach($items as $index => $award)
            <div class="award-card relative bg-white dark:bg-gray-700 rounded-xl p-6 shadow-lg border border-gray-100 dark:border-gray-600 text-center scroll-animate" data-animation="fade-up" style="animation-delay: {{ $index * 100 }}ms">
                <div class="award-ribbon"></div>

                @if(!empty($award['year']))
                <span class="award-year">{{ $award['year'] }}</span>
                @endif

                <div class="award-icon">
                    @php
                        $imageUrl = $award['image_url'] ?? $award['image'] ?? null;
                    @endphp
                    @if($imageUrl)
                    <img src="{{ $imageUrl }}" alt="{{ $award['title'] ?? '' }}" class="w-16 h-16 object-contain" loading="lazy">
                    @elseif(!empty($award['icon']))
                    <x-icon icon="{{ $award['icon'] }}" class="w-10 h-10 text-white" />
                    @else
                    <svg class="w-10 h-10 text-white" fill="currentColor" viewBox="0 0 24 24">
                        <path d="M12 2L15.09 8.26L22 9.27L17 14.14L18.18 21.02L12 17.77L5.82 21.02L7 14.14L2 9.27L8.91 8.26L12 2Z"/>
                    </svg>
                    @endif
                </div>

                <h3 class="text-lg font-bold text-gray-900 dark:text-white mb-2">
                    {{ $award['title'] ?? '' }}
                </h3>

                @if(!empty($award['organization']))
                <p class="text-primary-600 dark:text-primary-400 text-sm font-medium mb-2">
                    {{ $award['organization'] }}
                </p>
                @endif

                @if(!empty($award['description']))
                <p class="text-gray-600 dark:text-gray-300 text-sm">
                    {{ $award['description'] }}
                </p>
                @endif
            </div>
            @endforeach
        </div>
        @endif
    </div>
</section>

<style>
    .award-card {
        transition: transform 0.3s ease, box-shadow 0.3s ease;
    }
    .award-card:hover {
        transform: translateY(-8px) scale(1.02);
        box-shadow: 0 20px 40px -15px rgba(0, 0, 0, 0.2);
    }
    .award-icon {
        width: 80px;
        height: 80px;
        background: linear-gradient(135deg, #fbbf24, #f59e0b);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 1.5rem;
        box-shadow: 0 10px 30px -10px rgba(245, 158, 11, 0.5);
    }
    .award-year {
        position: absolute;
        top: 1rem;
        right: 1rem;
        background: var(--color-primary-500);
        color: white;
        font-size: 0.75rem;
        font-weight: 600;
        padding: 0.25rem 0.75rem;
        border-radius: 9999px;
    }
    [dir="rtl"] .award-year {
        right: auto;
        left: 1rem;
    }
    .award-ribbon {
        position: absolute;
        top: -5px;
        left: 50%;
        transform: translateX(-50%);
        width: 30px;
        height: 50px;
        background: linear-gradient(135deg, #fbbf24, #f59e0b);
        clip-path: polygon(0 0, 100% 0, 100% 100%, 50% 80%, 0 100%);
    }
</style>
</div>
