<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->section = $section;
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->items = !empty($items) ? $items : [
            ['icon' => 'heroicons:leaf', 'title' => 'Eco-Friendly', 'description' => '100% biodegradable detergents.'],
            ['icon' => 'heroicons:clock', 'title' => 'On-Time Delivery', 'description' => 'Guaranteed delivery within promised timeframe.'],
            ['icon' => 'heroicons:shield-check', 'title' => 'Fully Insured', 'description' => 'Your garments are protected.'],
            ['icon' => 'heroicons:heart', 'title' => 'Care & Quality', 'description' => 'Every item treated with attention.'],
        ];
    }

    public function getPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }

    public function getDefaultPaddingClasses(): string
    {
        $classes = [];
        if ($this->paddingTop === null) {
            $classes[] = 'pt-12';
        }
        if ($this->paddingBottom === null) {
            $classes[] = 'pb-12';
        }
        return implode(' ', $classes);
    }
};
?>

<div>
<section class="features-cards-section {{ $isWhiteBackground ? 'features-cards-white' : 'features-cards-gray' }}" @if($this->getPaddingStyle()) style="{{ $this->getPaddingStyle() }}" @endif>
    {{-- Background Decorations --}}
    <div class="features-cards-bg">
        <div class="features-cards-circle features-cards-circle-1"></div>
        <div class="features-cards-circle features-cards-circle-2"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        <div class="features-cards-grid">
            @foreach ($items as $index => $feature)
                <div class="features-cards-item scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    <div class="features-cards-inner">
                        {{-- Gradient Background --}}
                        <div class="features-cards-gradient"></div>

                        {{-- Icon --}}
                        <div class="features-cards-icon-wrapper">
                            <div class="features-cards-icon">
                                <x-icon icon="{{ $feature['icon'] ?? 'heroicons:heart' }}" class="w-8 h-8" />
                            </div>
                            <div class="features-cards-icon-glow"></div>
                        </div>

                        {{-- Content --}}
                        <div class="features-cards-content">
                            <h3 class="features-cards-title">{{ $feature['title'] }}</h3>
                            <p class="features-cards-desc">{{ $feature['description'] }}</p>
                        </div>

                        {{-- Decorative Corner --}}
                        <div class="features-cards-corner"></div>
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Features Cards Section ===== */
    .features-cards-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .features-cards-white {
        background: #ffffff;
    }

    .features-cards-gray {
        background: #f8fafc;
    }

    .dark .features-cards-white {
        background: var(--color-bg-secondary);
    }

    .dark .features-cards-gray {
        background: var(--color-bg-primary);
    }

    /* ===== Background ===== */
    .features-cards-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .features-cards-circle {
        position: absolute;
        border-radius: 50%;
        background: linear-gradient(135deg, var(--color-primary-200), var(--color-secondary-200));
        filter: blur(80px);
        opacity: 0.3;
    }

    .features-cards-circle-1 {
        width: 400px;
        height: 400px;
        top: -100px;
        right: -100px;
    }

    .features-cards-circle-2 {
        width: 500px;
        height: 500px;
        bottom: -150px;
        left: -150px;
    }

    .dark .features-cards-circle {
        background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
        opacity: 0.2;
    }

    /* ===== Grid ===== */
    .features-cards-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 2rem;
    }

    /* ===== Card ===== */
    .features-cards-item {
        position: relative;
    }

    .features-cards-inner {
        position: relative;
        height: 100%;
        background: white;
        border: 1px solid var(--color-gray-200);
        border-radius: 1.5rem;
        padding: 2rem;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        overflow: hidden;
    }

    .features-cards-item:hover .features-cards-inner {
        transform: translateY(-8px);
        border-color: var(--color-primary-300);
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.15);
    }

    .dark .features-cards-inner {
        background: var(--color-card-bg);
        border-color: var(--color-border);
    }

    .dark .features-cards-item:hover .features-cards-inner {
        border-color: var(--color-primary-700);
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.5);
    }

    /* ===== Gradient Background ===== */
    .features-cards-gradient {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 150px;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        opacity: 0;
        transition: opacity 0.4s ease;
        border-radius: 1.5rem 1.5rem 0 0;
    }

    .features-cards-item:hover .features-cards-gradient {
        opacity: 0.1;
    }

    /* ===== Icon ===== */
    .features-cards-icon-wrapper {
        position: relative;
        display: inline-flex;
        margin-bottom: 1.5rem;
    }

    .features-cards-icon {
        position: relative;
        width: 4rem;
        height: 4rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 1rem;
        color: white;
        transition: all 0.4s ease;
        z-index: 1;
    }

    .features-cards-item:hover .features-cards-icon {
        transform: rotate(-5deg) scale(1.1);
    }

    .features-cards-icon-glow {
        position: absolute;
        inset: -8px;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 1.25rem;
        opacity: 0;
        filter: blur(20px);
        transition: opacity 0.4s ease;
    }

    .features-cards-item:hover .features-cards-icon-glow {
        opacity: 0.4;
    }

    /* ===== Content ===== */
    .features-cards-content {
        position: relative;
        z-index: 1;
    }

    .features-cards-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.5rem;
    }

    .dark .features-cards-title {
        color: var(--color-text-primary);
    }

    .features-cards-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
    }

    .dark .features-cards-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Decorative Corner ===== */
    .features-cards-corner {
        position: absolute;
        bottom: 0;
        right: 0;
        width: 100px;
        height: 100px;
        background: linear-gradient(-135deg, transparent 50%, var(--color-primary-50) 50%);
        opacity: 0;
        transition: opacity 0.4s ease;
    }

    .features-cards-item:hover .features-cards-corner {
        opacity: 1;
    }

    .dark .features-cards-corner {
        background: linear-gradient(-135deg, transparent 50%, rgba(var(--color-primary-rgb, 37, 99, 235), 0.1) 50%);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .features-cards-section {
        direction: rtl;
    }

    [dir="rtl"] .features-cards-content {
        text-align: right;
    }

    [dir="rtl"] .features-cards-circle-1 {
        right: auto;
        left: -100px;
    }

    [dir="rtl"] .features-cards-circle-2 {
        left: auto;
        right: -150px;
    }

    [dir="rtl"] .features-cards-corner {
        right: auto;
        left: 0;
        background: linear-gradient(135deg, transparent 50%, var(--color-primary-50) 50%);
    }

    [dir="rtl"] .dark .features-cards-corner {
        background: linear-gradient(135deg, transparent 50%, rgba(var(--color-primary-rgb, 37, 99, 235), 0.1) 50%);
    }

    [dir="rtl"] .features-cards-item:hover .features-cards-icon {
        transform: rotate(5deg) scale(1.1);
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .features-cards-section {
            padding: 4rem 0;
        }

        .features-cards-grid {
            grid-template-columns: 1fr;
            gap: 1.5rem;
        }

        .features-cards-inner {
            padding: 1.5rem;
        }
    }
</style>
</div>
