<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->section = $section;
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->items = !empty($items) ? $items : [
            ['icon' => 'heroicons:leaf', 'title' => 'Eco-Friendly', 'description' => '100% biodegradable detergents.'],
            ['icon' => 'heroicons:clock', 'title' => 'On-Time Delivery', 'description' => 'Guaranteed delivery within promised timeframe.'],
            ['icon' => 'heroicons:shield-check', 'title' => 'Fully Insured', 'description' => 'Your garments are protected.'],
            ['icon' => 'heroicons:heart', 'title' => 'Care & Quality', 'description' => 'Every item treated with attention.'],
        ];
    }

    public function getPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }

    public function getDefaultPaddingClasses(): string
    {
        $classes = [];
        if ($this->paddingTop === null) {
            $classes[] = 'pt-12';
        }
        if ($this->paddingBottom === null) {
            $classes[] = 'pb-12';
        }
        return implode(' ', $classes);
    }
};
?>

<div>
<section class="features-grid-section {{ $isWhiteBackground ? 'features-grid-white' : 'features-grid-gray' }}" @if($this->getPaddingStyle()) style="{{ $this->getPaddingStyle() }}" @endif>
    {{-- Background Pattern --}}
    <div class="features-grid-pattern"></div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        <div class="features-grid-container">
            @foreach ($items as $index => $feature)
                <div class="features-grid-item scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Number Badge --}}
                    <div class="features-grid-number">{{ str_pad($index + 1, 2, '0', STR_PAD_LEFT) }}</div>

                    {{-- Icon with Pulse Effect --}}
                    <div class="features-grid-icon-container">
                        <div class="features-grid-pulse"></div>
                        <div class="features-grid-icon">
                            <x-icon icon="{{ $feature['icon'] ?? 'heroicons:heart' }}" class="w-6 h-6" />
                        </div>
                    </div>

                    {{-- Content --}}
                    <div class="features-grid-content">
                        <h3 class="features-grid-title">{{ $feature['title'] }}</h3>
                        <p class="features-grid-desc">{{ $feature['description'] }}</p>
                    </div>

                    {{-- Hover Border --}}
                    <div class="features-grid-border"></div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Features Grid Section ===== */
    .features-grid-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .features-grid-white {
        background: #ffffff;
    }

    .features-grid-gray {
        background: linear-gradient(180deg, #f8fafc 0%, #f1f5f9 100%);
    }

    .dark .features-grid-white {
        background: var(--color-bg-secondary);
    }

    .dark .features-grid-gray {
        background: linear-gradient(180deg, var(--color-bg-primary) 0%, var(--color-bg-secondary) 100%);
    }

    /* ===== Background Pattern ===== */
    .features-grid-pattern {
        position: absolute;
        inset: 0;
        background-image:
            linear-gradient(var(--color-gray-200) 1px, transparent 1px),
            linear-gradient(90deg, var(--color-gray-200) 1px, transparent 1px);
        background-size: 50px 50px;
        opacity: 0.3;
        pointer-events: none;
    }

    .dark .features-grid-pattern {
        background-image:
            linear-gradient(var(--color-gray-800) 1px, transparent 1px),
            linear-gradient(90deg, var(--color-gray-800) 1px, transparent 1px);
        opacity: 0.2;
    }

    /* ===== Container ===== */
    .features-grid-container {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
    }

    /* ===== Item ===== */
    .features-grid-item {
        position: relative;
        background: white;
        padding: 2rem 1.5rem;
        border-radius: 1rem;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        overflow: hidden;
    }

    .features-grid-item:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 40px -10px rgba(0, 0, 0, 0.1);
    }

    .dark .features-grid-item {
        background: var(--color-card-bg);
    }

    .dark .features-grid-item:hover {
        box-shadow: 0 15px 40px -10px rgba(0, 0, 0, 0.4);
    }

    /* ===== Number Badge ===== */
    .features-grid-number {
        position: absolute;
        top: 1rem;
        right: 1rem;
        width: 2.5rem;
        height: 2.5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: var(--color-primary-50);
        color: var(--color-primary-600);
        font-family: var(--font-heading);
        font-size: 0.875rem;
        font-weight: 700;
        border-radius: 0.5rem;
        transition: all 0.3s ease;
    }

    .features-grid-item:hover .features-grid-number {
        background: var(--color-primary-600);
        color: white;
        transform: scale(1.1);
    }

    .dark .features-grid-number {
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.2);
        color: var(--color-primary-400);
    }

    .dark .features-grid-item:hover .features-grid-number {
        background: var(--color-primary-600);
        color: white;
    }

    /* ===== Icon ===== */
    .features-grid-icon-container {
        position: relative;
        display: inline-flex;
        margin-bottom: 1.5rem;
    }

    .features-grid-pulse {
        position: absolute;
        inset: -8px;
        background: var(--color-primary-200);
        border-radius: 50%;
        opacity: 0;
        animation: features-grid-pulse-anim 2s ease-out infinite;
    }

    .dark .features-grid-pulse {
        background: var(--color-primary-900);
    }

    @keyframes features-grid-pulse-anim {
        0% {
            transform: scale(1);
            opacity: 0.5;
        }
        100% {
            transform: scale(1.5);
            opacity: 0;
        }
    }

    .features-grid-icon {
        width: 3.5rem;
        height: 3.5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        color: white;
        border-radius: 50%;
        transition: all 0.4s ease;
        position: relative;
        z-index: 1;
    }

    .features-grid-item:hover .features-grid-icon {
        transform: scale(1.1) rotate(10deg);
    }

    /* ===== Content ===== */
    .features-grid-content {
        position: relative;
        z-index: 1;
    }

    .features-grid-title {
        font-family: var(--font-heading);
        font-size: 1.125rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.5rem;
    }

    .dark .features-grid-title {
        color: var(--color-text-primary);
    }

    .features-grid-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
    }

    .dark .features-grid-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Hover Border ===== */
    .features-grid-border {
        position: absolute;
        inset: 0;
        border: 2px solid var(--color-primary-500);
        border-radius: 1rem;
        opacity: 0;
        transition: opacity 0.3s ease;
        pointer-events: none;
    }

    .features-grid-item:hover .features-grid-border {
        opacity: 1;
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .features-grid-section {
        direction: rtl;
    }

    [dir="rtl"] .features-grid-number {
        right: auto;
        left: 1rem;
    }

    [dir="rtl"] .features-grid-content {
        text-align: right;
    }

    [dir="rtl"] .features-grid-item:hover .features-grid-icon {
        transform: scale(1.1) rotate(-10deg);
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .features-grid-section {
            padding: 4rem 0;
        }

        .features-grid-container {
            grid-template-columns: 1fr;
            gap: 1rem;
        }

        .features-grid-item {
            padding: 1.5rem 1.25rem;
        }
    }
</style>
</div>
