<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->section = $section;
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->items = !empty($items) ? $items : [
            ['icon' => 'heroicons:leaf', 'title' => 'Eco-Friendly', 'description' => '100% biodegradable detergents.'],
            ['icon' => 'heroicons:clock', 'title' => 'On-Time Delivery', 'description' => 'Guaranteed delivery within promised timeframe.'],
            ['icon' => 'heroicons:shield-check', 'title' => 'Fully Insured', 'description' => 'Your garments are protected.'],
            ['icon' => 'heroicons:heart', 'title' => 'Care & Quality', 'description' => 'Every item treated with attention.'],
        ];
    }

    public function getPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }

    public function getDefaultPaddingClasses(): string
    {
        $classes = [];
        if ($this->paddingTop === null) {
            $classes[] = 'pt-12';
        }
        if ($this->paddingBottom === null) {
            $classes[] = 'pb-12';
        }
        return implode(' ', $classes);
    }
};
?>

<div>
<section class="features-hexagon-section {{ $isWhiteBackground ? 'features-hexagon-white' : 'features-hexagon-gray' }}" @if($this->getPaddingStyle()) style="{{ $this->getPaddingStyle() }}" @endif>
    {{-- Background Hexagons --}}
    <div class="features-hexagon-bg">
        <div class="features-hexagon-bg-shape features-hexagon-bg-shape-1"></div>
        <div class="features-hexagon-bg-shape features-hexagon-bg-shape-2"></div>
        <div class="features-hexagon-bg-shape features-hexagon-bg-shape-3"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        <div class="features-hexagon-grid">
            @foreach ($items as $index => $feature)
                <div class="features-hexagon-item scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Hexagon Shape --}}
                    <div class="features-hexagon-shape">
                        <div class="features-hexagon-shape-inner">
                            <x-icon icon="{{ $feature['icon'] ?? 'heroicons:heart' }}" class="w-8 h-8" />
                        </div>
                    </div>

                    {{-- Content --}}
                    <div class="features-hexagon-content">
                        <h3 class="features-hexagon-title">{{ $feature['title'] }}</h3>
                        <p class="features-hexagon-desc">{{ $feature['description'] }}</p>
                    </div>

                    {{-- Connecting Line --}}
                    @if ($index < count($items) - 1)
                        <div class="features-hexagon-connector"></div>
                    @endif
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Features Hexagon Section ===== */
    .features-hexagon-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .features-hexagon-white {
        background: #ffffff;
    }

    .features-hexagon-gray {
        background: radial-gradient(ellipse at top, #f8fafc 0%, #e2e8f0 100%);
    }

    .dark .features-hexagon-white {
        background: var(--color-bg-secondary);
    }

    .dark .features-hexagon-gray {
        background: radial-gradient(ellipse at top, var(--color-bg-secondary) 0%, var(--color-bg-primary) 100%);
    }

    /* ===== Background Shapes ===== */
    .features-hexagon-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .features-hexagon-bg-shape {
        position: absolute;
        width: 200px;
        height: 200px;
        clip-path: polygon(50% 0%, 100% 25%, 100% 75%, 50% 100%, 0% 75%, 0% 25%);
        opacity: 0.1;
    }

    .features-hexagon-bg-shape-1 {
        top: 10%;
        right: 5%;
        background: var(--color-primary-300);
        transform: rotate(15deg);
    }

    .features-hexagon-bg-shape-2 {
        bottom: 15%;
        left: 10%;
        background: var(--color-secondary-300);
        transform: rotate(-20deg);
    }

    .features-hexagon-bg-shape-3 {
        top: 50%;
        right: 15%;
        background: var(--color-primary-400);
        transform: rotate(30deg);
        width: 150px;
        height: 150px;
    }

    .dark .features-hexagon-bg-shape {
        opacity: 0.05;
    }

    /* ===== Grid ===== */
    .features-hexagon-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
        gap: 3rem;
    }

    /* ===== Item ===== */
    .features-hexagon-item {
        position: relative;
        display: flex;
        flex-direction: column;
        align-items: center;
        text-align: center;
    }

    /* ===== Hexagon Shape ===== */
    .features-hexagon-shape {
        position: relative;
        width: 120px;
        height: 120px;
        margin-bottom: 2rem;
        transition: all 0.4s ease;
    }

    .features-hexagon-item:hover .features-hexagon-shape {
        transform: translateY(-8px) rotate(180deg);
    }

    .features-hexagon-shape::before {
        content: '';
        position: absolute;
        inset: 0;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        clip-path: polygon(50% 0%, 100% 25%, 100% 75%, 50% 100%, 0% 75%, 0% 25%);
        transition: all 0.4s ease;
    }

    .features-hexagon-item:hover .features-hexagon-shape::before {
        transform: scale(1.1);
    }

    .features-hexagon-shape-inner {
        position: absolute;
        inset: 4px;
        background: white;
        clip-path: polygon(50% 0%, 100% 25%, 100% 75%, 50% 100%, 0% 75%, 0% 25%);
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--color-primary-600);
        transition: all 0.4s ease;
    }

    .dark .features-hexagon-shape-inner {
        background: var(--color-card-bg);
        color: var(--color-primary-400);
    }

    .features-hexagon-item:hover .features-hexagon-shape-inner {
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        color: white;
        transform: rotate(-180deg);
    }

    /* ===== Content ===== */
    .features-hexagon-content {
        flex: 1;
    }

    .features-hexagon-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.75rem;
    }

    .dark .features-hexagon-title {
        color: var(--color-text-primary);
    }

    .features-hexagon-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
    }

    .dark .features-hexagon-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Connector Line ===== */
    .features-hexagon-connector {
        display: none;
    }

    @media (min-width: 1024px) {
        .features-hexagon-connector {
            display: block;
            position: absolute;
            top: 60px;
            left: calc(100% - 1.5rem);
            width: 3rem;
            height: 2px;
            background: linear-gradient(90deg, var(--color-primary-300), transparent);
        }

        .dark .features-hexagon-connector {
            background: linear-gradient(90deg, var(--color-primary-800), transparent);
        }
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .features-hexagon-section {
        direction: rtl;
    }

    [dir="rtl"] .features-hexagon-bg-shape-1 {
        right: auto;
        left: 5%;
    }

    [dir="rtl"] .features-hexagon-bg-shape-2 {
        left: auto;
        right: 10%;
    }

    [dir="rtl"] .features-hexagon-bg-shape-3 {
        right: auto;
        left: 15%;
    }

    [dir="rtl"] .features-hexagon-connector {
        left: auto;
        right: calc(100% - 1.5rem);
        background: linear-gradient(-90deg, var(--color-primary-300), transparent);
    }

    [dir="rtl"] .dark .features-hexagon-connector {
        background: linear-gradient(-90deg, var(--color-primary-800), transparent);
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .features-hexagon-section {
            padding: 4rem 0;
        }

        .features-hexagon-grid {
            grid-template-columns: 1fr;
            gap: 2.5rem;
        }

        .features-hexagon-shape {
            width: 100px;
            height: 100px;
            margin-bottom: 1.5rem;
        }
    }
</style>
</div>
