<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->section = $section;
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->items = !empty($items) ? $items : [
            ['icon' => 'heroicons:leaf', 'title' => 'Eco-Friendly', 'description' => '100% biodegradable detergents.'],
            ['icon' => 'heroicons:clock', 'title' => 'On-Time Delivery', 'description' => 'Guaranteed delivery within promised timeframe.'],
            ['icon' => 'heroicons:shield-check', 'title' => 'Fully Insured', 'description' => 'Your garments are protected.'],
            ['icon' => 'heroicons:heart', 'title' => 'Care & Quality', 'description' => 'Every item treated with attention.'],
        ];
    }

    public function getPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }

    public function getDefaultPaddingClasses(): string
    {
        $classes = [];
        if ($this->paddingTop === null) {
            $classes[] = 'pt-12';
        }
        if ($this->paddingBottom === null) {
            $classes[] = 'pb-12';
        }
        return implode(' ', $classes);
    }
};
?>

<div>
<section class="features-split-section {{ $isWhiteBackground ? 'features-split-white' : 'features-split-gray' }}" @if($this->getPaddingStyle()) style="{{ $this->getPaddingStyle() }}" @endif>
    {{-- Background Gradient --}}
    <div class="features-split-bg">
        <div class="features-split-gradient-orb features-split-gradient-orb-1"></div>
        <div class="features-split-gradient-orb features-split-gradient-orb-2"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        <div class="features-split-container">
            @foreach ($items as $index => $feature)
                @php
                    $isEven = $index % 2 === 0;
                @endphp
                <div class="features-split-row {{ $isEven ? 'features-split-row-normal' : 'features-split-row-reverse' }} scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Icon Side --}}
                    <div class="features-split-icon-side">
                        <div class="features-split-icon-wrapper">
                            {{-- Animated Ring --}}
                            <div class="features-split-ring"></div>

                            {{-- Icon Container --}}
                            <div class="features-split-icon-container">
                                <div class="features-split-icon-bg"></div>
                                <div class="features-split-icon">
                                    <x-icon icon="{{ $feature['icon'] ?? 'heroicons:heart' }}" class="w-10 h-10" />
                                </div>
                            </div>

                            {{-- Decorative Dots --}}
                            <div class="features-split-dots">
                                <span></span>
                                <span></span>
                                <span></span>
                            </div>
                        </div>
                    </div>

                    {{-- Content Side --}}
                    <div class="features-split-content-side">
                        <div class="features-split-content">
                            {{-- Step Number --}}
                            <div class="features-split-step">{{ str_pad($index + 1, 2, '0', STR_PAD_LEFT) }}</div>

                            <h3 class="features-split-title">{{ $feature['title'] }}</h3>
                            <p class="features-split-desc">{{ $feature['description'] }}</p>

                            {{-- Progress Bar --}}
                            <div class="features-split-progress">
                                <div class="features-split-progress-bar"></div>
                            </div>
                        </div>
                    </div>

                    {{-- Connecting Line --}}
                    @if ($index < count($items) - 1)
                        <div class="features-split-connector"></div>
                    @endif
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Features Split Section ===== */
    .features-split-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .features-split-white {
        background: #ffffff;
    }

    .features-split-gray {
        background: linear-gradient(to bottom, #f8fafc, #ffffff, #f8fafc);
    }

    .dark .features-split-white {
        background: var(--color-bg-secondary);
    }

    .dark .features-split-gray {
        background: linear-gradient(to bottom, var(--color-bg-primary), var(--color-bg-secondary), var(--color-bg-primary));
    }

    /* ===== Background ===== */
    .features-split-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .features-split-gradient-orb {
        position: absolute;
        width: 500px;
        height: 500px;
        border-radius: 50%;
        filter: blur(100px);
        opacity: 0.2;
    }

    .features-split-gradient-orb-1 {
        top: 10%;
        left: -10%;
        background: linear-gradient(135deg, var(--color-primary-400), var(--color-secondary-400));
    }

    .features-split-gradient-orb-2 {
        bottom: 10%;
        right: -10%;
        background: linear-gradient(-135deg, var(--color-secondary-400), var(--color-primary-400));
    }

    .dark .features-split-gradient-orb {
        opacity: 0.1;
    }

    /* ===== Container ===== */
    .features-split-container {
        display: flex;
        flex-direction: column;
        gap: 4rem;
    }

    /* ===== Row ===== */
    .features-split-row {
        position: relative;
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 3rem;
        align-items: center;
    }

    .features-split-row-reverse {
        direction: rtl;
    }

    .features-split-row-reverse > * {
        direction: ltr;
    }

    /* ===== Icon Side ===== */
    .features-split-icon-side {
        display: flex;
        justify-content: center;
        align-items: center;
    }

    .features-split-icon-wrapper {
        position: relative;
        width: 180px;
        height: 180px;
    }

    /* ===== Animated Ring ===== */
    .features-split-ring {
        position: absolute;
        inset: 0;
        border: 2px solid var(--color-primary-300);
        border-radius: 50%;
        border-top-color: transparent;
        animation: features-split-spin 3s linear infinite;
    }

    .dark .features-split-ring {
        border-color: var(--color-primary-700);
        border-top-color: transparent;
    }

    @keyframes features-split-spin {
        to { transform: rotate(360deg); }
    }

    /* ===== Icon Container ===== */
    .features-split-icon-container {
        position: absolute;
        inset: 20px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        overflow: hidden;
    }

    .features-split-icon-bg {
        position: absolute;
        inset: 0;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        opacity: 0.1;
        transition: all 0.4s ease;
    }

    .features-split-row:hover .features-split-icon-bg {
        opacity: 1;
    }

    .features-split-icon {
        position: relative;
        z-index: 1;
        color: var(--color-primary-600);
        transition: all 0.4s ease;
    }

    .features-split-row:hover .features-split-icon {
        color: white;
        transform: scale(1.2) rotate(10deg);
    }

    .dark .features-split-icon {
        color: var(--color-primary-400);
    }

    /* ===== Decorative Dots ===== */
    .features-split-dots {
        position: absolute;
        bottom: 10px;
        left: 50%;
        transform: translateX(-50%);
        display: flex;
        gap: 8px;
    }

    .features-split-dots span {
        width: 8px;
        height: 8px;
        background: var(--color-primary-500);
        border-radius: 50%;
        animation: features-split-dot-pulse 2s ease-in-out infinite;
    }

    .features-split-dots span:nth-child(2) {
        animation-delay: 0.3s;
    }

    .features-split-dots span:nth-child(3) {
        animation-delay: 0.6s;
    }

    @keyframes features-split-dot-pulse {
        0%, 100% { opacity: 0.3; transform: scale(0.8); }
        50% { opacity: 1; transform: scale(1); }
    }

    /* ===== Content Side ===== */
    .features-split-content-side {
        position: relative;
    }

    .features-split-content {
        max-width: 500px;
    }

    .features-split-row-reverse .features-split-content {
        margin-left: auto;
        text-align: right;
    }

    /* ===== Step Number ===== */
    .features-split-step {
        display: inline-block;
        font-family: var(--font-heading);
        font-size: 0.875rem;
        font-weight: 700;
        color: var(--color-primary-600);
        background: var(--color-primary-50);
        padding: 0.375rem 0.875rem;
        border-radius: 9999px;
        margin-bottom: 1rem;
    }

    .dark .features-split-step {
        color: var(--color-primary-400);
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
    }

    /* ===== Content Text ===== */
    .features-split-title {
        font-family: var(--font-heading);
        font-size: 1.75rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.75rem;
    }

    .dark .features-split-title {
        color: var(--color-text-primary);
    }

    .features-split-desc {
        font-size: 1rem;
        color: var(--color-gray-600);
        line-height: 1.7;
        margin-bottom: 1.5rem;
    }

    .dark .features-split-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Progress Bar ===== */
    .features-split-progress {
        width: 100%;
        height: 4px;
        background: var(--color-gray-200);
        border-radius: 9999px;
        overflow: hidden;
    }

    .dark .features-split-progress {
        background: var(--color-gray-800);
    }

    .features-split-progress-bar {
        height: 100%;
        width: 0;
        background: linear-gradient(90deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 9999px;
        transition: width 0.8s ease;
    }

    .features-split-row:hover .features-split-progress-bar {
        width: 100%;
    }

    /* ===== Connector ===== */
    .features-split-connector {
        position: absolute;
        left: 50%;
        bottom: -2rem;
        width: 2px;
        height: 2rem;
        background: linear-gradient(to bottom, var(--color-primary-300), transparent);
        transform: translateX(-50%);
    }

    .dark .features-split-connector {
        background: linear-gradient(to bottom, var(--color-primary-800), transparent);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .features-split-section {
        direction: rtl;
    }

    [dir="rtl"] .features-split-row-normal {
        direction: ltr;
    }

    [dir="rtl"] .features-split-row-reverse {
        direction: rtl;
    }

    [dir="rtl"] .features-split-row-reverse .features-split-content {
        margin-left: 0;
        margin-right: auto;
        text-align: left;
    }

    [dir="rtl"] .features-split-row-normal .features-split-content {
        text-align: right;
    }

    [dir="rtl"] .features-split-gradient-orb-1 {
        left: auto;
        right: -10%;
    }

    [dir="rtl"] .features-split-gradient-orb-2 {
        right: auto;
        left: -10%;
    }

    [dir="rtl"] .features-split-row:hover .features-split-icon {
        transform: scale(1.2) rotate(-10deg);
    }

    /* ===== Responsive ===== */
    @media (max-width: 768px) {
        .features-split-section {
            padding: 4rem 0;
        }

        .features-split-container {
            gap: 3rem;
        }

        .features-split-row,
        .features-split-row-reverse {
            grid-template-columns: 1fr;
            gap: 2rem;
            direction: ltr;
        }

        .features-split-row-reverse .features-split-content,
        [dir="rtl"] .features-split-row-normal .features-split-content {
            margin-left: 0;
            text-align: left;
        }

        [dir="rtl"] .features-split-row .features-split-content {
            text-align: right;
        }

        .features-split-icon-wrapper {
            width: 140px;
            height: 140px;
        }

        .features-split-title {
            font-size: 1.5rem;
        }

        .features-split-connector {
            display: none;
        }
    }
</style>
</div>
