<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $company = [];
    public array $nav = [];
    public array $navItems = [];
    public array $languages = [];
    public ?string $currentLangKey = 'en';
    public ?string $websiteSlug = null;
    public string $homeUrl = '';

    public function mount(array $section = [], array $company = [], array $nav = [], array $navItems = [], array $languages = [], ?string $currentLangKey = 'en', ?string $websiteSlug = null)
    {
        $this->section = $section;
        $this->company = $company;
        $this->nav = $nav;
        $this->navItems = $navItems;
        $this->languages = $languages;
        $this->currentLangKey = $currentLangKey ?? 'en';
        $this->websiteSlug = $websiteSlug;

        // Generate home URL using global helper (handles is_primary for root URL)
        $frontWebsite = view()->shared('frontWebsite');
        $this->homeUrl = landing_home_url($frontWebsite, $this->websiteSlug);
    }

    /**
     * Change the frontend language
     * Stores in cookie with slug-based key (like theme mode), then triggers page reload
     * Cookie key format: lang_{slug} or lang_default
     */
    public function changeLanguage(string $langKey): void
    {
        // Use slug-based cookie key (same pattern as theme mode: theme_{slug})
        $langCookieKey = 'lang_' . ($this->websiteSlug ?? 'default');

        // Set cookie for persistence (1 year)
        cookie()->queue($langCookieKey, $langKey, 60 * 24 * 365);

        // Dispatch browser event to reload the page
        $this->dispatch('language-changed');
    }
};
?>

@php
    $textColorMode = $section['text_color_mode'] ?? 'auto';
    $textColorClass = match($textColorMode) {
        'light' => 'header2-text-light',
        'dark' => 'header2-text-dark',
        default => '',
    };
    $isSolidHeader = !($section['transparent'] ?? true);
@endphp
<div>
<!-- Floating Pill Header -->
<header id="header" class="header2-wrapper {{ $isSolidHeader ? 'header2-solid' : 'header2-transparent' }} {{ $textColorClass }}"
    data-transparent="{{ $isSolidHeader ? 'false' : 'true' }}"
    data-text-color-mode="{{ $textColorMode }}"
    data-solid="{{ $isSolidHeader ? 'true' : 'false' }}">
    <div class="header2-container">
        <div class="header2-pill">
            {{-- Logo with theme mode support based on header_logo_type setting --}}
            @php
                $headerLogoType = $company['header_logo_type'] ?? 'logo_with_name';
                $showName = in_array($headerLogoType, ['logo_with_name', 'light_dark_with_name']);
                $useFullLogo = in_array($headerLogoType, ['light_dark_with_name', 'light_dark_only']);
            @endphp
            <a href="{{ $homeUrl }}" wire:navigate class="header2-logo">
                @if ($useFullLogo)
                    {{-- Full Logo (Light/Dark based on theme) --}}
                    <img src="{{ $company['light_logo'] ?? asset('images/light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="header2-logo-img dark:hidden" />
                    <img src="{{ $company['dark_logo'] ?? asset('images/dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="header2-logo-img hidden dark:block" />
                @else
                    {{-- Small Logo (Light/Dark based on theme) --}}
                    <img src="{{ $company['light_logo_small'] ?? asset('images/small_light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="header2-logo-img dark:hidden" />
                    <img src="{{ $company['dark_logo_small'] ?? asset('images/small_dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="header2-logo-img hidden dark:block" />
                @endif

                @if ($showName && !empty($company['name']))
                    <span class="header2-logo-text">{{ $company['name'] }}</span>
                @endif
            </a>

            <!-- Desktop Navigation -->
            @php
                // Show max 3 items in pill header, rest go to More dropdown
                $maxVisibleItems = 3;
                $totalItems = count($navItems);
                $hasMoreItems = $totalItems > $maxVisibleItems;
            @endphp
            <nav class="header2-nav">
                @forelse ($navItems as $index => $item)
                    @if ($index < $maxVisibleItems)
                        <a href="{{ $item['url'] }}" wire:navigate class="header2-nav-link"
                            data-page="{{ $item['page_key'] }}">
                            {{ $item['name'] }}
                        </a>
                    @endif
                @empty
                    <a href="{{ $homeUrl }}#services" wire:navigate class="header2-nav-link"
                        data-section="services">
                        {{ $nav['services'] ?? 'Services' }}
                    </a>
                    <a href="{{ $homeUrl }}#how-it-works" wire:navigate class="header2-nav-link"
                        data-section="how-it-works">
                        {{ $nav['how_it_works'] ?? 'How It Works' }}
                    </a>
                    <a href="{{ $homeUrl }}#pricing" wire:navigate class="header2-nav-link"
                        data-section="pricing">
                        {{ $nav['pricing'] ?? 'Pricing' }}
                    </a>
                @endforelse

                {{-- More Dropdown for remaining items --}}
                @if ($hasMoreItems)
                    <div class="relative" x-data="{ open: false }">
                        <button @click="open = !open" @click.away="open = false"
                            class="header2-nav-link flex items-center gap-1">
                            {{ $section['more_text'] ?? 'More' }}
                            <svg class="w-3.5 h-3.5 transition-transform duration-200" :class="{ 'rotate-180': open }"
                                fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </button>
                        <div x-show="open"
                            x-transition:enter="transition ease-out duration-200"
                            x-transition:enter-start="opacity-0 translate-y-2"
                            x-transition:enter-end="opacity-100 translate-y-0"
                            x-transition:leave="transition ease-in duration-150"
                            x-transition:leave-start="opacity-100 translate-y-0"
                            x-transition:leave-end="opacity-0 translate-y-2"
                            class="header2-more-dropdown absolute top-full left-0 mt-2 min-w-[200px] rounded-lg shadow-lg bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 z-50 py-1.5 overflow-hidden"
                            style="display: none;">
                            @foreach ($navItems as $index => $item)
                                @if ($index >= $maxVisibleItems)
                                    <a href="{{ $item['url'] }}" wire:navigate
                                        data-page="{{ $item['page_key'] }}"
                                        class="header2-more-item group flex items-center justify-between gap-2 px-4 py-2.5 text-sm text-gray-600 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700/50 hover:text-primary-600 dark:hover:text-primary-400 transition-colors duration-150">
                                        <span>{{ $item['name'] }}</span>
                                        <svg class="w-3.5 h-3.5 opacity-0 -translate-x-1 group-hover:opacity-100 group-hover:translate-x-0 transition-all duration-150 text-primary-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                                        </svg>
                                    </a>
                                @endif
                            @endforeach
                        </div>
                    </div>
                @endif
            </nav>

            <!-- Right Actions -->
            <div class="header2-actions">
                <!-- Language Selector (Desktop) -->
                @if (!empty($languages) && count($languages) > 1)
                    <div class="header2-lang-wrapper hidden lg:block" x-data="{ open: false }">
                        <button @click="open = !open" @click.away="open = false" class="header2-lang-btn"
                            aria-label="Language selector">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M3 5h12M9 3v2m1.048 9.5A18.022 18.022 0 016.412 9m6.088 9h7M11 21l5-10 5 10M12.751 5C11.783 10.77 8.07 15.61 3 18.129">
                                </path>
                            </svg>
                            <span class="header2-lang-text">
                                {{ strtoupper($languages[$currentLangKey]['short_name'] ?? $currentLangKey) }}
                            </span>
                        </button>

                        <div x-show="open" x-transition class="header2-lang-dropdown">
                            @foreach ($languages as $lang)
                                <button wire:click="changeLanguage('{{ $lang['key'] }}')"
                                    class="header2-lang-option {{ $lang['key'] === $currentLangKey ? 'active' : '' }}">
                                    <div class="flex items-center gap-2">
                                        @if (!empty($lang['flag_emoji']))
                                            <span class="text-base">{{ $lang['flag_emoji'] }}</span>
                                        @endif
                                        <span>{{ $lang['name'] }}</span>
                                    </div>
                                    @if ($lang['key'] === $currentLangKey)
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M5 13l4 4L19 7"></path>
                                        </svg>
                                    @endif
                                </button>
                            @endforeach
                        </div>
                    </div>
                @endif

                <!-- Theme Toggle (Desktop) - Hidden for now, only light mode supported
                <button id="theme-toggle" class="header2-theme-toggle hidden lg:flex" aria-label="Toggle theme">
                    <svg class="w-5 h-5 sun-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z">
                        </path>
                    </svg>
                    <svg class="w-5 h-5 moon-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z">
                        </path>
                    </svg>
                </button>
                -->

                <!-- CTA Buttons (Desktop) -->
                @if (!empty($section['cta_buttons']) && is_array($section['cta_buttons']))
                    @foreach ($section['cta_buttons'] as $btnIndex => $ctaBtn)
                        @if (is_array($ctaBtn) && (!empty($ctaBtn['text']) || !empty($ctaBtn['icon'])))
                            <a href="{{ $ctaBtn['url'] ?? '#contact' }}"
                                @if (str_starts_with($ctaBtn['url'] ?? '#contact', '#')) wire:navigate data-section="{{ ltrim($ctaBtn['url'] ?? '#contact', '#') }}" @endif
                                class="header2-cta-btn hidden lg:flex">
                                @if (!empty($ctaBtn['icon']))<x-icon :icon="$ctaBtn['icon']" class="w-5 h-5" />@endif
                                @if (!empty($ctaBtn['text']))<span>{{ $ctaBtn['text'] }}</span>@endif
                            </a>
                        @endif
                    @endforeach
                @endif

                <!-- Mobile: Menu Button -->
                <button id="mobile-menu-btn" class="header2-mobile-btn lg:hidden" aria-label="Open menu">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M4 6h16M4 12h16M4 18h16"></path>
                    </svg>
                </button>
            </div>
        </div>
    </div>
</header>

<!-- Mobile Menu Overlay -->
<div id="mobile-menu-overlay" class="header2-overlay"></div>

<!-- Mobile Menu -->
<div id="mobile-menu" class="header2-mobile-menu">
    <!-- Mobile Menu Header -->
    <div class="header2-mobile-header">
        <div class="flex items-center gap-3">
            @if ($useFullLogo)
                <img src="{{ $company['light_logo'] ?? asset('images/light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                    class="h-10 w-auto dark:hidden" />
                <img src="{{ $company['dark_logo'] ?? asset('images/dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                    class="h-10 w-auto hidden dark:block" />
            @else
                <img src="{{ $company['light_logo_small'] ?? asset('images/small_light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                    class="h-10 w-auto dark:hidden" />
                <img src="{{ $company['dark_logo_small'] ?? asset('images/small_dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                    class="h-10 w-auto hidden dark:block" />
            @endif
            @if ($showName && !empty($company['name']))
                <span class="font-bold text-lg">{{ $company['name'] }}</span>
            @endif
        </div>
        <button id="mobile-menu-close" class="header2-mobile-close" aria-label="Close menu">
            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                </path>
            </svg>
        </button>
    </div>

    <!-- Mobile Navigation -->
    <nav class="flex-1 overflow-y-auto">
        @forelse ($navItems as $item)
            <a href="{{ $item['url'] }}" wire:navigate
                class="header2-mobile-link mobile-menu-link" data-page="{{ $item['page_key'] }}">
                <span>{{ $item['name'] }}</span>
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
        @empty
            <a href="{{ $homeUrl }}#services" wire:navigate class="header2-mobile-link mobile-menu-link"
                data-section="services">
                <span>{{ $nav['services'] ?? 'Services' }}</span>
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
            <a href="{{ $homeUrl }}#how-it-works" wire:navigate class="header2-mobile-link mobile-menu-link"
                data-section="how-it-works">
                <span>{{ $nav['how_it_works'] ?? 'How It Works' }}</span>
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
            <a href="{{ $homeUrl }}#pricing" wire:navigate class="header2-mobile-link mobile-menu-link"
                data-section="pricing">
                <span>{{ $nav['pricing'] ?? 'Pricing' }}</span>
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
            <a href="{{ $homeUrl }}#testimonials" wire:navigate class="header2-mobile-link mobile-menu-link"
                data-section="testimonials">
                <span>{{ $nav['reviews'] ?? 'Reviews' }}</span>
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
            <a href="{{ $homeUrl }}#faq" wire:navigate class="header2-mobile-link mobile-menu-link"
                data-section="faq">
                <span>{{ $nav['faq'] ?? 'FAQ' }}</span>
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
            <a href="{{ $homeUrl }}#contact" wire:navigate class="header2-mobile-link mobile-menu-link"
                data-section="contact">
                <span>{{ $nav['contact'] ?? 'Contact' }}</span>
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </a>
        @endforelse
    </nav>

    <!-- Mobile Menu Footer -->
    <div class="header2-mobile-footer">
        <div class="flex items-center justify-between mb-4">
            <!-- Theme Toggle Mobile - Hidden for now, only light mode supported
            <button id="theme-toggle-mobile" class="header2-theme-toggle-mobile" aria-label="Toggle theme">
                <svg class="w-5 h-5 sun-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z">
                    </path>
                </svg>
                <svg class="w-5 h-5 moon-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z">
                    </path>
                </svg>
            </button>
            -->

            <!-- Language Selector Mobile -->
            @if (!empty($languages) && count($languages) > 1)
                <div class="header2-lang-wrapper" x-data="{ open: false }">
                    <button @click="open = !open" @click.away="open = false" class="header2-lang-btn-mobile">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M3 5h12M9 3v2m1.048 9.5A18.022 18.022 0 016.412 9m6.088 9h7M11 21l5-10 5 10M12.751 5C11.783 10.77 8.07 15.61 3 18.129">
                            </path>
                        </svg>
                        <span>{{ strtoupper($languages[$currentLangKey]['short_name'] ?? $currentLangKey) }}</span>
                    </button>

                    <div x-show="open" x-transition class="header2-lang-dropdown">
                        @foreach ($languages as $lang)
                            <button wire:click="changeLanguage('{{ $lang['key'] }}')"
                                class="header2-lang-option {{ $lang['key'] === $currentLangKey ? 'active' : '' }}">
                                <div class="flex items-center gap-2">
                                    @if (!empty($lang['flag_emoji']))
                                        <span class="text-base">{{ $lang['flag_emoji'] }}</span>
                                    @endif
                                    <span>{{ $lang['name'] }}</span>
                                </div>
                                @if ($lang['key'] === $currentLangKey)
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M5 13l4 4L19 7"></path>
                                    </svg>
                                @endif
                            </button>
                        @endforeach
                    </div>
                </div>
            @endif
        </div>

        <!-- CTA Buttons Mobile -->
        @if (!empty($section['cta_buttons']) && is_array($section['cta_buttons']))
            <div class="space-y-2">
                @foreach ($section['cta_buttons'] as $btnIndex => $ctaBtn)
                    @if (is_array($ctaBtn) && (!empty($ctaBtn['text']) || !empty($ctaBtn['icon'])))
                        <a href="{{ $ctaBtn['url'] ?? '#contact' }}"
                            @if (str_starts_with($ctaBtn['url'] ?? '#contact', '#')) data-section="{{ ltrim($ctaBtn['url'] ?? '#contact', '#') }}" @endif
                            class="header2-mobile-cta mobile-menu-link flex items-center justify-center gap-2">
                            @if (!empty($ctaBtn['icon']))<x-icon :icon="$ctaBtn['icon']" class="w-5 h-5" />@endif
                            @if (!empty($ctaBtn['text']))<span>{{ $ctaBtn['text'] }}</span>@endif
                        </a>
                    @endif
                @endforeach
            </div>
        @endif
    </div>
</div>
</div>

<style>
        /* Header 2: Floating Pill Header */

        /* ===== Header Wrapper ===== */
        .header2-wrapper {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 50;
            padding-top: 1rem;
            pointer-events: none;
        }

        /* ===== Header Container ===== */
        .header2-container {
            max-width: 80rem;
            margin: 0 auto;
            padding: 0 1rem;
        }

        /* ===== Floating Pill ===== */
        .header2-pill {
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 1rem;
            padding: 0.75rem 1.5rem;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(12px);
            border-radius: 9999px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            border: 1px solid rgba(229, 231, 235, 0.8);
            pointer-events: auto;
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
            max-width: 100%;
        }

        .dark .header2-pill {
            background: rgba(31, 41, 55, 0.95);
            border-color: rgba(75, 85, 99, 0.5);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.3), 0 2px 4px -1px rgba(0, 0, 0, 0.2);
        }

        .header-scrolled .header2-pill {
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        }

        .dark .header-scrolled .header2-pill {
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.4), 0 4px 6px -2px rgba(0, 0, 0, 0.3);
        }

        /* ===== Logo ===== */
        .header2-logo {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            flex-shrink: 0;
            transition: transform 200ms;
            min-width: 0;
        }

        .header2-logo:hover {
            transform: scale(1.02);
        }

        .header2-logo-img {
            height: 2.5rem;
            width: auto;
        }

        .header2-logo-icon {
            width: 3rem;
            height: 3rem;
            border-radius: 0.75rem;
            background: linear-gradient(to bottom right, var(--color-primary-500), var(--color-primary-600), var(--color-primary-700));
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 10px 15px -3px rgba(37, 99, 235, 0.3);
            transition: all 300ms;
        }

        .header2-logo-icon-small {
            width: 2.5rem;
            height: 2.5rem;
            border-radius: 0.625rem;
            background: linear-gradient(to bottom right, var(--color-primary-500), var(--color-primary-600), var(--color-primary-700));
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .header2-logo-text {
            font-size: 1rem;
            line-height: 1.5rem;
            font-weight: 700;
            color: #111827;
            white-space: nowrap;
        }

        .dark .header2-logo-text {
            color: var(--color-text-primary);
        }

        /* ===== Desktop Navigation ===== */
        .header2-nav {
            display: none;
            align-items: center;
            gap: 0.25rem;
            flex: 0 1 auto;
            justify-content: center;
        }

        @media (min-width: 1024px) {
            .header2-nav {
                display: flex;
            }
        }

        .header2-nav-link {
            padding: 0.5rem 0.75rem;
            font-weight: 500;
            color: #374151;
            border-radius: 9999px;
            transition: all 200ms;
            white-space: nowrap;
            position: relative;
            flex-shrink: 0;
        }

        .dark .header2-nav-link {
            color: var(--color-text-secondary);
        }

        .header2-nav-link:hover {
            color: var(--color-primary-600);
            background-color: var(--color-primary-50);
        }

        .dark .header2-nav-link:hover {
            color: var(--color-primary-400);
            background-color: rgba(37, 99, 235, 0.1);
        }

        /* ===== More Dropdown ===== */
        .header2-more-dropdown {
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
        }

        .header2-more-dropdown::before {
            content: '';
            position: absolute;
            top: -6px;
            left: 20px;
            border-left: 6px solid transparent;
            border-right: 6px solid transparent;
            border-bottom: 6px solid white;
        }

        .dark .header2-more-dropdown::before {
            border-bottom-color: var(--color-gray-800);
        }

        .header2-more-item:hover span:first-child {
            opacity: 1;
        }

        /* ===== Actions ===== */
        .header2-actions {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            flex-shrink: 0;
        }

        /* ===== Language Selector ===== */
        .header2-lang-wrapper {
            position: relative;
        }

        .header2-lang-btn,
        .header2-lang-btn-mobile {
            display: flex;
            align-items: center;
            gap: 0.375rem;
            padding: 0.5rem 0.75rem;
            border-radius: 9999px;
            background-color: #f3f4f6;
            color: #374151;
            font-size: 0.875rem;
            font-weight: 500;
            transition: all 200ms;
            white-space: nowrap;
        }

        .dark .header2-lang-btn,
        .dark .header2-lang-btn-mobile {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .header2-lang-btn:hover,
        .header2-lang-btn-mobile:hover {
            background-color: #e5e7eb;
        }

        .dark .header2-lang-btn:hover,
        .dark .header2-lang-btn-mobile:hover {
            background-color: #475569;
        }

        .header2-lang-text {
            display: none;
        }

        @media (min-width: 640px) {
            .header2-lang-text {
                display: block;
            }
        }

        .header2-lang-dropdown {
            position: absolute;
            right: 0;
            margin-top: 0.5rem;
            width: 11rem;
            border-radius: 0.75rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            background-color: #ffffff;
            border: 1px solid #e5e7eb;
            padding: 0.5rem 0;
            z-index: 50;
        }

        .dark .header2-lang-dropdown {
            background-color: var(--color-card-bg);
            border-color: var(--color-border);
        }

        /* Language dropdown in mobile menu footer - opens upward */
        .header2-mobile-footer .header2-lang-dropdown {
            bottom: 100%;
            top: auto;
            margin-top: 0;
            margin-bottom: 0.5rem;
        }

        .header2-lang-option {
            display: flex;
            align-items: center;
            justify-content: space-between;
            width: 100%;
            padding: 0.625rem 1rem;
            text-align: left;
            font-size: 0.875rem;
            color: #374151;
            transition: all 200ms;
        }

        .dark .header2-lang-option {
            color: var(--color-text-secondary);
        }

        .header2-lang-option:hover {
            background-color: var(--color-primary-50);
            color: var(--color-primary-600);
        }

        .dark .header2-lang-option:hover {
            background-color: rgba(37, 99, 235, 0.1);
            color: var(--color-primary-400);
        }

        .header2-lang-option.active {
            color: var(--color-primary-600);
            font-weight: 600;
        }

        .dark .header2-lang-option.active {
            color: var(--color-primary-400);
        }

        /* ===== Theme Toggle ===== */
        .header2-theme-toggle,
        .header2-theme-toggle-mobile {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 0.5rem;
            border-radius: 9999px;
            background-color: #f3f4f6;
            color: #374151;
            transition: all 200ms;
        }

        .dark .header2-theme-toggle,
        .dark .header2-theme-toggle-mobile {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .header2-theme-toggle:hover,
        .header2-theme-toggle-mobile:hover {
            background-color: #e5e7eb;
        }

        .dark .header2-theme-toggle:hover,
        .dark .header2-theme-toggle-mobile:hover {
            background-color: #475569;
        }

        .header2-theme-toggle .sun-icon,
        .header2-theme-toggle-mobile .sun-icon {
            display: none;
        }

        .dark .header2-theme-toggle .sun-icon,
        .dark .header2-theme-toggle-mobile .sun-icon {
            display: block;
        }

        .header2-theme-toggle .moon-icon,
        .header2-theme-toggle-mobile .moon-icon {
            display: block;
        }

        .dark .header2-theme-toggle .moon-icon,
        .dark .header2-theme-toggle-mobile .moon-icon {
            display: none;
        }

        /* ===== CTA Button ===== */
        .header2-cta-btn {
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 9999px;
            font-weight: 600;
            font-size: 0.875rem;
            background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600));
            color: white;
            box-shadow: 0 4px 14px rgba(37, 99, 235, 0.4);
            transition: all 300ms;
            white-space: nowrap;
        }

        .header2-cta-btn:hover {
            background: linear-gradient(135deg, var(--color-primary-700), var(--color-secondary-700));
            box-shadow: 0 6px 20px rgba(37, 99, 235, 0.5);
            transform: translateY(-2px);
        }

        /* ===== Mobile Menu Button ===== */
        .header2-mobile-btn {
            align-items: center;
            justify-content: center;
            padding: 0.5rem;
            border-radius: 9999px;
            background-color: #f3f4f6;
            color: #374151;
            transition: all 200ms;
        }

        .dark .header2-mobile-btn {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .header2-mobile-btn:hover {
            background-color: #e5e7eb;
        }

        .dark .header2-mobile-btn:hover {
            background-color: #475569;
        }

        /* ===== Mobile Menu Overlay ===== */
        .header2-overlay {
            position: fixed;
            inset: 0;
            background-color: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(4px);
            opacity: 0;
            pointer-events: none;
            transition: opacity 300ms;
            z-index: 40;
        }

        .header2-overlay.active {
            opacity: 1;
            pointer-events: auto;
        }

        /* ===== Mobile Menu ===== */
        .header2-mobile-menu {
            position: fixed;
            top: 0;
            bottom: 0;
            right: 0;
            width: 100%;
            max-width: 24rem;
            background-color: #ffffff;
            box-shadow: -10px 0 25px -5px rgba(0, 0, 0, 0.1);
            transform: translateX(100%);
            transition: transform 300ms cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 50;
            display: flex;
            flex-direction: column;
        }

        .dark .header2-mobile-menu {
            background-color: var(--color-bg-primary);
        }

        .header2-mobile-menu.active {
            transform: translateX(0);
        }

        /* ===== Mobile Menu Header ===== */
        .header2-mobile-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 1.25rem;
            border-bottom: 1px solid #e5e7eb;
            background: linear-gradient(to right, var(--color-primary-50), rgba(37, 99, 235, 0.05));
        }

        .dark .header2-mobile-header {
            border-bottom-color: var(--color-border);
            background: linear-gradient(to right, rgba(37, 99, 235, 0.1), rgba(37, 99, 235, 0.05));
        }

        .header2-mobile-close {
            padding: 0.5rem;
            border-radius: 0.5rem;
            color: var(--color-primary-700);
            transition: background-color 200ms;
        }

        .dark .header2-mobile-close {
            color: var(--color-primary-400);
        }

        .header2-mobile-close:hover {
            background-color: rgba(255, 255, 255, 0.5);
        }

        .dark .header2-mobile-close:hover {
            background-color: rgba(31, 41, 55, 0.5);
        }

        /* ===== Mobile Navigation ===== */
        .header2-mobile-link {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 1rem 1.5rem;
            color: #374151;
            border-bottom: 1px solid #f3f4f6;
            transition: all 200ms;
        }

        .dark .header2-mobile-link {
            color: var(--color-text-secondary);
            border-bottom-color: var(--color-border);
        }

        .header2-mobile-link:hover {
            background-color: var(--color-primary-50);
            color: var(--color-primary-600);
        }

        .dark .header2-mobile-link:hover {
            background-color: rgba(37, 99, 235, 0.1);
            color: var(--color-primary-400);
        }

        /* ===== Mobile Footer ===== */
        .header2-mobile-footer {
            padding: 1rem 1.5rem;
            border-top: 1px solid #e5e7eb;
            background-color: #f9fafb;
        }

        .dark .header2-mobile-footer {
            border-top-color: var(--color-border);
            background-color: var(--color-bg-secondary);
        }

        .header2-mobile-cta {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            width: 100%;
            padding: 0.75rem 1.5rem;
            border-radius: 0.75rem;
            font-weight: 600;
            background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600));
            color: white;
            box-shadow: 0 10px 15px -3px rgba(37, 99, 235, 0.25);
            transition: all 300ms;
        }

        .header2-mobile-cta:hover {
            background: linear-gradient(135deg, var(--color-primary-700), var(--color-secondary-700));
            box-shadow: 0 10px 15px -3px rgba(37, 99, 235, 0.4);
        }

        /* ===== TRANSPARENT MODE ===== */
        .header2-transparent .header2-pill {
            background: rgba(0, 0, 0, 0.15);
            border-color: rgba(255, 255, 255, 0.2);
        }

        .header2-transparent .header2-logo-text {
            color: white;
        }

        .header2-transparent .header2-nav-link {
            color: rgba(255, 255, 255, 0.9);
        }

        .header2-transparent .header2-nav-link:hover {
            color: white;
            background-color: rgba(255, 255, 255, 0.15);
        }

        .header2-transparent .header2-lang-btn,
        .header2-transparent .header2-lang-btn-mobile {
            background-color: rgba(255, 255, 255, 0.15);
            color: rgba(255, 255, 255, 0.9);
        }

        .header2-transparent .header2-lang-btn:hover,
        .header2-transparent .header2-lang-btn-mobile:hover {
            background-color: rgba(255, 255, 255, 0.25);
            color: white;
        }

        .header2-transparent .header2-theme-toggle {
            background-color: rgba(255, 255, 255, 0.15);
            color: rgba(255, 255, 255, 0.9);
        }

        .header2-transparent .header2-theme-toggle:hover {
            background-color: rgba(255, 255, 255, 0.25);
            color: white;
        }

        .header2-transparent .header2-mobile-btn {
            background-color: rgba(255, 255, 255, 0.15);
            color: white;
        }

        .header2-transparent .header2-mobile-btn:hover {
            background-color: rgba(255, 255, 255, 0.25);
        }

        /* ===== TEXT COLOR MODE ===== */
        /* IMPORTANT: Text color mode settings only apply in LIGHT mode */
        /* In dark mode, colors are always controlled by dark mode variables */

        /* ===== TEXT COLOR MODE: LIGHT (white text - for dark hero backgrounds) - LIGHT MODE ONLY ===== */
        /* Only applies when NOT scrolled and NOT in dark mode */
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-pill {
            background: rgba(0, 0, 0, 0.15) !important;
            border-color: rgba(255, 255, 255, 0.2) !important;
        }

        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-logo-text,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-nav-link {
            color: rgba(255, 255, 255, 0.9) !important;
        }

        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-nav-link:hover,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-logo-text:hover {
            color: white !important;
        }

        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-lang-btn,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-lang-btn-mobile,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-theme-toggle,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-mobile-btn {
            background-color: rgba(255, 255, 255, 0.15) !important;
            color: rgba(255, 255, 255, 0.9) !important;
        }

        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-lang-btn:hover,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-lang-btn-mobile:hover,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-theme-toggle:hover,
        body:not(.dark) .header2-transparent.header2-text-light:not(.header-scrolled) .header2-mobile-btn:hover {
            background-color: rgba(255, 255, 255, 0.25) !important;
            color: white !important;
        }

        /* ===== TEXT COLOR MODE: DARK (black text - for light hero backgrounds) - LIGHT MODE ONLY ===== */
        /* Only applies when NOT scrolled and NOT in dark mode */
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-pill {
            background: rgba(255, 255, 255, 0.15) !important;
            border-color: rgba(0, 0, 0, 0.1) !important;
        }

        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-logo-text,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-nav-link {
            color: var(--color-gray-800) !important;
        }

        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-nav-link:hover,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-logo-text:hover {
            color: var(--color-gray-900) !important;
        }

        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-lang-btn,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-lang-btn-mobile,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-theme-toggle,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-mobile-btn {
            background-color: rgba(0, 0, 0, 0.08) !important;
            color: var(--color-gray-700) !important;
        }

        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-lang-btn:hover,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-lang-btn-mobile:hover,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-theme-toggle:hover,
        body:not(.dark) .header2-transparent.header2-text-dark:not(.header-scrolled) .header2-mobile-btn:hover {
            background-color: rgba(0, 0, 0, 0.12) !important;
            color: var(--color-gray-900) !important;
        }

        /* Dark mode for scrolled headers (both transparent and solid) */
        .dark #header.header-scrolled .header2-pill,
        .dark .header2-transparent.header2-scrolled .header2-pill {
            background-color: var(--color-bg-primary) !important;
            border-color: var(--color-border) !important;
        }

        .dark .header2-transparent.header2-scrolled .header2-logo-text {
            color: var(--color-text-primary) !important;
        }

        .dark .header2-transparent.header2-scrolled .header2-nav-link {
            color: var(--color-text-secondary) !important;
        }

        .dark .header2-transparent.header2-scrolled .header2-nav-link:hover {
            color: var(--color-primary-400) !important;
            background-color: rgba(37, 99, 235, 0.1) !important;
        }

        .dark .header2-transparent.header2-scrolled .header2-lang-btn,
        .dark .header2-transparent.header2-scrolled .header2-lang-btn-mobile {
            background-color: var(--color-bg-tertiary) !important;
            color: var(--color-text-tertiary) !important;
        }

        .dark .header2-transparent.header2-scrolled .header2-lang-btn:hover,
        .dark .header2-transparent.header2-scrolled .header2-lang-btn-mobile:hover {
            background-color: #475569 !important;
            color: var(--color-primary-400) !important;
        }

        .dark .header2-transparent.header2-scrolled .header2-theme-toggle,
        .dark .header2-transparent.header2-scrolled .header2-mobile-btn {
            background-color: var(--color-bg-tertiary) !important;
            color: var(--color-text-primary) !important;
        }

        .dark .header2-transparent.header2-scrolled .header2-theme-toggle:hover,
        .dark .header2-transparent.header2-scrolled .header2-mobile-btn:hover {
            background-color: #475569 !important;
        }

        /* Transparent mode when scrolled */
        .header2-transparent.header2-scrolled .header2-pill {
            background: rgba(255, 255, 255, 0.95);
            border-color: rgba(229, 231, 235, 0.8);
        }

        .header2-transparent.header2-scrolled .header2-logo-text {
            color: #111827;
        }

        .header2-transparent.header2-scrolled .header2-nav-link {
            color: #374151;
        }

        .header2-transparent.header2-scrolled .header2-nav-link:hover {
            color: var(--color-primary-600);
            background-color: var(--color-primary-50);
        }

        .header2-transparent.header2-scrolled .header2-lang-btn,
        .header2-transparent.header2-scrolled .header2-lang-btn-mobile {
            background-color: #f3f4f6;
            color: #374151;
        }

        .header2-transparent.header2-scrolled .header2-lang-btn:hover,
        .header2-transparent.header2-scrolled .header2-lang-btn-mobile:hover {
            background-color: #e5e7eb;
        }

        .header2-transparent.header2-scrolled .header2-theme-toggle {
            background-color: #f3f4f6;
            color: #374151;
        }

        .header2-transparent.header2-scrolled .header2-theme-toggle:hover {
            background-color: #e5e7eb;
        }

        .header2-transparent.header2-scrolled .header2-mobile-btn {
            background-color: #f3f4f6;
            color: #374151;
        }

        .header2-transparent.header2-scrolled .header2-mobile-btn:hover {
            background-color: #e5e7eb;
        }

        /* Dark mode - Transparent scrolled */
        .dark .header2-transparent.header2-scrolled .header2-pill {
            background: rgba(31, 41, 55, 0.95);
            border-color: rgba(75, 85, 99, 0.5);
        }

        .dark .header2-transparent.header2-scrolled .header2-logo-text {
            color: var(--color-text-primary);
        }

        .dark .header2-transparent.header2-scrolled .header2-nav-link {
            color: var(--color-text-secondary);
        }

        .dark .header2-transparent.header2-scrolled .header2-nav-link:hover {
            color: var(--color-primary-400);
            background-color: rgba(37, 99, 235, 0.1);
        }

        .dark .header2-transparent.header2-scrolled .header2-lang-btn,
        .dark .header2-transparent.header2-scrolled .header2-lang-btn-mobile {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .dark .header2-transparent.header2-scrolled .header2-lang-btn:hover,
        .dark .header2-transparent.header2-scrolled .header2-lang-btn-mobile:hover {
            background-color: #475569;
        }

        .dark .header2-transparent.header2-scrolled .header2-theme-toggle {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .dark .header2-transparent.header2-scrolled .header2-theme-toggle:hover {
            background-color: #475569;
        }

        .dark .header2-transparent.header2-scrolled .header2-mobile-btn {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .dark .header2-transparent.header2-scrolled .header2-mobile-btn:hover {
            background-color: #475569;
        }

        /* ===== RTL Support ===== */
        [dir="rtl"] .header2-mobile-menu {
            right: auto;
            left: 0;
            transform: translateX(-100%);
        }

        [dir="rtl"] .header2-mobile-menu.active {
            transform: translateX(0);
        }

        [dir="rtl"] .header2-lang-dropdown {
            right: auto;
            left: 0;
        }

        /* ===== Responsive Adjustments ===== */
        @media (max-width: 640px) {
            .header2-pill {
                padding: 0.625rem 1rem;
                gap: 1rem;
            }

            .header2-logo-text {
                font-size: 1rem;
            }

            .header2-actions {
                gap: 0.375rem;
            }
        }

        @media (max-width: 1024px) {
            .header2-container {
                padding: 0 0.75rem;
            }
        }

        /* ===== SOLID MODE (Non-transparent header) ===== */
        .header2-solid .header2-pill {
            background: rgba(255, 255, 255, 0.95) !important;
            border-color: rgba(229, 231, 235, 0.8) !important;
        }

        .header2-solid .header2-logo-text {
            color: #111827 !important;
        }

        .header2-solid .header2-nav-link {
            color: #374151 !important;
        }

        .header2-solid .header2-nav-link:hover {
            color: var(--color-primary-600) !important;
            background-color: var(--color-primary-50) !important;
        }

        .header2-solid .header2-lang-btn,
        .header2-solid .header2-lang-btn-mobile,
        .header2-solid .header2-theme-toggle,
        .header2-solid .header2-mobile-btn {
            background-color: #f3f4f6 !important;
            color: #374151 !important;
        }

        .header2-solid .header2-lang-btn:hover,
        .header2-solid .header2-lang-btn-mobile:hover,
        .header2-solid .header2-theme-toggle:hover,
        .header2-solid .header2-mobile-btn:hover {
            background-color: #e5e7eb !important;
        }

        /* Dark mode - Solid header */
        body.dark .header2-solid .header2-pill {
            background: #0f172a !important;
            border-color: rgba(255, 255, 255, 0.1) !important;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.3), 0 2px 4px -1px rgba(0, 0, 0, 0.2) !important;
        }

        body.dark .header2-solid .header2-logo-text {
            color: #ffffff !important;
        }

        body.dark .header2-solid .header2-nav-link {
            color: rgba(255, 255, 255, 0.8) !important;
        }

        body.dark .header2-solid .header2-nav-link:hover {
            color: var(--color-primary-400) !important;
            background-color: rgba(37, 99, 235, 0.1) !important;
        }

        body.dark .header2-solid .header2-lang-btn,
        body.dark .header2-solid .header2-lang-btn-mobile,
        body.dark .header2-solid .header2-theme-toggle,
        body.dark .header2-solid .header2-mobile-btn {
            background-color: var(--color-bg-tertiary) !important;
            color: rgba(255, 255, 255, 0.8) !important;
        }

        body.dark .header2-solid .header2-lang-btn:hover,
        body.dark .header2-solid .header2-lang-btn-mobile:hover,
        body.dark .header2-solid .header2-theme-toggle:hover,
        body.dark .header2-solid .header2-mobile-btn:hover {
            background-color: #475569 !important;
            color: #ffffff !important;
        }
</style>

@script
    <script>
        // Make closeMobileMenu globally accessible
        window.closeMobileMenu = function() {
            const mobileMenu = document.getElementById('mobile-menu');
            const overlay = document.getElementById('mobile-menu-overlay');

            if (mobileMenu) {
                mobileMenu.classList.remove('active');
            }
            if (overlay) {
                overlay.classList.remove('active');
                // Immediately hide overlay completely to prevent blur
                overlay.style.visibility = 'hidden';
                overlay.style.opacity = '0';
                overlay.style.pointerEvents = 'none';
                overlay.style.backdropFilter = 'none';
                overlay.style.webkitBackdropFilter = 'none';
                overlay.style.background = 'transparent';
            }
            // Always restore body scroll
            document.body.style.overflow = '';
        };

        // Mobile menu toggle
        const menuBtn = document.getElementById('mobile-menu-btn');
        const closeBtn = document.getElementById('mobile-menu-close');
        const mobileMenu = document.getElementById('mobile-menu');
        const overlay = document.getElementById('mobile-menu-overlay');

        if (menuBtn && mobileMenu && overlay) {
            menuBtn.addEventListener('click', () => {
                mobileMenu.classList.add('active');
                overlay.classList.add('active');
                // Reset inline styles when opening
                overlay.style.visibility = '';
                overlay.style.opacity = '';
                overlay.style.pointerEvents = '';
                overlay.style.backdropFilter = '';
                overlay.style.webkitBackdropFilter = '';
                overlay.style.background = '';
                document.body.style.overflow = 'hidden';
            });
        }

        if (closeBtn) {
            closeBtn.addEventListener('click', window.closeMobileMenu);
        }

        if (overlay) {
            overlay.addEventListener('click', window.closeMobileMenu);
        }

        // Close menu when clicking on mobile menu links
        const mobileMenuLinks = document.querySelectorAll('.mobile-menu-link');
        mobileMenuLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                // Always close menu immediately
                window.closeMobileMenu();
            });
        });

        // Reload page after language change
        // Remove 'lang' query parameter if present before reload
        $wire.on('language-changed', () => {
            const url = new URL(window.location.href);
            if (url.searchParams.has('lang')) {
                url.searchParams.delete('lang');
                window.location.href = url.toString();
            } else {
                window.location.reload();
            }
        });
    </script>
@endscript

</div>
