<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $company = [];
    public array $nav = [];
    public array $navItems = [];
    public array $languages = [];
    public ?string $currentLangKey = 'en';
    public ?string $websiteSlug = null;
    public string $homeUrl = '';

    public function mount(array $section = [], array $company = [], array $nav = [], array $navItems = [], array $languages = [], ?string $currentLangKey = 'en', ?string $websiteSlug = null)
    {
        $this->section = $section;
        $this->company = $company;
        $this->nav = $nav;
        $this->navItems = $navItems;
        $this->languages = $languages;
        $this->currentLangKey = $currentLangKey;
        $this->websiteSlug = $websiteSlug;

        // Generate home URL using global helper (handles is_primary for root URL)
        $frontWebsite = view()->shared('frontWebsite');
        $this->homeUrl = landing_home_url($frontWebsite, $this->websiteSlug);
    }

    /**
     * Change the frontend language
     * Stores in cookie with slug-based key (like theme mode), then triggers page reload
     * Cookie key format: lang_{slug} or lang_default
     */
    public function changeLanguage(string $langKey): void
    {
        // Use slug-based cookie key (same pattern as theme mode: theme_{slug})
        $langCookieKey = 'lang_' . ($this->websiteSlug ?? 'default');

        // Set cookie for persistence (1 year)
        cookie()->queue($langCookieKey, $langKey, 60 * 24 * 365);

        // Dispatch browser event to reload the page
        $this->dispatch('language-changed');
    }
};
?>

@php
    $textColorMode = $section['text_color_mode'] ?? 'auto';
    $textColorClass = match($textColorMode) {
        'light' => 'header3-text-light',
        'dark' => 'header3-text-dark',
        default => '',
    };
    $isSolidHeader = !($section['transparent'] ?? true);
@endphp
<div>
{{-- Header Style 3: Split Layout with Contact Bar --}}
<div id="header" class="{{ $section['sticky'] ?? true ? 'fixed' : 'relative' }} top-0 left-0 right-0 z-50 transition-all duration-300 {{ $isSolidHeader ? 'header3-solid' : 'header3-transparent' }} {{ $textColorClass }}"
    data-transparent="{{ $isSolidHeader ? 'false' : 'true' }}"
    data-text-color-mode="{{ $textColorMode }}"
    data-solid="{{ $isSolidHeader ? 'true' : 'false' }}">
    {{-- Contact Info Bar --}}
    <div class="header3-contact-bar">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex items-center justify-between h-12 text-sm">
                {{-- Left: Address --}}
                <div class="flex items-center gap-6">
                    @if (!empty($company['address']))
                        <div class="hidden sm:flex items-center gap-2 header3-welcome">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z">
                                </path>
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                            </svg>
                            <span>{{ $company['address'] }}</span>
                        </div>
                    @endif
                </div>

                {{-- Right: Contact & Social --}}
                <div class="flex items-center gap-4">
                    @if (!empty($company['phone']))
                        <a href="tel:{{ $company['phone'] }}" class="header3-contact-link flex items-center gap-2">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z">
                                </path>
                            </svg>
                            <span class="hidden md:inline">{{ $company['phone'] }}</span>
                        </a>
                    @endif
                    @if (!empty($company['email']))
                        <a href="mailto:{{ $company['email'] }}"
                            class="header3-contact-link hidden md:flex items-center gap-2">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z">
                                </path>
                            </svg>
                            <span>{{ $company['email'] }}</span>
                        </a>
                    @endif
                </div>
            </div>
        </div>
    </div>

    {{-- Main Header --}}
    <header class="header3-main">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex items-center justify-between h-20">
                {{-- Left: Logo & Tagline with theme mode support based on header_logo_type setting --}}
                @php
                    $headerLogoType = $company['header_logo_type'] ?? 'logo_with_name';
                    $showName = in_array($headerLogoType, ['logo_with_name', 'light_dark_with_name']);
                    $useFullLogo = in_array($headerLogoType, ['light_dark_with_name', 'light_dark_only']);
                @endphp
                <div class="flex items-center gap-4">
                    <a href="{{ $homeUrl }}" wire:navigate class="flex items-center gap-3 group">
                        @if ($useFullLogo)
                            {{-- Full Logo (Light/Dark based on theme) --}}
                            <img src="{{ $company['light_logo'] ?? asset('images/light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                                class="h-12 w-auto transition-transform group-hover:scale-105 dark:hidden" />
                            <img src="{{ $company['dark_logo'] ?? asset('images/dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                                class="h-12 w-auto transition-transform group-hover:scale-105 hidden dark:block" />
                        @else
                            {{-- Small Logo (Light/Dark based on theme) --}}
                            <img src="{{ $company['light_logo_small'] ?? asset('images/small_light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                                class="h-10 w-auto transition-transform group-hover:scale-105 dark:hidden" />
                            <img src="{{ $company['dark_logo_small'] ?? asset('images/small_dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                                class="h-10 w-auto transition-transform group-hover:scale-105 hidden dark:block" />
                        @endif

                        @if ($showName)
                            <div class="flex flex-col">
                                <span
                                    class="header3-logo-text text-xl font-bold">{{ $company['name'] ?? 'Washville' }}</span>
                                <span
                                    class="header3-tagline text-xs">{{ $company['tagline'] ?? 'Premium Laundry Service' }}</span>
                            </div>
                        @endif
                    </a>
                </div>

                {{-- Center: Desktop Navigation --}}
                <nav class="hidden lg:flex items-center gap-2">
                    @forelse ($navItems as $index => $item)
                        @if ($index < 4)
                            <a href="{{ $item['url'] }}" wire:navigate
                                class="header3-nav-link" data-page="{{ $item['page_key'] }}">
                                {{ $item['name'] }}
                            </a>
                        @endif
                    @empty
                        <a href="{{ $homeUrl }}#services" wire:navigate class="header3-nav-link"
                            data-section="services">
                            {{ $nav['services'] ?? 'Services' }}
                        </a>
                        <a href="{{ $homeUrl }}#how-it-works" wire:navigate class="header3-nav-link"
                            data-section="how-it-works">
                            {{ $nav['how_it_works'] ?? 'How It Works' }}
                        </a>
                        <a href="{{ $homeUrl }}#pricing" wire:navigate class="header3-nav-link"
                            data-section="pricing">
                            {{ $nav['pricing'] ?? 'Pricing' }}
                        </a>
                        <a href="{{ $homeUrl }}#testimonials" wire:navigate class="header3-nav-link"
                            data-section="testimonials">
                            {{ $nav['reviews'] ?? 'Reviews' }}
                        </a>
                        <a href="{{ $homeUrl }}#faq" wire:navigate class="header3-nav-link" data-section="faq">
                            {{ $nav['faq'] ?? 'FAQ' }}
                        </a>
                        <a href="{{ $homeUrl }}#contact" wire:navigate class="header3-nav-link"
                            data-section="contact">
                            {{ $nav['contact'] ?? 'Contact' }}
                        </a>
                    @endforelse

                    {{-- More Dropdown for remaining items --}}
                    @if (count($navItems) > 4)
                        <div class="relative" x-data="{ open: false }">
                            <button @click="open = !open" @click.away="open = false"
                                class="header3-nav-link flex items-center gap-1.5">
                                {{ $section['more_text'] ?? 'More' }}
                                <svg class="w-4 h-4 transition-transform duration-200" :class="{ 'rotate-180': open }"
                                    fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M19 9l-7 7-7-7"></path>
                                </svg>
                            </button>
                            <div x-show="open"
                                x-transition:enter="transition ease-out duration-200"
                                x-transition:enter-start="opacity-0 translate-y-2"
                                x-transition:enter-end="opacity-100 translate-y-0"
                                x-transition:leave="transition ease-in duration-150"
                                x-transition:leave-start="opacity-100 translate-y-0"
                                x-transition:leave-end="opacity-0 translate-y-2"
                                class="header3-more-dropdown absolute top-full left-0 mt-2 min-w-[180px] rounded-lg shadow-lg bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 z-50 py-1.5 overflow-hidden"
                                style="display: none;">
                                @foreach ($navItems as $index => $item)
                                    @if ($index >= 4)
                                        <a href="{{ $item['url'] }}" wire:navigate
                                            data-page="{{ $item['page_key'] }}"
                                            class="header3-more-item group flex items-center justify-between gap-2 px-4 py-2.5 text-sm text-gray-600 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700/50 hover:text-primary-600 dark:hover:text-primary-400 transition-colors duration-150">
                                            <span>{{ $item['name'] }}</span>
                                            <svg class="w-3.5 h-3.5 opacity-0 -translate-x-1 group-hover:opacity-100 group-hover:translate-x-0 transition-all duration-150 text-primary-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                                            </svg>
                                        </a>
                                    @endif
                                @endforeach
                            </div>
                        </div>
                    @endif
                </nav>

                {{-- Right: Actions --}}
                <div class="flex items-center gap-3">
                    {{-- Language Dropdown (Desktop) --}}
                    @if (count($languages ?? []) > 1)
                        <div class="relative hidden lg:block" x-data="{ open: false }">
                            <button @click="open = !open" @click.away="open = false" class="header3-lang-btn"
                                aria-label="Select language">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9a9 9 0 01-9-9m9 9c1.657 0 3-4.03 3-9s-1.343-9-3-9m0 18c-1.657 0-3-4.03-3-9s1.343-9 3-9m-9 9a9 9 0 019-9">
                                    </path>
                                </svg>
                                <span class="uppercase text-sm font-medium">{{ $currentLangKey }}</span>
                            </button>
                            <div x-show="open" x-transition class="header3-lang-dropdown" style="display: none;">
                                @foreach ($languages as $lang)
                                    <button type="button" wire:click="changeLanguage('{{ $lang['key'] }}')"
                                        class="w-full text-left px-4 py-2.5 text-sm hover:bg-gray-100 dark:hover:bg-gray-700 flex items-center gap-2 {{ $currentLangKey === $lang['key'] ? 'bg-primary-50 dark:bg-primary-900/20 text-primary-600 font-medium' : 'text-gray-700 dark:text-gray-300' }}">
                                        @if (!empty($lang['flag_emoji']))
                                            <span class="text-base">{{ $lang['flag_emoji'] }}</span>
                                        @endif
                                        <span>{{ $lang['name'] }}</span>
                                        @if ($currentLangKey === $lang['key'])
                                            <svg class="w-4 h-4 ml-auto text-primary-600" fill="none"
                                                stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M5 13l4 4L19 7"></path>
                                            </svg>
                                        @endif
                                    </button>
                                @endforeach
                            </div>
                        </div>
                    @endif

                    {{-- Theme Toggle (Desktop) - Hidden for now, only light mode supported
                    @if ($section['show_theme_toggle'] ?? true)
                        <button id="theme-toggle" class="header3-theme-toggle hidden lg:flex"
                            aria-label="Toggle theme">
                            <svg class="sun-icon w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z">
                                </path>
                            </svg>
                            <svg class="moon-icon w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z">
                                </path>
                            </svg>
                        </button>
                    @endif
                    --}}

                    {{-- CTA Buttons (Desktop) --}}
                    @if (!empty($section['cta_buttons']) && is_array($section['cta_buttons']))
                        @foreach ($section['cta_buttons'] as $btnIndex => $ctaBtn)
                            @if (is_array($ctaBtn) && (!empty($ctaBtn['text']) || !empty($ctaBtn['icon'])))
                                <a href="{{ $ctaBtn['url'] ?? '#contact' }}"
                                    @if (str_starts_with($ctaBtn['url'] ?? '#contact', '#')) wire:navigate data-section="{{ ltrim($ctaBtn['url'] ?? '#contact', '#') }}" @endif
                                    class="header3-cta-btn hidden lg:flex items-center gap-2">
                                    @if (!empty($ctaBtn['icon']))<x-icon :icon="$ctaBtn['icon']" class="w-5 h-5" />@endif
                                    @if (!empty($ctaBtn['text']))<span>{{ $ctaBtn['text'] }}</span>@endif
                                </a>
                            @endif
                        @endforeach
                    @endif

                    {{-- Mobile: Menu Button --}}
                    <div class="flex items-center gap-2 lg:hidden">
                        {{-- Mobile Theme Toggle - Hidden for now, only light mode supported
                        @if ($section['show_theme_toggle'] ?? true)
                            <button id="theme-toggle-mobile" class="header3-theme-toggle-mobile"
                                aria-label="Toggle theme">
                                <svg class="sun-icon w-5 h-5" fill="none" stroke="currentColor"
                                    viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z">
                                    </path>
                                </svg>
                                <svg class="moon-icon w-5 h-5" fill="none" stroke="currentColor"
                                    viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z">
                                    </path>
                                </svg>
                            </button>
                        @endif
                        --}}
                        <button id="mobile-menu-btn" class="header3-mobile-btn lg:hidden">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M4 6h16M4 12h16M4 18h16">
                                </path>
                            </svg>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </header>
</div>

{{-- Mobile Menu --}}
<div id="mobile-menu" class="header3-mobile-menu">
    <div class="flex flex-col h-full">
        {{-- Mobile Header with header_logo_type support --}}
        @php
            $mobileHeaderLogoType = $company['header_logo_type'] ?? 'logo_with_name';
            $mobileShowName = in_array($mobileHeaderLogoType, ['logo_with_name', 'light_dark_with_name']);
            $mobileUseFullLogo = in_array($mobileHeaderLogoType, ['light_dark_with_name', 'light_dark_only']);
        @endphp
        <div class="header3-mobile-header">
            <div class="flex items-center gap-3">
                @if ($mobileUseFullLogo)
                    {{-- Full Logo for mobile (using dark logo since mobile menu has dark background) --}}
                    <img src="{{ $company['dark_logo'] ?? asset('images/dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-10 w-auto" />
                @else
                    {{-- Small Logo for mobile (using dark logo since mobile menu has dark background) --}}
                    <img src="{{ $company['dark_logo_small'] ?? asset('images/small_dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto" />
                @endif

                @if ($mobileShowName)
                    <div>
                        <div class="text-lg font-bold">{{ $company['name'] ?? 'Washville' }}</div>
                        <div class="text-xs opacity-90">{{ $company['tagline'] ?? 'Premium Service' }}</div>
                    </div>
                @endif
            </div>
            <button id="mobile-menu-close" class="header3-mobile-close">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                    </path>
                </svg>
            </button>
        </div>

        {{-- Mobile Navigation --}}
        <nav class="flex-1 overflow-y-auto py-4">
            @forelse ($navItems as $item)
                <a href="{{ $item['url'] }}" data-page="{{ $item['page_key'] }}"
                    class="header3-mobile-link mobile-menu-link">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7">
                        </path>
                    </svg>
                    <span>{{ $item['name'] }}</span>
                </a>
            @empty
                <a href="{{ $homeUrl }}#services" data-section="services"
                    class="header3-mobile-link mobile-menu-link">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7">
                        </path>
                    </svg>
                    <span>{{ $nav['services'] ?? 'Services' }}</span>
                </a>
                <a href="{{ $homeUrl }}#how-it-works" data-section="how-it-works"
                    class="header3-mobile-link mobile-menu-link">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7">
                        </path>
                    </svg>
                    <span>{{ $nav['how_it_works'] ?? 'How It Works' }}</span>
                </a>
                <a href="{{ $homeUrl }}#pricing" data-section="pricing"
                    class="header3-mobile-link mobile-menu-link">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7">
                        </path>
                    </svg>
                    <span>{{ $nav['pricing'] ?? 'Pricing' }}</span>
                </a>
                <a href="{{ $homeUrl }}#testimonials" data-section="testimonials"
                    class="header3-mobile-link mobile-menu-link">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7">
                        </path>
                    </svg>
                    <span>{{ $nav['reviews'] ?? 'Reviews' }}</span>
                </a>
                <a href="{{ $homeUrl }}#faq" data-section="faq" class="header3-mobile-link mobile-menu-link">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7">
                        </path>
                    </svg>
                    <span>{{ $nav['faq'] ?? 'FAQ' }}</span>
                </a>
                <a href="{{ $homeUrl }}#contact" data-section="contact"
                    class="header3-mobile-link mobile-menu-link">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7">
                        </path>
                    </svg>
                    <span>{{ $nav['contact'] ?? 'Contact' }}</span>
                </a>
            @endforelse
        </nav>

        {{-- Mobile Language Selector --}}
        @if (count($languages ?? []) > 1)
            <div class="px-4 py-3 border-t border-gray-200 dark:border-gray-700">
                <p class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-3">Language</p>
                <div class="flex flex-wrap gap-2">
                    @foreach ($languages as $lang)
                        <button type="button" wire:click="changeLanguage('{{ $lang['key'] }}')"
                            class="px-3 py-2 text-sm rounded-lg flex items-center gap-2 transition-colors {{ $currentLangKey === $lang['key'] ? 'bg-primary-600 text-white' : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600' }}">
                            @if (!empty($lang['flag_emoji']))
                                <span>{{ $lang['flag_emoji'] }}</span>
                            @endif
                            <span>{{ $lang['name'] }}</span>
                        </button>
                    @endforeach
                </div>
            </div>
        @endif

        {{-- Mobile Footer --}}
        <div class="header3-mobile-footer">
            {{-- Contact Info --}}
            <div class="space-y-2 mb-4">
                @if (!empty($company['phone']))
                    <a href="tel:{{ $company['phone'] }}" class="header3-mobile-contact">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z">
                            </path>
                        </svg>
                        <span>{{ $company['phone'] }}</span>
                    </a>
                @endif
                @if (!empty($company['email']))
                    <a href="mailto:{{ $company['email'] }}" class="header3-mobile-contact">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z">
                            </path>
                        </svg>
                        <span>{{ $company['email'] }}</span>
                    </a>
                @endif
            </div>

            {{-- CTA Buttons --}}
            @if (!empty($section['cta_buttons']) && is_array($section['cta_buttons']))
                <div class="space-y-2">
                    @foreach ($section['cta_buttons'] as $btnIndex => $ctaBtn)
                        @if (is_array($ctaBtn) && (!empty($ctaBtn['text']) || !empty($ctaBtn['icon'])))
                            <a href="{{ $ctaBtn['url'] ?? '#contact' }}"
                                @if (str_starts_with($ctaBtn['url'] ?? '#contact', '#')) data-section="{{ ltrim($ctaBtn['url'] ?? '#contact', '#') }}" @endif
                                class="header3-mobile-cta mobile-menu-link flex items-center justify-center gap-2">
                                @if (!empty($ctaBtn['icon']))<x-icon :icon="$ctaBtn['icon']" class="w-5 h-5" />@endif
                                @if (!empty($ctaBtn['text']))<span>{{ $ctaBtn['text'] }}</span>@endif
                            </a>
                        @endif
                    @endforeach
                </div>
            @endif
        </div>
    </div>
</div>

{{-- Mobile Menu Overlay --}}
<div id="mobile-menu-overlay" class="header3-overlay"></div>
</div>

<style>
        /* Header 3: Split Layout with Contact Bar */

        /* ===== Contact Info Bar ===== */
        .header3-contact-bar {
            background: linear-gradient(135deg, var(--color-primary-600) 0%, var(--color-secondary-600) 100%);
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .dark .header3-contact-bar {
            background: linear-gradient(135deg, var(--color-primary-700) 0%, var(--color-secondary-700) 100%);
            border-bottom-color: rgba(255, 255, 255, 0.05);
        }

        .header3-welcome {
            color: rgba(255, 255, 255, 0.9);
        }

        .dark .header3-welcome {
            color: rgba(255, 255, 255, 0.85);
        }

        .header3-contact-link {
            color: rgba(255, 255, 255, 0.9);
            transition: color 200ms;
        }

        .dark .header3-contact-link {
            color: rgba(255, 255, 255, 0.85);
        }

        .header3-contact-link:hover {
            color: #ffffff;
        }

        .dark .header3-contact-link:hover {
            color: #ffffff;
        }

        /* ===== Main Header ===== */
        .header3-main {
            background-color: #ffffff;
            border-bottom: 1px solid #e5e7eb;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
        }

        .dark .header3-main {
            background-color: var(--color-bg-primary);
            border-bottom-color: var(--color-border);
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.3);
        }

        /* ===== Logo ===== */
        .header3-logo-icon {
            width: 2.75rem;
            height: 2.75rem;
            border-radius: 0.75rem;
            background: linear-gradient(to bottom right, var(--color-primary-500), var(--color-primary-600), var(--color-primary-700));
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 10px 15px -3px rgba(37, 99, 235, 0.3);
            transition: all 300ms;
        }

        .header3-logo-icon-small {
            width: 2.25rem;
            height: 2.25rem;
            border-radius: 0.5rem;
            background: linear-gradient(to bottom right, var(--color-primary-500), var(--color-primary-600), var(--color-primary-700));
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .header3-logo-text {
            color: #111827;
        }

        .dark .header3-logo-text {
            color: var(--color-text-primary);
        }

        .header3-tagline {
            color: #6b7280;
            font-weight: 500;
        }

        .dark .header3-tagline {
            color: var(--color-text-tertiary);
        }

        /* ===== Desktop Navigation ===== */
        .header3-nav-link {
            padding: 0.625rem 1rem;
            font-size: 1rem;
            font-weight: 500;
            color: #374151;
            border-radius: 0.5rem;
            transition: all 200ms;
            position: relative;
        }

        .dark .header3-nav-link {
            color: var(--color-text-secondary);
        }

        .header3-nav-link:hover {
            color: var(--color-primary-600);
            background-color: var(--color-primary-50);
        }

        .dark .header3-nav-link:hover {
            color: var(--color-primary-400);
            background-color: rgba(37, 99, 235, 0.1);
        }

        .header3-nav-link::after {
            content: '';
            position: absolute;
            bottom: 0.25rem;
            left: 50%;
            transform: translateX(-50%);
            width: 0;
            height: 2px;
            background-color: var(--color-primary-600);
            transition: all 300ms;
        }

        .header3-nav-link:hover::after {
            width: 2rem;
        }

        /* ===== More Dropdown ===== */
        .header3-more-dropdown {
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
        }

        .header3-more-dropdown::before {
            content: '';
            position: absolute;
            top: -6px;
            left: 20px;
            border-left: 6px solid transparent;
            border-right: 6px solid transparent;
            border-bottom: 6px solid white;
        }

        .dark .header3-more-dropdown::before {
            border-bottom-color: var(--color-gray-800);
        }

        .header3-more-item:hover span:first-child {
            opacity: 1;
        }

        /* ===== Language Dropdown ===== */
        .header3-lang-btn {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 0.75rem;
            border-radius: 0.5rem;
            background-color: #f3f4f6;
            color: #374151;
            transition: all 200ms;
        }

        .dark .header3-lang-btn {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .header3-lang-btn:hover {
            background-color: #e5e7eb;
        }

        .dark .header3-lang-btn:hover {
            background-color: #475569;
        }

        .header3-lang-dropdown {
            position: absolute;
            right: 0;
            margin-top: 0.5rem;
            width: 11rem;
            border-radius: 0.75rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
            background-color: #ffffff;
            border: 1px solid #e5e7eb;
            padding: 0.5rem 0;
            z-index: 50;
        }

        .dark .header3-lang-dropdown {
            background-color: var(--color-card-bg);
            border-color: var(--color-border);
        }

        /* ===== Theme Toggle ===== */
        .header3-theme-toggle,
        .header3-theme-toggle-mobile {
            padding: 0.625rem;
            border-radius: 0.5rem;
            background-color: #f3f4f6;
            color: #374151;
            transition: all 200ms;
        }

        .dark .header3-theme-toggle,
        .dark .header3-theme-toggle-mobile {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .header3-theme-toggle:hover,
        .header3-theme-toggle-mobile:hover {
            background-color: #e5e7eb;
        }

        .dark .header3-theme-toggle:hover,
        .dark .header3-theme-toggle-mobile:hover {
            background-color: #475569;
        }

        .header3-theme-toggle .sun-icon,
        .header3-theme-toggle-mobile .sun-icon {
            display: none;
        }

        .dark .header3-theme-toggle .sun-icon,
        .dark .header3-theme-toggle-mobile .sun-icon {
            display: block;
        }

        .header3-theme-toggle .moon-icon,
        .header3-theme-toggle-mobile .moon-icon {
            display: block;
        }

        .dark .header3-theme-toggle .moon-icon,
        .dark .header3-theme-toggle-mobile .moon-icon {
            display: none;
        }

        /* ===== CTA Button ===== */
        .header3-cta-btn {
            align-items: center;
            justify-content: center;
            padding: 0.625rem 1.25rem;
            border-radius: 0.5rem;
            font-weight: 600;
            font-size: 0.875rem;
            background: linear-gradient(135deg, var(--color-primary-600), var(--color-primary-700));
            color: white;
            box-shadow: 0 4px 14px rgba(37, 99, 235, 0.4);
            transition: all 300ms;
            white-space: nowrap;
        }

        .header3-cta-btn:hover {
            background: linear-gradient(135deg, var(--color-primary-700), var(--color-primary-800));
            box-shadow: 0 6px 20px rgba(37, 99, 235, 0.5);
            transform: translateY(-2px);
        }

        /* ===== Mobile Menu Button ===== */
        .header3-mobile-btn {
            padding: 0.5rem;
            border-radius: 0.5rem;
            background-color: #f3f4f6;
            color: #374151;
            transition: color 200ms;
        }

        .dark .header3-mobile-btn {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .header3-mobile-btn:hover {
            background-color: #e5e7eb;
        }

        .dark .header3-mobile-btn:hover {
            background-color: #475569;
        }

        /* ===== Mobile Menu ===== */
        .header3-mobile-menu {
            position: fixed;
            top: 0;
            bottom: 0;
            right: 0;
            width: 100%;
            max-width: 24rem;
            background-color: #ffffff;
            box-shadow: -10px 0 25px -5px rgba(0, 0, 0, 0.1);
            transform: translateX(100%);
            transition: transform 300ms cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 50;
        }

        .dark .header3-mobile-menu {
            background-color: var(--color-bg-primary);
        }

        .header3-mobile-menu.active {
            transform: translateX(0);
        }

        /* ===== Mobile Menu Header ===== */
        .header3-mobile-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 1.25rem;
            border-bottom: 1px solid #e5e7eb;
            background: linear-gradient(to right, var(--color-primary-50), var(--color-primary-100));
            color: #111827;
        }

        .dark .header3-mobile-header {
            border-bottom-color: var(--color-border);
            background: linear-gradient(to right, var(--color-bg-secondary), var(--color-bg-tertiary));
            color: var(--color-text-primary);
        }

        .header3-mobile-close {
            padding: 0.5rem;
            border-radius: 0.5rem;
            color: var(--color-primary-700);
            transition: background-color 200ms;
        }

        .dark .header3-mobile-close {
            color: var(--color-primary-400);
        }

        .header3-mobile-close:hover {
            background-color: rgba(255, 255, 255, 0.5);
        }

        .dark .header3-mobile-close:hover {
            background-color: rgba(31, 41, 55, 0.5);
        }

        /* ===== Mobile Navigation ===== */
        .header3-mobile-link {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 1rem 1.5rem;
            color: #374151;
            border-bottom: 1px solid #f3f4f6;
            transition: all 200ms;
        }

        .dark .header3-mobile-link {
            color: var(--color-text-secondary);
            border-bottom-color: var(--color-border);
        }

        .header3-mobile-link:hover {
            background-color: var(--color-primary-50);
            color: var(--color-primary-600);
        }

        .dark .header3-mobile-link:hover {
            background-color: rgba(37, 99, 235, 0.1);
            color: var(--color-primary-400);
        }

        /* ===== Mobile Footer ===== */
        .header3-mobile-footer {
            padding: 1rem 1.5rem;
            border-top: 1px solid #e5e7eb;
            background-color: #f9fafb;
        }

        .dark .header3-mobile-footer {
            border-top-color: var(--color-border);
            background-color: var(--color-bg-secondary);
        }

        .header3-mobile-contact {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.875rem;
            color: #6b7280;
            transition: color 200ms;
        }

        .dark .header3-mobile-contact {
            color: var(--color-text-tertiary);
        }

        .header3-mobile-contact:hover {
            color: var(--color-primary-600);
        }

        .dark .header3-mobile-contact:hover {
            color: var(--color-primary-400);
        }

        .header3-mobile-cta {
            width: 100%;
            padding: 0.75rem 1.5rem;
            border-radius: 0.5rem;
            font-weight: 600;
            background: linear-gradient(to right, var(--color-primary-600), var(--color-primary-700));
            color: white;
            box-shadow: 0 10px 15px -3px rgba(37, 99, 235, 0.25);
            transition: all 300ms;
        }

        .header3-mobile-cta:hover {
            background: linear-gradient(to right, var(--color-primary-700), var(--color-primary-800));
        }

        /* ===== Mobile Menu Overlay ===== */
        .header3-overlay {
            position: fixed;
            inset: 0;
            background-color: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(4px);
            opacity: 0;
            pointer-events: none;
            transition: opacity 300ms;
            z-index: 40;
        }

        .header3-overlay.active {
            opacity: 1;
            pointer-events: auto;
        }

        /* ===== TRANSPARENT MODE ===== */
        .header3-transparent .header3-contact-bar {
            background: rgba(0, 0, 0, 0.2) !important;
            backdrop-filter: blur(8px);
            border-bottom: 1px solid rgba(255, 255, 255, 0.1) !important;
        }

        .header3-transparent .header3-main {
            background: transparent !important;
            background-color: transparent !important;
            box-shadow: none !important;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1) !important;
        }

        .header3-transparent .header3-logo-text {
            color: white;
        }

        .header3-transparent .header3-tagline {
            color: rgba(255, 255, 255, 0.8);
        }

        .header3-transparent .header3-nav-link {
            color: rgba(255, 255, 255, 0.9);
        }

        .header3-transparent .header3-nav-link:hover {
            color: white;
            background-color: rgba(255, 255, 255, 0.1);
        }

        .header3-transparent .header3-nav-link::after {
            background-color: white;
        }

        .header3-transparent .header3-lang-btn {
            background-color: rgba(255, 255, 255, 0.15);
            color: rgba(255, 255, 255, 0.9);
        }

        .header3-transparent .header3-lang-btn:hover {
            background-color: rgba(255, 255, 255, 0.25);
            color: white;
        }

        .header3-transparent .header3-theme-toggle,
        .header3-transparent .header3-theme-toggle-mobile {
            background-color: rgba(255, 255, 255, 0.15);
            color: rgba(255, 255, 255, 0.9);
        }

        .header3-transparent .header3-theme-toggle:hover,
        .header3-transparent .header3-theme-toggle-mobile:hover {
            background-color: rgba(255, 255, 255, 0.25);
            color: white;
        }

        .header3-transparent .header3-mobile-btn {
            background-color: rgba(255, 255, 255, 0.15);
            color: white;
        }

        .header3-transparent .header3-mobile-btn:hover {
            background-color: rgba(255, 255, 255, 0.25);
        }

        /* Transparent mode when scrolled */
        .header3-transparent.header-scrolled .header3-contact-bar {
            background: linear-gradient(135deg, var(--color-primary-600) 0%, var(--color-secondary-600) 100%) !important;
            backdrop-filter: none !important;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1) !important;
        }

        .header3-transparent.header-scrolled .header3-main {
            background-color: #ffffff !important;
            backdrop-filter: blur(10px) !important;
            border-bottom: 1px solid #e5e7eb !important;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1) !important;
        }

        .header3-transparent.header-scrolled .header3-logo-text {
            color: #111827;
        }

        .header3-transparent.header-scrolled .header3-tagline {
            color: #6b7280;
        }

        .header3-transparent.header-scrolled .header3-nav-link {
            color: #374151;
        }

        .header3-transparent.header-scrolled .header3-nav-link:hover {
            color: var(--color-primary-600);
            background-color: var(--color-primary-50);
        }

        .header3-transparent.header-scrolled .header3-nav-link::after {
            background-color: var(--color-primary-600);
        }

        .header3-transparent.header-scrolled .header3-lang-btn {
            background-color: #f3f4f6;
            color: #374151;
        }

        .header3-transparent.header-scrolled .header3-lang-btn:hover {
            background-color: #e5e7eb;
        }

        .header3-transparent.header-scrolled .header3-theme-toggle,
        .header3-transparent.header-scrolled .header3-theme-toggle-mobile {
            background-color: #f3f4f6;
            color: #374151;
        }

        .header3-transparent.header-scrolled .header3-theme-toggle:hover,
        .header3-transparent.header-scrolled .header3-theme-toggle-mobile:hover {
            background-color: #e5e7eb;
        }

        .header3-transparent.header-scrolled .header3-mobile-btn {
            background-color: #f3f4f6;
            color: #374151;
        }

        .header3-transparent.header-scrolled .header3-mobile-btn:hover {
            background-color: #e5e7eb;
        }

        /* Dark mode - Transparent scrolled */
        .dark .header3-transparent.header-scrolled .header3-contact-bar {
            background: linear-gradient(135deg, var(--color-primary-700) 0%, var(--color-secondary-700) 100%);
            border-bottom-color: rgba(255, 255, 255, 0.05);
        }

        .dark .header3-transparent.header-scrolled .header3-main {
            background-color: var(--color-bg-primary) !important;
            border-bottom-color: var(--color-border) !important;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.3) !important;
        }

        .dark .header3-transparent.header-scrolled .header3-logo-text {
            color: var(--color-text-primary);
        }

        .dark .header3-transparent.header-scrolled .header3-tagline {
            color: var(--color-text-tertiary);
        }

        .dark .header3-transparent.header-scrolled .header3-nav-link {
            color: var(--color-text-secondary);
        }

        .dark .header3-transparent.header-scrolled .header3-nav-link:hover {
            color: var(--color-primary-400);
            background-color: rgba(37, 99, 235, 0.1);
        }

        .dark .header3-transparent.header-scrolled .header3-lang-btn {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .dark .header3-transparent.header-scrolled .header3-lang-btn:hover {
            background-color: #475569;
        }

        .dark .header3-transparent.header-scrolled .header3-theme-toggle,
        .dark .header3-transparent.header-scrolled .header3-theme-toggle-mobile {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .dark .header3-transparent.header-scrolled .header3-theme-toggle:hover,
        .dark .header3-transparent.header-scrolled .header3-theme-toggle-mobile:hover {
            background-color: #475569;
        }

        .dark .header3-transparent.header-scrolled .header3-mobile-btn {
            background-color: var(--color-bg-tertiary);
            color: var(--color-text-secondary);
        }

        .dark .header3-transparent.header-scrolled .header3-mobile-btn:hover {
            background-color: #475569;
        }

        /* ===== SOLID MODE (Non-Transparent) ===== */
        /* Light mode solid header */
        .header3-solid .header3-main {
            background-color: rgba(255, 255, 255, 0.95) !important;
            border-bottom: 1px solid rgba(229, 231, 235, 0.8);
            backdrop-filter: blur(10px);
        }

        .header3-solid .header3-logo-text {
            color: #111827 !important;
        }

        .header3-solid .header3-tagline {
            color: #6b7280 !important;
        }

        .header3-solid .header3-nav-link {
            color: #374151 !important;
        }

        .header3-solid .header3-nav-link:hover {
            color: var(--color-primary-600) !important;
        }

        .header3-solid .header3-lang-btn,
        .header3-solid .header3-theme-toggle,
        .header3-solid .header3-mobile-btn {
            background-color: #f3f4f6 !important;
            color: #374151 !important;
        }

        .header3-solid .header3-lang-btn:hover,
        .header3-solid .header3-theme-toggle:hover,
        .header3-solid .header3-mobile-btn:hover {
            background-color: #e5e7eb !important;
        }

        /* Dark mode solid header */
        body.dark .header3-solid .header3-main {
            background-color: #0f172a !important;
            border-bottom-color: rgba(255, 255, 255, 0.1) !important;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.3) !important;
        }

        body.dark .header3-solid .header3-logo-text {
            color: #ffffff !important;
        }

        body.dark .header3-solid .header3-tagline {
            color: rgba(255, 255, 255, 0.7) !important;
        }

        body.dark .header3-solid .header3-nav-link {
            color: rgba(255, 255, 255, 0.8) !important;
        }

        body.dark .header3-solid .header3-nav-link:hover {
            color: var(--color-primary-400) !important;
        }

        body.dark .header3-solid .header3-lang-btn,
        body.dark .header3-solid .header3-theme-toggle,
        body.dark .header3-solid .header3-mobile-btn {
            background-color: var(--color-bg-tertiary) !important;
            color: rgba(255, 255, 255, 0.8) !important;
        }

        body.dark .header3-solid .header3-lang-btn:hover,
        body.dark .header3-solid .header3-theme-toggle:hover,
        body.dark .header3-solid .header3-mobile-btn:hover {
            background-color: #475569 !important;
            color: #ffffff !important;
        }

        /* Dark mode for ANY header when scrolled (both transparent and solid) */
        /* This ensures dark background is maintained when scrolling */
        body.dark #header.header-scrolled .header3-main,
        body.dark .header3-solid.header-scrolled .header3-main {
            background-color: #0f172a !important;
            border-bottom-color: rgba(255, 255, 255, 0.1) !important;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.3) !important;
        }

        /* ===== TEXT COLOR MODE ===== */
        /* IMPORTANT: Text color mode settings only apply in LIGHT mode */
        /* In dark mode, colors are always controlled by dark mode variables */

        /* Light text mode (white text) - for dark backgrounds - LIGHT MODE ONLY */
        /* Only applies when NOT scrolled and NOT in dark mode */
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-logo-text,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-tagline,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-nav-link,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-lang-btn,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-theme-toggle,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-theme-toggle-mobile,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-mobile-btn {
            color: rgba(255, 255, 255, 0.95) !important;
        }

        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-nav-link:hover {
            color: #ffffff !important;
            background-color: rgba(255, 255, 255, 0.15) !important;
        }

        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-lang-btn:hover,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-theme-toggle:hover,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-theme-toggle-mobile:hover,
        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-mobile-btn:hover {
            color: #ffffff !important;
            background-color: rgba(255, 255, 255, 0.2) !important;
        }

        body:not(.dark) .header3-text-light.header3-transparent:not(.header-scrolled) .header3-nav-link::after {
            background-color: #ffffff !important;
        }

        /* Dark text mode (black text) - for light backgrounds - LIGHT MODE ONLY */
        /* Only applies when NOT scrolled and NOT in dark mode */
        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-logo-text {
            color: #111827 !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-tagline {
            color: #6b7280 !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-nav-link {
            color: #374151 !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-nav-link:hover {
            color: var(--color-primary-600) !important;
            background-color: rgba(0, 0, 0, 0.05) !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-nav-link::after {
            background-color: var(--color-primary-600) !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-lang-btn {
            color: #374151 !important;
            background-color: rgba(0, 0, 0, 0.05) !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-lang-btn:hover {
            color: var(--color-primary-600) !important;
            background-color: rgba(0, 0, 0, 0.1) !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-theme-toggle,
        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-theme-toggle-mobile {
            color: #374151 !important;
            background-color: rgba(0, 0, 0, 0.05) !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-theme-toggle:hover,
        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-theme-toggle-mobile:hover {
            color: var(--color-primary-600) !important;
            background-color: rgba(0, 0, 0, 0.1) !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-mobile-btn {
            color: #374151 !important;
            background-color: rgba(0, 0, 0, 0.05) !important;
        }

        body:not(.dark) .header3-text-dark.header3-transparent:not(.header-scrolled) .header3-mobile-btn:hover {
            color: var(--color-primary-600) !important;
            background-color: rgba(0, 0, 0, 0.1) !important;
        }

        /* ===== RTL Support ===== */
        [dir="rtl"] .header3-mobile-menu {
            right: auto;
            left: 0;
            transform: translateX(-100%);
        }

        [dir="rtl"] .header3-mobile-menu.active {
            transform: translateX(0);
        }

        [dir="rtl"] .header3-nav-link::after {
            left: auto;
            right: 50%;
            transform: translateX(50%);
        }

        [dir="rtl"] .header3-mobile-link {
            flex-direction: row-reverse;
        }

        /* ===== Responsive Adjustments ===== */
        @media (max-width: 1024px) {
            .header3-contact-bar {
                display: none;
            }
        }

        @media (max-width: 640px) {
            .header3-logo-text {
                font-size: 1.125rem;
                line-height: 1.75rem;
            }

            .header3-tagline {
                display: none;
            }
        }
</style>

@script
    <script>
        // Language change reload
        // Remove 'lang' query parameter if present before reload
        $wire.on('language-changed', () => {
            const url = new URL(window.location.href);
            if (url.searchParams.has('lang')) {
                url.searchParams.delete('lang');
                window.location.href = url.toString();
            } else {
                window.location.reload();
            }
        });

        // Function to close mobile menu
        function closeMobileMenu() {
            const mobileMenu = document.getElementById('mobile-menu');
            const overlay = document.getElementById('mobile-menu-overlay');

            if (mobileMenu) {
                mobileMenu.classList.remove('active');
            }
            if (overlay) {
                overlay.classList.remove('active');
            }
            document.body.style.overflow = '';
        }

        // Mobile menu overlay functionality specific to header3
        const overlay = document.getElementById('mobile-menu-overlay');
        if (overlay) {
            overlay.addEventListener('click', closeMobileMenu);
        }

        // Update mobile menu toggle for header3 to also control overlay
        const menuBtn = document.getElementById('mobile-menu-btn');
        const closeBtn = document.getElementById('mobile-menu-close');
        const mobileMenu = document.getElementById('mobile-menu');

        if (menuBtn && mobileMenu && overlay) {
            menuBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                mobileMenu.classList.add('active');
                overlay.classList.add('active');
                document.body.style.overflow = 'hidden';
            });
        }

        if (closeBtn) {
            closeBtn.addEventListener('click', closeMobileMenu);
        }

        // Close menu when clicking on mobile menu links
        const mobileMenuLinks = document.querySelectorAll('.mobile-menu-link');
        mobileMenuLinks.forEach(link => {
            link.addEventListener('click', () => {
                // Small delay to allow navigation to start
                setTimeout(closeMobileMenu, 100);
            });
        });
    </script>
@endscript

</div>
