<?php

use Livewire\Component;
use App\Models\WebsitePage;
use App\Models\WebsitePageTranslation;
use App\Models\WebsiteLanguage;

new class extends Component {
    public array $section = [];
    public array $company = [];
    public array $nav = [];
    public array $navItems = [];
    public array $languages = [];
    public ?string $currentLangKey = 'en';
    public ?string $websiteSlug = null;
    public string $homeUrl = '';
    public array $headerPages = []; // Business pages with show_in_header = true

    public function mount(array $section = [], array $company = [], array $nav = [], array $navItems = [], array $languages = [], ?string $currentLangKey = 'en', ?string $websiteSlug = null)
    {
        $this->section = $section;
        $this->company = $company;
        $this->nav = $nav;
        $this->navItems = $navItems;
        $this->languages = $languages;
        $this->currentLangKey = $currentLangKey;
        $this->websiteSlug = $websiteSlug;

        // Generate home URL using global helper (handles is_primary for root URL)
        $frontWebsite = view()->shared('frontWebsite');
        $this->homeUrl = landing_home_url($frontWebsite, $this->websiteSlug);

        // Load business pages with show_in_header = true for current language
        $this->loadHeaderPages($frontWebsite);
    }

    /**
     * Load business pages that should be shown in header for the current language
     */
    protected function loadHeaderPages($frontWebsite): void
    {
        if (!$frontWebsite) {
            return;
        }

        // Get the current language ID
        $frontLangId = view()->shared('frontLangId');

        if (!$frontLangId) {
            // Try to get from current lang key
            $lang = WebsiteLanguage::withoutGlobalScopes()
                ->where('key', $this->currentLangKey)
                ->where('company_id', $frontWebsite->company_id)
                ->first();
            $frontLangId = $lang?->id;
        }

        if (!$frontLangId) {
            return;
        }

        // Get business pages with translations that have show_in_header = true
        $pages = WebsitePage::withoutGlobalScopes()
            ->where('website_id', $frontWebsite->id)
            ->whereHas('translations', function ($query) use ($frontLangId) {
                $query->where('website_language_id', $frontLangId)
                    ->where('show_in_header', true);
            })
            ->with(['translations' => function ($query) use ($frontLangId) {
                $query->where('website_language_id', $frontLangId);
            }])
            ->get();

        // Build header pages array
        $this->headerPages = $pages->map(function ($page) {
            $translation = $page->translations->first();
            return [
                'page_key' => $page->page_key,
                'name' => $translation?->name ?? $page->page_key,
                'url' => $this->getPageUrl($page->page_key),
            ];
        })->toArray();
    }

    /**
     * Get the URL for a business page
     */
    protected function getPageUrl(string $pageKey): string
    {
        // Home page goes to home URL
        if ($pageKey === 'home') {
            return $this->homeUrl;
        }

        // Other pages append page_key to home URL
        return rtrim($this->homeUrl, '/') . '/' . $pageKey;
    }

    /**
     * Change the frontend language
     * Stores in cookie with slug-based key (like theme mode), then triggers page reload
     * Cookie key format: lang_{slug} or lang_default
     */
    public function changeLanguage(string $langKey): void
    {
        // Use slug-based cookie key (same pattern as theme mode: theme_{slug})
        $langCookieKey = 'lang_' . ($this->websiteSlug ?? 'default');

        // Set cookie for persistence (1 year)
        cookie()->queue($langCookieKey, $langKey, 60 * 24 * 365);

        // Dispatch browser event to reload the page
        $this->dispatch('language-changed');
    }
};
?>

@php
    $textColorMode = $section['text_color_mode'] ?? 'auto';
    $textColorClass = match($textColorMode) {
        'light' => 'header-text-light',
        'dark' => 'header-text-dark',
        default => '',
    };
    $isSolidHeader = !($section['transparent'] ?? true);
@endphp
<div>
<header id="header"
    class="{{ $section['sticky'] ?? true ? 'fixed' : 'relative' }} top-0 left-0 right-0 z-50 transition-all duration-300 {{ $isSolidHeader ? 'header-solid' : 'header-transparent' }} {{ $textColorClass }}"
    data-transparent="{{ $isSolidHeader ? 'false' : 'true' }}"
    data-text-color-mode="{{ $textColorMode }}"
    data-solid="{{ $isSolidHeader ? 'true' : 'false' }}">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="flex items-center justify-between h-20">
            {{-- Logo with theme mode support based on header_logo_type setting --}}
            @php
                $headerLogoType = $company['header_logo_type'] ?? 'logo_with_name';
                $showName = in_array($headerLogoType, ['logo_with_name', 'light_dark_with_name']);
                $useFullLogo = in_array($headerLogoType, ['light_dark_with_name', 'light_dark_only']);
            @endphp
            <a href="{{ $homeUrl }}" wire:navigate class="flex items-center gap-2">
                @if ($useFullLogo)
                    {{-- Full Logo (Light/Dark based on theme) --}}
                    <img src="{{ $company['light_logo'] ?? asset('images/light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto dark:hidden" />
                    <img src="{{ $company['dark_logo'] ?? asset('images/dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto hidden dark:block" />
                @else
                    {{-- Small Logo (Light/Dark based on theme) --}}
                    <img src="{{ $company['light_logo_small'] ?? asset('images/small_light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto dark:hidden" />
                    <img src="{{ $company['dark_logo_small'] ?? asset('images/small_dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto hidden dark:block" />
                @endif

                @if ($showName)
                    <span class="header-logo-text text-xl font-bold">{{ $company['name'] ?? 'Washville' }}</span>
                @endif
            </a>

            {{-- Desktop Navigation - Business Pages with show_in_header --}}
            @php
                // Show max 4 items in header, rest go to More dropdown
                $maxVisibleItems = 4;
                $totalHeaderPages = count($headerPages);
                $hasMoreHeaderPages = $totalHeaderPages > $maxVisibleItems;
            @endphp
            <nav class="hidden md:flex items-center gap-4 lg:gap-6 flex-nowrap">
                @forelse ($headerPages as $index => $page)
                    @if ($index < $maxVisibleItems)
                        <a href="{{ $page['url'] }}" wire:navigate class="nav-link whitespace-nowrap text-sm lg:text-base">{{ $page['name'] }}</a>
                    @endif
                @empty
                    {{-- No pages configured to show in header --}}
                @endforelse

                {{-- More Dropdown for remaining items --}}
                @if ($hasMoreHeaderPages)
                    <div class="relative" x-data="{ open: false }">
                        <button @click="open = !open" @click.away="open = false"
                            class="nav-link flex items-center gap-1.5 group">
                            {{ $section['more_text'] ?? 'More' }}
                            <x-icon icon="heroicons:chevron-down"
                                class="w-4 h-4 transition-transform duration-200"
                                x-bind:class="{ 'rotate-180': open }" />
                        </button>
                        <div x-show="open"
                            x-transition:enter="transition ease-out duration-200"
                            x-transition:enter-start="opacity-0 translate-y-2"
                            x-transition:enter-end="opacity-100 translate-y-0"
                            x-transition:leave="transition ease-in duration-150"
                            x-transition:leave-start="opacity-100 translate-y-0"
                            x-transition:leave-end="opacity-0 translate-y-2"
                            class="nav-more-dropdown absolute top-full left-1/2 -translate-x-1/2 mt-2 min-w-[180px] rounded-lg shadow-lg bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 z-50 py-1.5 overflow-hidden"
                            style="display: none;">
                            @foreach ($headerPages as $index => $page)
                                @if ($index >= $maxVisibleItems)
                                    <a href="{{ $page['url'] }}" wire:navigate
                                        class="nav-more-item group flex items-center justify-between gap-2 px-4 py-2.5 text-sm text-gray-600 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700/50 hover:text-primary-600 dark:hover:text-primary-400 transition-colors duration-150">
                                        <span>{{ $page['name'] }}</span>
                                        <svg class="w-3.5 h-3.5 opacity-0 -translate-x-1 group-hover:opacity-100 group-hover:translate-x-0 transition-all duration-150 text-primary-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                                        </svg>
                                    </a>
                                @endif
                            @endforeach
                        </div>
                    </div>
                @endif
            </nav>

            {{-- CTA Button & Theme Toggle --}}
            <div class="hidden md:flex items-center gap-4">
                {{-- Language Dropdown --}}
                @if (count($languages ?? []) > 1)
                    <div class="relative" x-data="{ open: false }">
                        <button @click="open = !open" @click.away="open = false"
                            class="flex items-center gap-1 px-2 py-1 rounded-lg text-sm font-medium header-lang-btn transition-colors"
                            aria-label="Select language">
                            <x-icon icon="heroicons:globe-alt" class="w-4 h-4" />
                            <span class="uppercase" id="current-lang-display">{{ $currentLangKey }}</span>
                            <x-icon icon="heroicons:chevron-down" class="w-3 h-3 transition-transform"
                                x-bind:class="{ 'rotate-180': open }" />
                        </button>
                        <div x-show="open" x-transition:enter="transition ease-out duration-100"
                            x-transition:enter-start="transform opacity-0 scale-95"
                            x-transition:enter-end="transform opacity-100 scale-100"
                            x-transition:leave="transition ease-in duration-75"
                            x-transition:leave-start="transform opacity-100 scale-100"
                            x-transition:leave-end="transform opacity-0 scale-95"
                            class="absolute right-0 mt-2 w-40 rounded-lg shadow-lg bg-white dark:bg-gray-800 ring-1 ring-black ring-opacity-5 z-50 py-1"
                            style="display: none;">
                            @foreach ($languages as $lang)
                                <button type="button" wire:click="changeLanguage('{{ $lang['key'] }}')"
                                    class="w-full text-left px-4 py-2 text-sm hover:bg-gray-100 dark:hover:bg-gray-700 flex items-center gap-2 {{ $currentLangKey === $lang['key'] ? 'bg-primary-50 dark:bg-primary-900/20 text-primary-600' : 'text-gray-700 dark:text-gray-300' }}">
                                    @if (!empty($lang['flag_code']))
                                        <span class="text-lg">{{ $lang['flag_emoji'] ?? '' }}</span>
                                    @endif
                                    <span>{{ $lang['name'] }}</span>
                                    @if ($currentLangKey === $lang['key'])
                                        <x-icon icon="heroicons:check" class="w-4 h-4 ml-auto text-primary-600" />
                                    @endif
                                </button>
                            @endforeach
                        </div>
                    </div>
                @endif

                {{-- Theme Toggle Button - Hidden for now, only light mode supported
                @if ($section['show_theme_toggle'] ?? true)
                    <button id="theme-toggle" class="theme-toggle" aria-label="Toggle theme">
                        <x-icon icon="heroicons:sun" class="sun-icon w-5 h-5" />
                        <x-icon icon="heroicons:moon" class="moon-icon w-5 h-5" />
                    </button>
                @endif
                --}}
                @if (!empty($section['cta_buttons']) && is_array($section['cta_buttons']))
                    @foreach ($section['cta_buttons'] as $btnIndex => $ctaBtn)
                        @if (is_array($ctaBtn) && (!empty($ctaBtn['text']) || !empty($ctaBtn['icon'])))
                            <a href="{{ $ctaBtn['url'] ?? '#contact' }}"
                                class="header-cta-btn hidden md:flex items-center gap-2"
                                @if (str_starts_with($ctaBtn['url'] ?? '#contact', '#')) wire:navigate data-section="{{ ltrim($ctaBtn['url'] ?? '#contact', '#') }}" @endif>
                                @if (!empty($ctaBtn['icon']))<x-icon :icon="$ctaBtn['icon']" class="w-5 h-5" />@endif
                                @if (!empty($ctaBtn['text']))<span>{{ $ctaBtn['text'] }}</span>@endif
                            </a>
                        @endif
                    @endforeach
                @endif
            </div>

            {{-- Mobile Menu Button --}}
            <div class="flex items-center gap-2 md:hidden">
                {{-- Mobile Theme Toggle - Hidden for now, only light mode supported
                @if ($section['show_theme_toggle'] ?? true)
                    <button id="theme-toggle-mobile" class="theme-toggle" aria-label="Toggle theme">
                        <x-icon icon="heroicons:sun" class="sun-icon w-5 h-5" />
                        <x-icon icon="heroicons:moon" class="moon-icon w-5 h-5" />
                    </button>
                @endif
                --}}
                <button id="mobile-menu-btn" class="header-mobile-btn p-2 rounded-lg transition-colors">
                    <x-icon icon="heroicons:bars-3" class="w-6 h-6" />
                </button>
            </div>
        </div>
    </div>
</header>

{{-- Mobile Menu (Outside header for proper z-index stacking) --}}
<div id="mobile-menu" class="mobile-menu">
    <div class="flex flex-col h-full">
        <div class="flex items-center justify-between p-4 border-b">
            <div class="flex items-center gap-2">
                @if ($useFullLogo)
                    <img src="{{ $company['light_logo'] ?? asset('images/light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto dark:hidden" />
                    <img src="{{ $company['dark_logo'] ?? asset('images/dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto hidden dark:block" />
                @else
                    <img src="{{ $company['light_logo_small'] ?? asset('images/small_light.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto dark:hidden" />
                    <img src="{{ $company['dark_logo_small'] ?? asset('images/small_dark.png') }}" alt="{{ $company['name'] ?? 'Logo' }}"
                        class="h-8 w-auto hidden dark:block" />
                @endif
                @if ($showName)
                    <span class="text-xl font-bold">{{ $company['name'] ?? 'Washville' }}</span>
                @endif
            </div>
            <button id="mobile-menu-close" class="p-2 rounded-lg hover:bg-gray-100">
                <x-icon icon="heroicons:x-mark" class="w-6 h-6" />
            </button>
        </div>
        <nav class="flex-1 p-4 space-y-4 overflow-y-auto">
            @forelse ($headerPages as $page)
                <a href="{{ $page['url'] }}"
                    class="mobile-menu-link block py-3 text-lg font-medium text-gray-700 hover:text-primary-600">{{ $page['name'] }}</a>
            @empty
                {{-- No pages configured to show in header --}}
            @endforelse
        </nav>
        {{-- Mobile Language Selector --}}
        @if (count($languages ?? []) > 1)
            <div class="p-4 border-t">
                <p class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-2">Language</p>
                <div class="flex flex-wrap gap-2">
                    @foreach ($languages as $lang)
                        <button type="button" wire:click="changeLanguage('{{ $lang['key'] }}')"
                            class="px-3 py-2 text-sm rounded-lg flex items-center gap-2 {{ $currentLangKey === $lang['key'] ? 'bg-primary-600 text-white' : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600' }}">
                            @if (!empty($lang['flag_emoji']))
                                <span>{{ $lang['flag_emoji'] }}</span>
                            @endif
                            <span>{{ $lang['name'] }}</span>
                        </button>
                    @endforeach
                </div>
            </div>
        @endif

        @if (!empty($section['cta_buttons']) && is_array($section['cta_buttons']))
            <div class="p-4 border-t space-y-2">
                @foreach ($section['cta_buttons'] as $btnIndex => $ctaBtn)
                    @if (is_array($ctaBtn) && (!empty($ctaBtn['text']) || !empty($ctaBtn['icon'])))
                        <a href="{{ $ctaBtn['url'] ?? '#contact' }}"
                            @if (str_starts_with($ctaBtn['url'] ?? '#contact', '#')) data-section="{{ ltrim($ctaBtn['url'] ?? '#contact', '#') }}" @endif
                            class="header-mobile-cta mobile-menu-link flex items-center justify-center gap-2">
                            @if (!empty($ctaBtn['icon']))<x-icon :icon="$ctaBtn['icon']" class="w-5 h-5" />@endif
                            @if (!empty($ctaBtn['text']))<span>{{ $ctaBtn['text'] }}</span>@endif
                        </a>
                    @endif
                @endforeach
            </div>
        @endif
    </div>
</div>
</div>

<style>
        /* ===== Header Default Styles ===== */

        /* ===== TRANSPARENT MODE (Default - shows on hero) ===== */
        .header-transparent .nav-link {
            position: relative;
            color: rgba(255, 255, 255, 0.9);
            font-weight: 500;
            transition: color 0.3s ease;
        }

        .header-transparent .nav-link:hover {
            color: white;
        }

        .header-transparent .nav-link::after {
            content: '';
            position: absolute;
            bottom: -4px;
            left: 0;
            width: 0;
            height: 2px;
            background: white;
            transition: width 0.3s ease;
        }

        .header-transparent .nav-link:hover::after {
            width: 100%;
        }

        /* ===== TEXT COLOR MODE ===== */
        /* IMPORTANT: Text color mode settings only apply in LIGHT mode */
        /* In dark mode, colors are always controlled by dark mode variables */

        /* ===== TEXT COLOR MODE: LIGHT (white text - for dark hero backgrounds) - LIGHT MODE ONLY ===== */
        /* Only applies when NOT scrolled and NOT in dark mode */
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .nav-link,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .header-logo-text,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .header-phone-icon,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .header-lang-btn,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .header-mobile-btn,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .theme-toggle {
            color: rgba(255, 255, 255, 0.9) !important;
        }

        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .nav-link:hover,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .header-logo-text:hover {
            color: white !important;
        }

        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .nav-link::after {
            background: white !important;
        }

        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .header-lang-btn:hover,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .header-mobile-btn:hover,
        body:not(.dark) .header-transparent.header-text-light:not(.header-scrolled) .theme-toggle:hover {
            background: rgba(255, 255, 255, 0.1) !important;
            color: white !important;
        }

        /* ===== TEXT COLOR MODE: DARK (black text - for light hero backgrounds) - LIGHT MODE ONLY ===== */
        /* Only applies when NOT scrolled and NOT in dark mode */
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .nav-link,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .header-logo-text,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .header-phone-icon,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .header-lang-btn,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .header-mobile-btn,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .theme-toggle {
            color: var(--color-gray-800) !important;
        }

        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .nav-link:hover,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .header-logo-text:hover {
            color: var(--color-gray-900) !important;
        }

        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .nav-link::after {
            background: var(--color-gray-900) !important;
        }

        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .header-lang-btn:hover,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .header-mobile-btn:hover,
        body:not(.dark) .header-transparent.header-text-dark:not(.header-scrolled) .theme-toggle:hover {
            background: rgba(0, 0, 0, 0.05) !important;
            color: var(--color-gray-900) !important;
        }

        /* ===== SOLID MODE (Non-transparent header) ===== */
        .header-solid {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }

        .header-solid .nav-link {
            position: relative;
            color: var(--color-gray-700);
            font-weight: 500;
            transition: color 0.3s ease;
        }

        .header-solid .nav-link:hover {
            color: var(--color-primary-600);
        }

        .header-solid .nav-link::after {
            content: '';
            position: absolute;
            bottom: -4px;
            left: 0;
            width: 0;
            height: 2px;
            background: var(--color-primary-600);
            transition: width 0.3s ease;
        }

        .header-solid .nav-link:hover::after {
            width: 100%;
        }

        .header-solid .header-logo-text {
            color: var(--color-gray-900);
        }

        .header-solid .header-phone-icon {
            color: var(--color-gray-600);
        }

        .header-solid .header-phone-icon:hover {
            color: var(--color-primary-600);
        }

        .header-solid .header-lang-btn {
            color: var(--color-gray-600);
        }

        .header-solid .header-lang-btn:hover {
            color: var(--color-primary-600);
            background: var(--color-gray-100);
        }

        .header-solid .header-mobile-btn {
            color: var(--color-gray-900);
        }

        .header-solid .header-mobile-btn:hover {
            background: var(--color-gray-100);
        }

        .header-solid .theme-toggle {
            color: var(--color-text-secondary);
        }

        .header-solid .theme-toggle:hover {
            background: var(--color-bg-tertiary);
            color: var(--color-text-primary);
        }

        /* Dark mode - Solid header */
        body.dark .header-solid,
        body.dark #header.header-solid {
            background: #0f172a !important;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1) !important;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.3) !important;
        }

        body.dark .header-solid .nav-link {
            color: rgba(255, 255, 255, 0.8) !important;
        }

        body.dark .header-solid .nav-link:hover {
            color: var(--color-primary-400) !important;
        }

        body.dark .header-solid .nav-link::after {
            background: var(--color-primary-400) !important;
        }

        body.dark .header-solid .header-logo-text {
            color: #ffffff !important;
        }

        body.dark .header-solid .header-phone-icon {
            color: rgba(255, 255, 255, 0.7) !important;
        }

        body.dark .header-solid .header-phone-icon:hover {
            color: var(--color-primary-400) !important;
        }

        body.dark .header-solid .header-lang-btn {
            color: rgba(255, 255, 255, 0.7) !important;
        }

        body.dark .header-solid .header-lang-btn:hover {
            color: var(--color-primary-400) !important;
            background: rgba(255, 255, 255, 0.1) !important;
        }

        body.dark .header-solid .header-mobile-btn {
            color: #ffffff !important;
        }

        body.dark .header-solid .header-mobile-btn:hover {
            background: rgba(255, 255, 255, 0.1) !important;
        }

        body.dark .header-solid .theme-toggle {
            color: rgba(255, 255, 255, 0.8) !important;
        }

        body.dark .header-solid .theme-toggle:hover {
            background: rgba(255, 255, 255, 0.1) !important;
            color: #ffffff !important;
        }

        /* Dark mode for ANY header when scrolled (both transparent and solid) */
        /* This ensures dark background is maintained when scrolling */
        .dark #header.header-scrolled,
        .dark .header-solid.header-scrolled,
        .dark .header-transparent.header-scrolled {
            background: var(--color-bg-primary) !important;
            border-bottom-color: var(--color-border) !important;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.3) !important;
        }

        .dark .header-transparent.header-scrolled .nav-link {
            color: var(--color-text-secondary) !important;
        }

        .dark .header-transparent.header-scrolled .nav-link:hover {
            color: var(--color-primary-400) !important;
        }

        .dark .header-transparent.header-scrolled .nav-link::after {
            background: var(--color-primary-400) !important;
        }

        .dark .header-transparent.header-scrolled .header-logo-text {
            color: var(--color-text-primary) !important;
        }

        .dark .header-transparent.header-scrolled .header-phone-icon {
            color: var(--color-text-tertiary) !important;
        }

        .dark .header-transparent.header-scrolled .header-phone-icon:hover {
            color: var(--color-primary-400) !important;
        }

        .dark .header-transparent.header-scrolled .header-lang-btn {
            color: var(--color-text-tertiary) !important;
        }

        .dark .header-transparent.header-scrolled .header-lang-btn:hover {
            color: var(--color-primary-400) !important;
            background: var(--color-bg-tertiary) !important;
        }

        .dark .header-transparent.header-scrolled .header-mobile-btn {
            color: var(--color-text-primary) !important;
        }

        .dark .header-transparent.header-scrolled .header-mobile-btn:hover {
            background: var(--color-bg-tertiary) !important;
        }

        /* Navigation when header is scrolled (transparent mode only) */
        .header-transparent.header-scrolled .nav-link {
            color: var(--color-gray-700);
        }

        .header-transparent.header-scrolled .nav-link:hover {
            color: var(--color-primary-600);
        }

        .header-transparent.header-scrolled .nav-link::after {
            background: var(--color-primary-600);
        }

        /* More Dropdown */
        .nav-more-dropdown {
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
        }

        .nav-more-dropdown::before {
            content: '';
            position: absolute;
            top: -6px;
            left: 50%;
            transform: translateX(-50%);
            border-left: 6px solid transparent;
            border-right: 6px solid transparent;
            border-bottom: 6px solid white;
        }

        .dark .nav-more-dropdown::before {
            border-bottom-color: var(--color-gray-800);
        }

        .nav-more-item:hover span:first-child {
            opacity: 1;
        }

        /* Mobile Menu */
        .mobile-menu {
            position: fixed;
            inset: 0;
            background: white;
            z-index: 9999;
            transform: translateX(100%);
            transition: transform 0.3s ease;
        }

        .mobile-menu.active {
            transform: translateX(0);
        }

        /* Header Scroll Effect (Transparent mode only) */
        .header-transparent.header-scrolled {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        /* Header Elements - Transparent mode (on hero) */
        .header-transparent .header-logo-text {
            color: white;
            transition: color 0.3s ease;
        }

        .header-transparent .header-phone-icon {
            color: rgba(255, 255, 255, 0.9);
        }

        .header-transparent .header-phone-icon:hover {
            color: white;
        }

        .header-transparent .header-lang-btn {
            color: rgba(255, 255, 255, 0.9);
        }

        .header-transparent .header-lang-btn:hover {
            color: white;
            background: rgba(255, 255, 255, 0.1);
        }

        .header-transparent .header-mobile-btn {
            color: white;
        }

        .header-transparent .header-mobile-btn:hover {
            background: rgba(255, 255, 255, 0.1);
        }

        /* Header Elements - Transparent mode when scrolled */
        .header-transparent.header-scrolled .header-logo-text {
            color: var(--color-gray-900);
        }

        .header-transparent.header-scrolled .header-phone-icon {
            color: var(--color-gray-600);
        }

        .header-transparent.header-scrolled .header-phone-icon:hover {
            color: var(--color-primary-600);
        }

        .header-transparent.header-scrolled .header-lang-btn {
            color: var(--color-gray-600);
        }

        .header-transparent.header-scrolled .header-lang-btn:hover {
            color: var(--color-primary-600);
            background: var(--color-gray-100);
        }

        .header-transparent.header-scrolled .header-mobile-btn {
            color: var(--color-gray-900);
        }

        .header-transparent.header-scrolled .header-mobile-btn:hover {
            background: var(--color-gray-100);
        }

        /* ===== Dark Mode Header Styles ===== */

        /* Header scrolled state in dark mode */
        .dark .header-scrolled {
            background: rgba(15, 23, 42, 0.95);
        }

        .dark .header-scrolled .header-logo-text {
            color: var(--color-text-primary);
        }

        .dark .header-scrolled .nav-link {
            color: var(--color-text-secondary);
        }

        .dark .header-scrolled .nav-link:hover {
            color: var(--color-primary-400);
        }

        .dark .header-scrolled .header-phone-icon {
            color: var(--color-text-tertiary);
        }

        .dark .header-scrolled .header-phone-icon:hover {
            color: var(--color-primary-400);
        }

        .dark .header-scrolled .header-lang-btn {
            color: var(--color-text-tertiary);
        }

        .dark .header-scrolled .header-lang-btn:hover {
            color: var(--color-primary-400);
            background: var(--color-bg-tertiary);
        }

        .dark .header-scrolled .header-mobile-btn {
            color: var(--color-text-primary);
        }

        .dark .header-scrolled .header-mobile-btn:hover {
            background: var(--color-bg-tertiary);
        }

        /* Mobile menu */
        .dark .mobile-menu {
            background: var(--color-bg-primary);
        }

        .dark .mobile-menu .border-b {
            border-color: var(--color-border);
        }

        /* Theme toggle button */
        .theme-toggle {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 2.5rem;
            height: 2.5rem;
            border-radius: 0.5rem;
            cursor: pointer;
            transition: all 0.3s ease;
            background: transparent;
            border: none;
            color: rgba(255, 255, 255, 0.9);
        }

        .theme-toggle:hover {
            background: rgba(255, 255, 255, 0.1);
            color: white;
        }

        .header-scrolled .theme-toggle {
            color: var(--color-text-secondary);
        }

        .header-scrolled .theme-toggle:hover {
            background: var(--color-bg-tertiary);
            color: var(--color-text-primary);
        }

        .dark .header-scrolled .theme-toggle:hover {
            background: var(--color-bg-tertiary);
        }

        /* Sun and moon icons */
        .theme-toggle .sun-icon {
            display: none;
        }

        .theme-toggle .moon-icon {
            display: block;
        }

        .dark .theme-toggle .sun-icon {
            display: block;
        }

        .dark .theme-toggle .moon-icon {
            display: none;
        }

        /* ===== CTA Button ===== */
        .header-cta-btn {
            align-items: center;
            justify-content: center;
            padding: 0.625rem 1.25rem;
            border-radius: 0.5rem;
            font-weight: 600;
            font-size: 0.875rem;
            background: linear-gradient(135deg, var(--color-primary-600), var(--color-primary-700));
            color: white;
            box-shadow: 0 4px 14px rgba(37, 99, 235, 0.4);
            transition: all 300ms;
            white-space: nowrap;
        }

        .header-cta-btn:hover {
            background: linear-gradient(135deg, var(--color-primary-700), var(--color-primary-800));
            box-shadow: 0 6px 20px rgba(37, 99, 235, 0.5);
            transform: translateY(-2px);
        }

        /* ===== Mobile CTA Button ===== */
        .header-mobile-cta {
            width: 100%;
            padding: 0.75rem 1.5rem;
            border-radius: 0.5rem;
            font-weight: 600;
            background: linear-gradient(to right, var(--color-primary-600), var(--color-primary-700));
            color: white;
            box-shadow: 0 10px 15px -3px rgba(37, 99, 235, 0.25);
            transition: all 300ms;
        }

        .header-mobile-cta:hover {
            background: linear-gradient(to right, var(--color-primary-700), var(--color-primary-800));
        }

        /* ===== RTL Support for Header ===== */

        /* Navigation link underline animation for RTL */
        [dir="rtl"] .nav-link::after {
            left: auto;
            right: 0;
        }

        /* Mobile menu slide direction for RTL */
        [dir="rtl"] .mobile-menu {
            transform: translateX(-100%);
        }

        [dir="rtl"] .mobile-menu.active {
            transform: translateX(0);
        }

        /* Header phone icon spacing for RTL */
        [dir="rtl"] .header-phone-icon {
            margin-right: 0;
        }
</style>

{{-- Script to handle dark mode for header (solid and transparent when scrolled) --}}
<script>
    (function() {
        function updateHeaderDarkMode() {
            const header = document.getElementById('header');
            if (!header) return;

            const isSolid = header.dataset.solid === 'true';
            const isTransparent = header.dataset.transparent === 'true';
            const isScrolled = header.classList.contains('header-scrolled');
            const isDark = document.body.classList.contains('dark');

            // Solid header (non-transparent)
            if (isSolid) {
                if (isDark) {
                    header.style.setProperty('background', '#0f172a', 'important');
                    header.style.setProperty('border-bottom', '1px solid rgba(255, 255, 255, 0.1)', 'important');
                } else {
                    header.style.setProperty('background', 'rgba(255, 255, 255, 0.95)', 'important');
                    header.style.setProperty('border-bottom', 'none', 'important');
                }
            }
            // Transparent header when scrolled
            else if (isTransparent && isScrolled) {
                if (isDark) {
                    header.style.setProperty('background', '#0f172a', 'important');
                    header.style.setProperty('border-bottom', '1px solid rgba(255, 255, 255, 0.1)', 'important');
                } else {
                    header.style.setProperty('background', 'rgba(255, 255, 255, 0.95)', 'important');
                    header.style.setProperty('border-bottom', 'none', 'important');
                }
            }
            // Transparent header not scrolled - keep transparent
            else if (isTransparent && !isScrolled) {
                header.style.removeProperty('background');
                header.style.removeProperty('border-bottom');
            }
        }

        // Run IMMEDIATELY since header element exists at this point (script is after header)
        updateHeaderDarkMode();

        // Also run on DOMContentLoaded as a fallback
        document.addEventListener('DOMContentLoaded', updateHeaderDarkMode);

        // Watch for dark mode changes on body
        const bodyObserver = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.attributeName === 'class') {
                    updateHeaderDarkMode();
                }
            });
        });

        // Watch for scroll state changes on header (header-scrolled class)
        function observeHeader() {
            const header = document.getElementById('header');
            if (header) {
                const headerObserver = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.attributeName === 'class') {
                            updateHeaderDarkMode();
                        }
                    });
                });
                headerObserver.observe(header, { attributes: true });
            }
        }

        // Start observing body for class changes
        if (document.body) {
            bodyObserver.observe(document.body, { attributes: true });
            observeHeader();
        } else {
            document.addEventListener('DOMContentLoaded', function() {
                bodyObserver.observe(document.body, { attributes: true });
                observeHeader();
            });
        }

        // Also run on Livewire navigation
        document.addEventListener('livewire:navigated', function() {
            updateHeaderDarkMode();
            observeHeader();
        });

        // Run on scroll to catch scroll state changes
        window.addEventListener('scroll', updateHeaderDarkMode);
    })();
</script>

{{-- Language Change Script --}}
@script
    <script>
        // Listen for Livewire language-changed event and reload the page
        // Remove 'lang' query parameter if present before reload
        $wire.on('language-changed', () => {
            const url = new URL(window.location.href);
            if (url.searchParams.has('lang')) {
                url.searchParams.delete('lang');
                window.location.href = url.toString();
            } else {
                window.location.reload();
            }
        });
    </script>
@endscript

</div>
