<?php

use Livewire\Component;

/**
 * Hero Section 1 Component
 *
 * Receives standardized props from homepage:
 * - section: Contains title, subtitle, badge, and all content fields from database
 * - items: Array of items (stats in this case)
 * - company, social, services: Shared data
 */
new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Computed hero data for template
    public array $hero = [];

    public function mount(array $section = [], array $items = [], array $company = [], array $social = [], array $services = [], ?int $paddingTop = null, ?int $paddingBottom = null)
    {
        $this->section = $section;
        $this->items = $items;
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Build hero data from section (for backward compatibility with template)
        $this->hero = array_merge(
            [
                'badge' => '#1 Rated Laundry Service in Town',
                'title' => 'Fresh & Clean Laundry',
                'subtitle' => 'Every Time, On Time',
                'description' => 'Experience premium laundry services with free pickup & delivery.',
                'hero_image' => 'https://images.unsplash.com/photo-1545173168-9f1947eebb7f?w=600&h=500&fit=crop',
                'buttons' => [['text' => 'Schedule Pickup', 'url' => '#contact', 'icon' => 'heroicons:arrow-right', 'is_transparent' => false], ['text' => 'View Pricing', 'url' => '#pricing', 'icon' => 'heroicons:play-circle', 'is_transparent' => true]],
                'floating_cards' => [['icon' => 'heroicons:check', 'title' => 'Free Pickup', 'subtitle' => '& Delivery'], ['icon' => 'heroicons:clock', 'title' => 'Express', 'subtitle' => '24h Service']],
                'stats' => [['value' => '15K+', 'label' => 'Happy Customers'], ['value' => '50K+', 'label' => 'Orders Completed'], ['value' => '99%', 'label' => 'Satisfaction Rate'], ['value' => '24h', 'label' => 'Express Delivery']],
            ],
            $this->section,
        );

        // Use items as stats if provided
        if (!empty($this->items)) {
            $this->hero['stats'] = $this->items;
        }
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
    <section class="hero1-section" @if ($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
        {{-- Background Pattern --}}
        <div class="hero1-bg-pattern"></div>

        {{-- Gradient Overlay --}}
        <div class="hero1-gradient-overlay"></div>

        <div class="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="hero1-content-wrapper">
                {{-- Main Content - Centered --}}
                <div class="hero1-center-content">
                    {{-- Badge --}}
                    <div class="hero1-badge animate-fade-in-down">
                        <span class="hero1-badge-dot"></span>
                        <span>{{ $hero['badge'] }}</span>
                    </div>

                    {{-- Title --}}
                    <h1 class="hero1-title animate-fade-in-up">
                        {{ $hero['title'] }}
                        <span class="hero1-title-highlight">{{ $hero['subtitle'] }}</span>
                    </h1>

                    {{-- Description --}}
                    <p class="hero1-description animate-fade-in-up delay-200">
                        {{ $hero['description'] }}
                    </p>

                    {{-- Stats - Inline List --}}
                    @if (!empty($hero['stats']))
                        <div class="hero1-inline-stats animate-fade-in-up delay-250">
                            @foreach ($hero['stats'] as $stat)
                                <div class="hero1-inline-stat-item">
                                    @if (!empty($stat['icon']))
                                        <div class="hero1-inline-stat-icon">
                                            <x-icon icon="{{ $stat['icon'] }}" class="w-5 h-5" />
                                        </div>
                                    @endif
                                    <div class="hero1-inline-stat-text">
                                        @if (!empty($stat['value']))
                                            <span class="hero1-inline-stat-value">{{ $stat['value'] }}</span>
                                        @endif
                                        @if (!empty($stat['label']))
                                            <span class="hero1-inline-stat-label">{{ $stat['label'] }}</span>
                                        @endif
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @endif

                    {{-- CTA Buttons --}}
                    <div class="hero1-cta-group animate-fade-in-up delay-300">
                        @foreach ($hero['buttons'] ?? [] as $index => $button)
                            @if ($button['is_transparent'] ?? false)
                                <a href="{{ $button['url'] ?? '#' }}" class="hero1-btn-secondary">
                                    @if (!empty($button['icon']))
                                        <x-icon icon="{{ $button['icon'] }}" class="w-5 h-5" />
                                    @endif
                                    @if (!empty($button['text']))
                                        <span>{{ $button['text'] }}</span>
                                    @endif
                                </a>
                            @else
                                <a href="{{ $button['url'] ?? '#' }}" class="hero1-btn-primary">
                                    @if (!empty($button['icon']))
                                        <x-icon icon="{{ $button['icon'] }}" class="w-5 h-5" />
                                    @endif
                                    @if (!empty($button['text']))
                                        <span>{{ $button['text'] }}</span>
                                    @endif
                                </a>
                            @endif
                        @endforeach
                    </div>
                </div>

                {{-- Image Section --}}
                <div class="hero1-image-section animate-fade-in-up delay-300">
                    <div class="hero1-image-container">
                        {{-- Main Image --}}
                        <div class="hero1-main-image">
                            <img src="{{ $hero['hero_image_url'] ?? ($hero['hero_image'] ?? 'https://images.unsplash.com/photo-1545173168-9f1947eebb7f?w=600&h=500&fit=crop') }}"
                                alt="Professional Laundry Service" loading="lazy">
                        </div>

                        {{-- Floating Cards --}}
                        @foreach ($hero['floating_cards'] ?? [] as $index => $card)
                            <div
                                class="hero1-float-card hero1-float-card-{{ $index + 1 }} animate-bounce-soft {{ $index > 0 ? 'delay-300' : '' }}">
                                <div
                                    class="hero1-float-icon {{ $index === 0 ? 'hero1-float-icon-success' : 'hero1-float-icon-primary' }}">
                                    <x-icon icon="{{ $card['icon'] ?? 'heroicons:check' }}"
                                        class="w-6 h-6 text-white" />
                                </div>
                                <div>
                                    <p class="hero1-float-title">{{ $card['title'] ?? '' }}</p>
                                    <p class="hero1-float-subtitle">{{ $card['subtitle'] ?? '' }}</p>
                                </div>
                            </div>
                        @endforeach

                        {{-- Decorative Circles --}}
                        <div class="hero1-deco-circle hero1-deco-circle-1"></div>
                        <div class="hero1-deco-circle hero1-deco-circle-2"></div>
                    </div>
                </div>
            </div>
        </div>

        {{-- Bottom Curve --}}
        <div class="hero1-bottom-curve">
            <svg viewBox="0 0 1440 100" fill="none" xmlns="http://www.w3.org/2000/svg" preserveAspectRatio="none">
                <path d="M0 100V0C240 60 480 90 720 90C960 90 1200 60 1440 0V100H0Z" class="hero1-curve-fill" />
            </svg>
        </div>
    </section>

    <style>
        /* ===== Hero Section Style 1 ===== */

        .hero1-section {
            position: relative;
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 8rem 0 6rem;
            background: linear-gradient(135deg, var(--color-bg-primary) 0%, var(--color-bg-secondary) 100%);
            overflow: hidden;
        }

        .dark .hero1-section {
            background: linear-gradient(135deg, var(--color-bg-primary) 0%, var(--color-bg-secondary) 100%);
        }

        /* Background Pattern */
        .hero1-bg-pattern {
            position: absolute;
            inset: 0;
            background-image: radial-gradient(var(--color-primary-200) 1px, transparent 1px);
            background-size: 40px 40px;
            opacity: 0.3;
        }

        .dark .hero1-bg-pattern {
            background-image: radial-gradient(var(--color-primary-800) 1px, transparent 1px);
            opacity: 0.15;
        }

        /* Gradient Overlay */
        .hero1-gradient-overlay {
            position: absolute;
            inset: 0;
            background: radial-gradient(ellipse at top right, var(--color-primary-100) 0%, transparent 50%),
                radial-gradient(ellipse at bottom left, var(--color-secondary-100) 0%, transparent 50%);
            opacity: 0.6;
        }

        .dark .hero1-gradient-overlay {
            background: radial-gradient(ellipse at top right, var(--color-primary-900) 0%, transparent 50%),
                radial-gradient(ellipse at bottom left, var(--color-secondary-900) 0%, transparent 50%);
            opacity: 0.3;
        }

        /* Content Wrapper */
        .hero1-content-wrapper {
            display: grid;
            grid-template-columns: 1fr;
            gap: 3rem;
            align-items: center;
        }

        @media (min-width: 1024px) {
            .hero1-content-wrapper {
                grid-template-columns: 1fr 1fr;
                gap: 4rem;
            }
        }

        /* Center Content */
        .hero1-center-content {
            text-align: center;
        }

        @media (min-width: 1024px) {
            .hero1-center-content {
                text-align: left;
            }
        }

        /* Badge */
        .hero1-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: var(--color-primary-50);
            border: 1px solid var(--color-primary-200);
            border-radius: 9999px;
            font-size: 0.875rem;
            font-weight: 500;
            color: var(--color-primary-700);
            margin-bottom: 1.5rem;
        }

        .dark .hero1-badge {
            background: var(--color-primary-900);
            border-color: var(--color-primary-700);
            color: var(--color-primary-300);
        }

        .hero1-badge-dot {
            width: 0.5rem;
            height: 0.5rem;
            background: var(--color-primary-500);
            border-radius: 50%;
            animation: pulse-soft 2s ease-in-out infinite;
        }

        /* Title */
        .hero1-title {
            font-family: var(--font-heading);
            font-size: 2.5rem;
            font-weight: 800;
            line-height: 1.1;
            color: var(--color-gray-900);
            margin-bottom: 1.5rem;
        }

        @media (min-width: 640px) {
            .hero1-title {
                font-size: 3rem;
            }
        }

        @media (min-width: 1024px) {
            .hero1-title {
                font-size: 3.5rem;
            }
        }

        @media (min-width: 1280px) {
            .hero1-title {
                font-size: 4rem;
            }
        }

        .dark .hero1-title {
            color: var(--color-text-primary);
        }

        .hero1-title-highlight {
            display: block;
            background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-500));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        /* Description */
        .hero1-description {
            font-size: 1.125rem;
            line-height: 1.75;
            color: var(--color-gray-600);
            max-width: 540px;
            margin-bottom: 2rem;
        }

        @media (min-width: 1024px) {
            .hero1-description {
                margin-left: 0;
                margin-right: auto;
            }
        }

        .dark .hero1-description {
            color: var(--color-text-tertiary);
        }

        /* CTA Group */
        .hero1-cta-group {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            justify-content: center;
            margin-bottom: 2rem;
        }

        @media (min-width: 1024px) {
            .hero1-cta-group {
                justify-content: flex-start;
            }
        }

        .hero1-btn-primary {
            display: inline-flex;
            align-items: center;
            gap: 0.75rem;
            padding: 1rem 2rem;
            background: linear-gradient(135deg, var(--color-primary-600), var(--color-primary-700));
            color: white;
            font-weight: 600;
            font-size: 1rem;
            border-radius: 0.75rem;
            box-shadow: 0 4px 14px rgba(37, 99, 235, 0.4);
            transition: all 0.3s ease;
        }

        .hero1-btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.5);
        }

        .hero1-btn-primary svg,
        .hero1-btn-secondary svg {
            color: inherit;
            flex-shrink: 0;
        }

        .hero1-btn-secondary {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 1rem 1.5rem;
            background: white;
            color: var(--color-gray-700);
            font-weight: 600;
            font-size: 1rem;
            border-radius: 0.75rem;
            border: 2px solid var(--color-gray-200);
            transition: all 0.3s ease;
        }

        .hero1-btn-secondary:hover {
            border-color: var(--color-primary-400);
            color: var(--color-primary-600);
        }

        .dark .hero1-btn-secondary {
            background: var(--color-bg-tertiary);
            color: var(--color-text-primary);
            border-color: var(--color-border);
        }

        .dark .hero1-btn-secondary:hover {
            border-color: var(--color-primary-500);
            color: var(--color-primary-400);
        }

        /* Inline Stats */
        .hero1-inline-stats {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            margin-bottom: 2rem;
        }

        .hero1-inline-stat-item {
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }

        .hero1-inline-stat-icon {
            width: 1.5rem;
            flex-shrink: 0;
            color: var(--color-primary-500);
        }

        .hero1-inline-stat-text {
            display: flex;
            flex-wrap: wrap;
            align-items: center;
            gap: 0.5rem;
        }

        .hero1-inline-stat-value {
            font-family: var(--font-heading);
            font-weight: 800;
            font-size: 1.125rem;
            color: var(--color-primary-600);
        }

        .hero1-inline-stat-label {
            font-size: 1.125rem;
            color: var(--color-gray-600);
        }

        .dark .hero1-inline-stat-icon {
            color: var(--color-primary-400);
        }

        .dark .hero1-inline-stat-value {
            color: var(--color-primary-400);
        }

        .dark .hero1-inline-stat-label {
            color: var(--color-text-tertiary);
        }

        /* Image Section */
        .hero1-image-section {
            display: none;
        }

        @media (min-width: 1024px) {
            .hero1-image-section {
                display: block;
            }
        }

        .hero1-image-container {
            position: relative;
            padding: 2rem;
        }

        .hero1-main-image {
            position: relative;
            border-radius: 1.5rem;
            overflow: hidden;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }

        .hero1-main-image img {
            width: 100%;
            height: auto;
            object-fit: cover;
            aspect-ratio: 4/3;
        }

        /* Floating Cards */
        .hero1-float-card {
            position: absolute;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 1rem;
            background: white;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.12);
        }

        .dark .hero1-float-card {
            background: var(--color-card-bg);
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
        }

        .hero1-float-card-1 {
            bottom: 2rem;
            left: -1rem;
        }

        .hero1-float-card-2 {
            top: 2rem;
            right: -1rem;
        }

        .hero1-float-icon {
            width: 3rem;
            height: 3rem;
            border-radius: 0.75rem;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }

        .hero1-float-icon svg {
            color: white;
        }

        .hero1-float-icon-success {
            background: linear-gradient(135deg, var(--color-success-500, #10b981), var(--color-success-600, #059669));
        }

        .hero1-float-icon-primary {
            background: linear-gradient(135deg, var(--color-primary-500), var(--color-primary-600));
        }

        .dark .hero1-float-icon-success {
            background: linear-gradient(135deg, var(--color-success-400, #34d399), var(--color-success-500, #10b981));
        }

        .dark .hero1-float-icon-primary {
            background: linear-gradient(135deg, var(--color-primary-400), var(--color-primary-500));
        }

        .hero1-float-title {
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--color-gray-900);
        }

        .dark .hero1-float-title {
            color: var(--color-text-primary);
        }

        .hero1-float-subtitle {
            font-size: 0.75rem;
            color: var(--color-gray-500);
        }

        .dark .hero1-float-subtitle {
            color: var(--color-text-tertiary);
        }

        /* Decorative Circles */
        .hero1-deco-circle {
            position: absolute;
            border-radius: 50%;
            border: 2px dashed var(--color-primary-300);
            opacity: 0.5;
        }

        .dark .hero1-deco-circle {
            border-color: var(--color-primary-700);
            opacity: 0.3;
        }

        .hero1-deco-circle-1 {
            width: 300px;
            height: 300px;
            top: -50px;
            right: -100px;
            animation: spin-slow 30s linear infinite;
        }

        .hero1-deco-circle-2 {
            width: 200px;
            height: 200px;
            bottom: -50px;
            left: -50px;
            animation: spin-slow 25s linear infinite reverse;
        }

        /* Bottom Curve */
        .hero1-bottom-curve {
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            line-height: 0;
        }

        .hero1-bottom-curve svg {
            width: 100%;
            height: 60px;
        }

        @media (min-width: 768px) {
            .hero1-bottom-curve svg {
                height: 80px;
            }
        }

        .hero1-curve-fill {
            fill: white;
        }

        .dark .hero1-curve-fill {
            fill: var(--color-bg-primary);
        }

        /* RTL Support */
        [dir="rtl"] .hero1-float-card-1 {
            left: auto;
            right: -1rem;
        }

        [dir="rtl"] .hero1-float-card-2 {
            right: auto;
            left: -1rem;
        }

        [dir="rtl"] .hero1-deco-circle-1 {
            right: auto;
            left: -100px;
        }

        [dir="rtl"] .hero1-deco-circle-2 {
            left: auto;
            right: -50px;
        }
    </style>
</div>
