<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public array $hero = [];
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(array $section = [], array $items = [], array $company = [], array $social = [], array $services = [], ?int $paddingTop = null, ?int $paddingBottom = null)
    {
        $this->section = $section;
        $this->items = $items;
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->hero = array_merge(
            [
                'badge' => '#1 Rated Laundry Service in Town',
                'title' => 'Fresh & Clean Laundry',
                'subtitle' => 'Every Time, On Time',
                'description' => 'Experience premium laundry services with free pickup & delivery.',
                'hero_image' => 'https://images.unsplash.com/photo-1545173168-9f1947eebb7f?w=600&h=500&fit=crop',
                'buttons' => [['text' => 'Schedule Pickup', 'url' => '#contact', 'icon' => 'heroicons:calendar', 'is_transparent' => false], ['text' => 'View Pricing', 'url' => '#pricing', 'icon' => '', 'is_transparent' => true]],
                'floating_cards' => [['icon' => 'heroicons:check', 'title' => 'Free Pickup', 'subtitle' => '& Delivery'], ['icon' => 'heroicons:clock', 'title' => 'Express', 'subtitle' => '24h Service']],
                'stats' => [],
            ],
            $this->section,
        );

        if (!empty($this->items)) {
            $this->hero['stats'] = $this->items;
        }
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
    <section class="relative min-h-screen flex items-center gradient-hero overflow-hidden"
        @if ($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
        {{-- Floating Elements --}}
        <div class="absolute inset-0 overflow-hidden">
            <div class="floating-element w-64 h-64 bg-white top-20 -left-32 animate-float"></div>
            <div class="floating-element w-96 h-96 bg-white bottom-20 -right-48 animate-float delay-300"></div>
            <div class="floating-element w-32 h-32 bg-white top-1/2 left-1/4 animate-float delay-500"></div>
        </div>

        <div class="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-32">
            <div class="grid lg:grid-cols-2 gap-12 items-center">
                {{-- Hero Content --}}
                <div class="text-white">
                    <span
                        class="inline-block px-4 py-2 rounded-full bg-white/20 text-sm font-medium mb-6 animate-fade-in-down">
                        {{ $hero['badge'] }}
                    </span>
                    <h1 class="text-4xl sm:text-5xl lg:text-6xl font-bold leading-tight mb-6 animate-fade-in-up">
                        {{ $hero['title'] }}
                        <span class="block text-secondary-300">{{ $hero['subtitle'] }}</span>
                    </h1>
                    <p class="text-lg text-white/90 mb-8 max-w-lg animate-fade-in-up delay-200">
                        {{ $hero['description'] }}
                    </p>
                    <div class="flex flex-wrap gap-4 animate-fade-in-up delay-300">
                        @foreach ($hero['buttons'] ?? [] as $index => $button)
                            @if ($button['is_transparent'] ?? false)
                                <a href="{{ $button['url'] ?? '#' }}"
                                    class="btn btn-outline-white flex items-center gap-2">
                                    @if (!empty($button['icon']))
                                        <x-icon icon="{{ $button['icon'] }}" class="w-5 h-5" />
                                    @endif
                                    @if (!empty($button['text']))
                                        <span>{{ $button['text'] }}</span>
                                    @endif
                                </a>
                            @else
                                <a href="{{ $button['url'] ?? '#' }}" class="btn btn-white flex items-center gap-2">
                                    @if (!empty($button['icon']))
                                        <x-icon icon="{{ $button['icon'] }}" class="w-5 h-5" />
                                    @endif
                                    @if (!empty($button['text']))
                                        <span>{{ $button['text'] }}</span>
                                    @endif
                                </a>
                            @endif
                        @endforeach
                    </div>
                </div>

                {{-- Hero Image --}}
                <div class="hidden lg:block animate-fade-in-right delay-400">
                    <div class="relative">
                        <div class="absolute inset-0 bg-white/10 rounded-3xl transform rotate-6"></div>
                        <img src="{{ $hero['hero_image_url'] ?? ($hero['hero_image'] ?? 'https://images.unsplash.com/photo-1545173168-9f1947eebb7f?w=600&h=500&fit=crop') }}"
                            alt="Professional Laundry Service"
                            class="relative rounded-3xl shadow-2xl w-full object-cover" loading="lazy">

                        {{-- Floating Cards --}}
                        @foreach ($hero['floating_cards'] ?? [] as $index => $card)
                            <div
                                class="hero-float-card hero-float-card-{{ $index + 1 }} animate-bounce-soft {{ $index > 0 ? 'delay-300' : '' }}">
                                <div
                                    class="hero-float-icon {{ $index === 0 ? 'hero-float-icon-success' : 'hero-float-icon-primary' }}">
                                    <x-icon icon="{{ $card['icon'] ?? 'heroicons:check' }}" class="w-6 h-6" />
                                </div>
                                <div>
                                    <p class="hero-float-title">{{ $card['title'] ?? '' }}</p>
                                    <p class="hero-float-subtitle">{{ $card['subtitle'] ?? '' }}</p>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            </div>

            {{-- Stats --}}
            @if (!empty($hero['stats']))
                <div class="hero-stats-grid mt-16 grid grid-cols-2 md:grid-cols-4 gap-4 animate-fade-in-up delay-500">
                    @foreach ($hero['stats'] as $stat)
                        <div class="hero-stat-item glass rounded-xl">
                            @if (!empty($stat['icon']))
                                <div class="hero-stat-icon">
                                    <x-icon icon="{{ $stat['icon'] }}" class="w-8 h-8" />
                                </div>
                            @endif
                            <div class="hero-stat-content">
                                <div class="hero-stat-value" data-counter="{{ $stat['value'] }}">{{ $stat['value'] }}
                                </div>
                                <div class="hero-stat-label">{{ $stat['label'] }}</div>
                            </div>
                        </div>
                    @endforeach
                </div>
            @endif
        </div>

        {{-- Wave Decoration --}}
        <div class="absolute bottom-0 left-0 right-0">
            <svg viewBox="0 0 1440 120" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path
                    d="M0 120L60 110C120 100 240 80 360 70C480 60 600 60 720 65C840 70 960 80 1080 85C1200 90 1320 90 1380 90L1440 90V120H1380C1320 120 1200 120 1080 120C960 120 840 120 720 120C600 120 480 120 360 120C240 120 120 120 60 120H0Z"
                    fill="white" class="dark:fill-gray-900" />
            </svg>
        </div>
    </section>

    <style>
        /* ===== Hero Stats - Horizontal Layout ===== */
        .hero-stat-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem 1.25rem;
        }

        .hero-stat-icon {
            width: 3rem;
            height: 3rem;
            border-radius: 0.75rem;
            background: rgba(255, 255, 255, 0.2);
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            color: var(--color-primary-200);
        }

        .hero-stat-content {
            flex: 1;
            min-width: 0;
        }

        .hero-stat-value {
            font-family: var(--font-heading);
            font-size: 1.5rem;
            font-weight: 800;
            line-height: 1.2;
            background: linear-gradient(135deg, white, rgba(255, 255, 255, 0.8));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .hero-stat-label {
            font-size: 0.75rem;
            color: rgba(255, 255, 255, 0.8);
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        /* Dark mode support */
        .dark .hero-stat-item.glass {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .dark .hero-stat-icon {
            background: rgba(255, 255, 255, 0.1);
            color: var(--color-primary-400);
        }

        .dark .hero-stat-value {
            background: linear-gradient(135deg, var(--color-primary-300), var(--color-primary-400));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .dark .hero-stat-label {
            color: rgba(255, 255, 255, 0.6);
        }

        /* Responsive adjustments */
        @media (max-width: 640px) {
            .hero-stat-item {
                padding: 0.875rem 1rem;
                gap: 0.75rem;
            }

            .hero-stat-icon {
                width: 2.5rem;
                height: 2.5rem;
            }

            .hero-stat-icon svg {
                width: 1.25rem;
                height: 1.25rem;
            }

            .hero-stat-value {
                font-size: 1.25rem;
            }

            .hero-stat-label {
                font-size: 0.625rem;
            }
        }

        /* ===== Floating Cards ===== */
        .hero-float-card {
            position: absolute;
            display: flex;
            align-items: center;
            gap: 0.875rem;
            padding: 1rem 1.25rem;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 1rem;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
            z-index: 10;
        }

        .hero-float-card-1 {
            bottom: -1.5rem;
            left: -1.5rem;
        }

        .hero-float-card-2 {
            top: -1rem;
            right: -1rem;
        }

        .hero-float-icon {
            width: 3rem;
            height: 3rem;
            border-radius: 0.75rem;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }

        .hero-float-icon svg {
            color: white;
        }

        .hero-float-icon-success {
            background: linear-gradient(135deg, var(--color-success-500, #10b981), var(--color-success-600, #059669));
        }

        .hero-float-icon-primary {
            background: linear-gradient(135deg, var(--color-primary-500), var(--color-primary-600));
        }

        .hero-float-title {
            font-weight: 700;
            font-size: 0.9375rem;
            color: var(--color-gray-900);
            line-height: 1.3;
        }

        .hero-float-subtitle {
            font-size: 0.8125rem;
            color: var(--color-gray-500);
        }

        /* Dark mode for floating cards */
        .dark .hero-float-card {
            background: rgba(30, 41, 59, 0.95);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .dark .hero-float-title {
            color: var(--color-text-primary);
        }

        .dark .hero-float-subtitle {
            color: var(--color-text-tertiary);
        }

        .dark .hero-float-icon-success {
            background: linear-gradient(135deg, var(--color-success-400, #34d399), var(--color-success-500, #10b981));
        }

        .dark .hero-float-icon-primary {
            background: linear-gradient(135deg, var(--color-primary-400), var(--color-primary-500));
        }

        /* RTL Support */
        [dir="rtl"] .hero-float-card-1 {
            left: auto;
            right: -1.5rem;
        }

        [dir="rtl"] .hero-float-card-2 {
            right: auto;
            left: -1rem;
        }
    </style>
</div>
