<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(array $section = [], array $items = [], array $company = [], array $social = [], array $services = [], bool $isWhiteBackground = false, ?int $paddingTop = null, ?int $paddingBottom = null)
    {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Use items from standardized props
        $this->items = !empty($items) ? $items : [['step' => '01', 'title' => 'Schedule Pickup', 'description' => 'Book online or call us.', 'icon' => 'mdi:calendar-clock'], ['step' => '02', 'title' => 'We Collect', 'description' => 'Our driver arrives at your doorstep.', 'icon' => 'heroicons:truck'], ['step' => '03', 'title' => 'Expert Cleaning', 'description' => 'Your clothes are cleaned with care.', 'icon' => 'heroicons:sparkles'], ['step' => '04', 'title' => 'Delivery', 'description' => 'Fresh, clean laundry delivered back to you.', 'icon' => 'heroicons:home']];

        $this->section = array_merge(
            [
                'badge' => 'How It Works',
                'title' => 'Simple 4-Step Process',
                'subtitle' => 'Getting your laundry done has never been easier.',
            ],
            $section,
        );
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="how-it-works"
    class="section {{ $isWhiteBackground ? 'bg-white dark:bg-gray-800' : 'bg-gray-50 dark:bg-gray-900' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {{-- Section Header --}}
        <div class="text-center mb-16">
            <span class="badge badge-primary mb-4">{{ $section['badge'] }}</span>
            <h2 class="section-title">{{ $section['title'] }}</h2>
            <p class="section-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Cards Grid --}}
        <div class="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            @foreach ($items as $index => $step)
                <div class="scroll-animate hiw-card group">
                    {{-- Card Top Border Gradient --}}
                    <div class="hiw-card-border"></div>

                    {{-- Step Number --}}
                    <div class="hiw-card-step">
                        <span>{{ $step['step'] }}</span>
                    </div>

                    {{-- Icon Container --}}
                    <div class="hiw-card-icon-wrap">
                        <div class="hiw-card-icon">
                            <x-icon icon="{{ $step['icon'] ?? 'heroicons:home' }}" class="hiw-card-icon-svg w-8 h-8" />
                        </div>
                        {{-- Floating Ring --}}
                        <div class="hiw-card-ring"></div>
                    </div>

                    {{-- Content --}}
                    <h3
                        class="text-xl font-bold text-gray-900 dark:text-white mb-3 group-hover:text-primary-600 dark:group-hover:text-primary-400 transition-colors duration-300">
                        {{ $step['title'] }}
                    </h3>
                    <p class="text-gray-600 dark:text-gray-400 text-sm leading-relaxed">
                        {{ $step['description'] }}
                    </p>

                    {{-- Arrow Connector (not on last item) --}}
                    @if (!$loop->last)
                        <div class="hiw-card-arrow">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2"
                                stroke="currentColor" class="w-6 h-6">
                                <path stroke-linecap="round" stroke-linejoin="round"
                                    d="M13.5 4.5 21 12m0 0-7.5 7.5M21 12H3" />
                            </svg>
                        </div>
                    @endif
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* Card Base */
    .hiw-card {
        position: relative;
        background: var(--color-card-bg, #fff);
        border-radius: 1.5rem;
        padding: 2rem 1.5rem;
        text-align: center;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        overflow: hidden;
    }

    .dark .hiw-card {
        background: var(--color-bg-secondary);
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.2);
    }

    .hiw-card:hover {
        transform: translateY(-10px);
        box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    }

    .dark .hiw-card:hover {
        box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.3);
    }

    /* Top Border Gradient */
    .hiw-card-border {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, var(--color-primary-500), var(--color-secondary-500));
        transform: scaleX(0);
        transform-origin: left;
        transition: transform 0.4s ease;
    }

    [dir="rtl"] .hiw-card-border {
        transform-origin: right;
    }

    .hiw-card:hover .hiw-card-border {
        transform: scaleX(1);
    }

    /* Step Number */
    .hiw-card-step {
        position: absolute;
        top: 1rem;
        right: 1rem;
        font-size: 3rem;
        font-weight: 800;
        color: var(--color-primary-100);
        line-height: 1;
        transition: all 0.3s ease;
    }

    [dir="rtl"] .hiw-card-step {
        right: auto;
        left: 1rem;
    }

    .dark .hiw-card-step {
        color: var(--color-primary-900);
        opacity: 0.5;
    }

    .hiw-card:hover .hiw-card-step {
        color: var(--color-primary-200);
        transform: scale(1.1);
    }

    .dark .hiw-card:hover .hiw-card-step {
        color: var(--color-primary-800);
    }

    /* Icon Wrapper */
    .hiw-card-icon-wrap {
        position: relative;
        display: inline-flex;
        margin-bottom: 1.5rem;
    }

    /* Icon Container */
    .hiw-card-icon {
        width: 5rem;
        height: 5rem;
        background: linear-gradient(135deg, var(--color-primary-50), var(--color-secondary-50));
        border-radius: 1.25rem;
        display: flex;
        align-items: center;
        justify-content: center;
        position: relative;
        z-index: 1;
        transition: all 0.4s ease;
    }

    .dark .hiw-card-icon {
        background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
    }

    .hiw-card:hover .hiw-card-icon {
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-500));
        transform: rotate(6deg) scale(1.05);
    }

    /* Icon SVG colors */
    .hiw-card-icon-svg,
    .hiw-card-icon-svg svg {
        color: var(--color-primary-600);
        fill: var(--color-primary-600);
        transition: color 0.3s ease, fill 0.3s ease;
    }

    .dark .hiw-card-icon-svg,
    .dark .hiw-card-icon-svg svg {
        color: var(--color-primary-400);
        fill: var(--color-primary-400);
    }

    .hiw-card:hover .hiw-card-icon-svg,
    .hiw-card:hover .hiw-card-icon-svg svg,
    .hiw-card:hover .hiw-card-icon svg {
        color: #fff !important;
        fill: #fff !important;
    }

    /* Floating Ring */
    .hiw-card-ring {
        position: absolute;
        inset: -0.5rem;
        border: 2px dashed var(--color-primary-200);
        border-radius: 1.5rem;
        opacity: 0;
        transition: all 0.4s ease;
    }

    .dark .hiw-card-ring {
        border-color: var(--color-primary-700);
    }

    .hiw-card:hover .hiw-card-ring {
        opacity: 1;
        animation: ring-spin 8s linear infinite;
    }

    @keyframes ring-spin {
        from {
            transform: rotate(0deg);
        }

        to {
            transform: rotate(360deg);
        }
    }

    /* Arrow Connector */
    .hiw-card-arrow {
        position: absolute;
        top: 50%;
        right: -1.5rem;
        transform: translateY(-50%);
        color: var(--color-primary-300);
        opacity: 0;
        transition: all 0.3s ease;
        z-index: 10;
    }

    [dir="rtl"] .hiw-card-arrow {
        right: auto;
        left: -1.5rem;
        transform: translateY(-50%) scaleX(-1);
    }

    .dark .hiw-card-arrow {
        color: var(--color-primary-600);
    }

    .hiw-card:hover .hiw-card-arrow {
        opacity: 1;
        right: -2rem;
    }

    [dir="rtl"] .hiw-card:hover .hiw-card-arrow {
        right: auto;
        left: -2rem;
    }

    /* Mobile Responsive */
    @media (max-width: 1024px) {
        .hiw-card-arrow {
            display: none;
        }
    }

    @media (max-width: 768px) {
        .hiw-card {
            padding: 1.5rem 1rem;
        }

        .hiw-card-step {
            font-size: 2.5rem;
        }

        .hiw-card-icon {
            width: 4rem;
            height: 4rem;
        }
    }
</style>
</div>
