<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(array $section = [], array $items = [], array $company = [], array $social = [], array $services = [], bool $isWhiteBackground = false, ?int $paddingTop = null, ?int $paddingBottom = null)
    {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->items = !empty($items) ? $items : [['step' => '01', 'title' => 'Schedule Pickup', 'description' => 'Book online or call us.', 'icon' => 'mdi:calendar-clock'], ['step' => '02', 'title' => 'We Collect', 'description' => 'Our driver arrives at your doorstep.', 'icon' => 'heroicons:truck'], ['step' => '03', 'title' => 'Expert Cleaning', 'description' => 'Your clothes are cleaned with care.', 'icon' => 'heroicons:sparkles'], ['step' => '04', 'title' => 'Delivery', 'description' => 'Fresh, clean laundry delivered back to you.', 'icon' => 'heroicons:home']];

        $this->section = array_merge(
            [
                'badge' => 'How It Works',
                'title' => 'Simple 4-Step Process',
                'subtitle' => 'Getting your laundry done has never been easier.',
            ],
            $section,
        );
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="how-it-works" class="section {{ $isWhiteBackground ? 'bg-white dark:bg-gray-800' : 'bg-gray-50 dark:bg-gray-900' }} overflow-hidden" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {{-- Section Header - Right Aligned --}}
        <div class="hiw-horizontal-header">
            <div class="hiw-horizontal-header-content">
                <span class="badge badge-primary mb-4">{{ $section['badge'] }}</span>
                <h2 class="section-title">{{ $section['title'] }}</h2>
                <p class="section-subtitle">{{ $section['subtitle'] }}</p>
            </div>
        </div>

        {{-- Horizontal Steps with Progress Line --}}
        <div class="hiw-horizontal-wrapper">
            {{-- Progress Line --}}
            <div class="hiw-horizontal-progress">
                <div class="hiw-horizontal-progress-line"></div>
                <div class="hiw-horizontal-progress-fill"></div>
            </div>

            {{-- Steps --}}
            <div class="hiw-horizontal-steps">
                @foreach ($items as $index => $step)
                    <div class="scroll-animate hiw-horizontal-step" style="--step-index: {{ $index }}">
                        {{-- Step Marker --}}
                        <div class="hiw-horizontal-marker">
                            <div class="hiw-horizontal-marker-inner">
                                <span>{{ $step['step'] }}</span>
                            </div>
                            <div class="hiw-horizontal-marker-ring"></div>
                        </div>

                        {{-- Step Card --}}
                        <div class="hiw-horizontal-card">
                            <div class="hiw-horizontal-icon">
                                <x-icon icon="{{ $step['icon'] ?? 'heroicons:home' }}" class="w-7 h-7" />
                            </div>
                            <h3 class="hiw-horizontal-title">{{ $step['title'] }}</h3>
                            <p class="hiw-horizontal-desc">{{ $step['description'] }}</p>
                        </div>
                    </div>
                @endforeach
            </div>
        </div>
    </div>
</section>

<style>
/* Header - Right Aligned */
.hiw-horizontal-header {
    display: flex;
    justify-content: flex-end;
    margin-bottom: 4rem;
}

[dir="rtl"] .hiw-horizontal-header {
    justify-content: flex-start;
}

.hiw-horizontal-header-content {
    text-align: right;
    max-width: 500px;
}

[dir="rtl"] .hiw-horizontal-header-content {
    text-align: left;
}

.hiw-horizontal-header .section-subtitle {
    margin: 0;
    margin-left: auto;
}

[dir="rtl"] .hiw-horizontal-header .section-subtitle {
    margin-left: 0;
    margin-right: auto;
}

/* Wrapper */
.hiw-horizontal-wrapper {
    position: relative;
    padding-top: 2rem;
}

/* Progress Line */
.hiw-horizontal-progress {
    position: absolute;
    top: 3.5rem;
    left: 5%;
    right: 5%;
    height: 4px;
}

.hiw-horizontal-progress-line {
    position: absolute;
    inset: 0;
    background: var(--color-gray-200);
    border-radius: 2px;
}

.dark .hiw-horizontal-progress-line {
    background: var(--color-gray-700);
}

.hiw-horizontal-progress-fill {
    position: absolute;
    top: 0;
    left: 0;
    height: 100%;
    width: 0;
    background: linear-gradient(90deg, var(--color-primary-500), var(--color-secondary-500));
    border-radius: 2px;
    transition: width 1.5s ease-out;
}

[dir="rtl"] .hiw-horizontal-progress-fill {
    left: auto;
    right: 0;
    background: linear-gradient(270deg, var(--color-primary-500), var(--color-secondary-500));
}

.hiw-horizontal-wrapper.animate .hiw-horizontal-progress-fill {
    width: 100%;
}

/* Steps Container */
.hiw-horizontal-steps {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 1.5rem;
}

/* Single Step */
.hiw-horizontal-step {
    display: flex;
    flex-direction: column;
    align-items: center;
    text-align: center;
}

/* Marker */
.hiw-horizontal-marker {
    position: relative;
    margin-bottom: 1.5rem;
    z-index: 1;
}

.hiw-horizontal-marker-inner {
    width: 3rem;
    height: 3rem;
    background: linear-gradient(135deg, var(--color-primary-600), var(--color-primary-700));
    color: #fff;
    font-size: 1rem;
    font-weight: 700;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    z-index: 2;
    box-shadow: 0 4px 15px rgba(37, 99, 235, 0.3);
    transition: all 0.3s ease;
}

.hiw-horizontal-step:hover .hiw-horizontal-marker-inner {
    transform: scale(1.1);
    box-shadow: 0 6px 20px rgba(37, 99, 235, 0.4);
}

.hiw-horizontal-marker-ring {
    position: absolute;
    inset: -6px;
    border: 2px solid var(--color-primary-300);
    border-radius: 50%;
    opacity: 0;
    transition: all 0.3s ease;
}

.dark .hiw-horizontal-marker-ring {
    border-color: var(--color-primary-600);
}

.hiw-horizontal-step:hover .hiw-horizontal-marker-ring {
    opacity: 1;
    animation: marker-pulse 1.5s ease-out infinite;
}

@keyframes marker-pulse {
    0% { transform: scale(1); opacity: 0.6; }
    100% { transform: scale(1.3); opacity: 0; }
}

/* Card */
.hiw-horizontal-card {
    background: var(--color-card-bg, #fff);
    border-radius: 1rem;
    padding: 1.5rem;
    width: 100%;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
    border: 1px solid transparent;
    transition: all 0.3s ease;
}

.dark .hiw-horizontal-card {
    background: var(--color-bg-secondary);
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
}

.hiw-horizontal-step:hover .hiw-horizontal-card {
    transform: translateY(-5px);
    box-shadow: 0 12px 24px rgba(0, 0, 0, 0.1);
    border-color: var(--color-primary-200);
}

.dark .hiw-horizontal-step:hover .hiw-horizontal-card {
    box-shadow: 0 12px 24px rgba(0, 0, 0, 0.3);
    border-color: var(--color-primary-700);
}

/* Icon */
.hiw-horizontal-icon {
    width: 3.5rem;
    height: 3.5rem;
    background: linear-gradient(135deg, var(--color-primary-50), var(--color-secondary-50));
    border-radius: 0.75rem;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 1rem;
    transition: all 0.3s ease;
}

.dark .hiw-horizontal-icon {
    background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
}

.hiw-horizontal-icon svg {
    color: var(--color-primary-600);
    fill: var(--color-primary-600);
    transition: all 0.3s ease;
}

.dark .hiw-horizontal-icon svg {
    color: var(--color-primary-400);
    fill: var(--color-primary-400);
}

.hiw-horizontal-step:hover .hiw-horizontal-icon {
    background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-500));
}

.hiw-horizontal-step:hover .hiw-horizontal-icon svg {
    color: #fff !important;
    fill: #fff !important;
}

/* Title & Description */
.hiw-horizontal-title {
    font-size: 1.125rem;
    font-weight: 600;
    color: var(--color-gray-900);
    margin-bottom: 0.5rem;
}

.dark .hiw-horizontal-title {
    color: var(--color-text-primary);
}

.hiw-horizontal-desc {
    font-size: 0.875rem;
    color: var(--color-gray-600);
    line-height: 1.5;
}

.dark .hiw-horizontal-desc {
    color: var(--color-text-tertiary);
}

/* Mobile Responsive */
@media (max-width: 1024px) {
    .hiw-horizontal-steps {
        grid-template-columns: repeat(2, 1fr);
    }

    .hiw-horizontal-progress {
        display: none;
    }
}

@media (max-width: 768px) {
    .hiw-horizontal-header {
        justify-content: center;
    }

    [dir="rtl"] .hiw-horizontal-header {
        justify-content: center;
    }

    .hiw-horizontal-header-content {
        text-align: center;
    }

    [dir="rtl"] .hiw-horizontal-header-content {
        text-align: center;
    }

    .hiw-horizontal-header .section-subtitle {
        margin: 0 auto;
    }
}

@media (max-width: 640px) {
    .hiw-horizontal-steps {
        grid-template-columns: 1fr;
        max-width: 320px;
        margin: 0 auto;
    }
}
</style>
</div>

@script
<script>
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate');
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.3 });

    const wrapper = document.querySelector('.hiw-horizontal-wrapper');
    if (wrapper) {
        observer.observe(wrapper);
    }
</script>
@endscript

</div>
