<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(array $section = [], array $items = [], array $company = [], array $social = [], array $services = [], bool $isWhiteBackground = false, ?int $paddingTop = null, ?int $paddingBottom = null)
    {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Use items from standardized props
        $this->items = !empty($items) ? $items : [['step' => '01', 'title' => 'Schedule Pickup', 'description' => 'Book online or call us.', 'icon' => 'mdi:calendar-clock'], ['step' => '02', 'title' => 'We Collect', 'description' => 'Our driver arrives at your doorstep.', 'icon' => 'heroicons:truck'], ['step' => '03', 'title' => 'Expert Cleaning', 'description' => 'Your clothes are cleaned with care.', 'icon' => 'heroicons:sparkles'], ['step' => '04', 'title' => 'Delivery', 'description' => 'Fresh, clean laundry delivered back to you.', 'icon' => 'heroicons:home']];

        $this->section = array_merge(
            [
                'badge' => 'How It Works',
                'title' => 'Simple 4-Step Process',
                'subtitle' => 'Getting your laundry done has never been easier.',
            ],
            $section,
        );
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="how-it-works" class="section {{ $isWhiteBackground ? 'bg-white dark:bg-gray-800' : 'bg-gray-50 dark:bg-gray-900' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        {{-- Section Header --}}
        <div class="text-center mb-20">
            <span class="badge badge-primary mb-4">{{ $section['badge'] }}</span>
            <h2 class="section-title">{{ $section['title'] }}</h2>
            <p class="section-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Minimal Steps --}}
        <div class="hiw-minimal">
            @foreach ($items as $index => $step)
                <div class="scroll-animate hiw-minimal-step">
                    {{-- Left: Number --}}
                    <div class="hiw-minimal-number">
                        <span class="hiw-minimal-num-text">{{ $step['step'] }}</span>
                    </div>

                    {{-- Center: Line --}}
                    <div class="hiw-minimal-divider">
                        <div class="hiw-minimal-line"></div>
                        <div class="hiw-minimal-dot"></div>
                        @if (!$loop->last)
                            <div class="hiw-minimal-connector"></div>
                        @endif
                    </div>

                    {{-- Right: Content --}}
                    <div class="hiw-minimal-content">
                        {{-- Icon + Title Row --}}
                        <div class="hiw-minimal-header">
                            <div class="hiw-minimal-icon">
                                <x-icon icon="{{ $step['icon'] ?? 'heroicons:home' }}" class="hiw-minimal-icon-svg w-5 h-5" />
                            </div>
                            <h3 class="hiw-minimal-title">{{ $step['title'] }}</h3>
                        </div>

                        {{-- Description --}}
                        <p class="hiw-minimal-desc">{{ $step['description'] }}</p>
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
/* Minimal Container */
.hiw-minimal {
    display: flex;
    flex-direction: column;
    gap: 0;
    width: 100%;
    max-width: 550px;
    margin-left: auto;
    margin-right: auto;
}

/* Step Row */
.hiw-minimal-step {
    display: grid;
    grid-template-columns: 80px 60px 1fr;
    gap: 0;
    min-height: 120px;
    align-items: flex-start;
}

[dir="rtl"] .hiw-minimal-step {
    grid-template-columns: 1fr 60px 80px;
}

/* Number Section */
.hiw-minimal-number {
    display: flex;
    align-items: flex-start;
    justify-content: flex-end;
    padding-top: 0.5rem;
    padding-right: 1rem;
}

[dir="rtl"] .hiw-minimal-number {
    padding-right: 0;
    padding-left: 1rem;
    justify-content: flex-start;
    order: 3;
}

.hiw-minimal-num-text {
    font-size: 2.5rem;
    font-weight: 800;
    background: linear-gradient(135deg, var(--color-primary-400), var(--color-secondary-400));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    line-height: 1;
    transition: all 0.3s ease;
}

.hiw-minimal-step:hover .hiw-minimal-num-text {
    background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-500));
    -webkit-background-clip: text;
    background-clip: text;
    transform: scale(1.1);
}

/* Divider Section */
.hiw-minimal-divider {
    display: flex;
    flex-direction: column;
    align-items: center;
    position: relative;
    padding-top: 0.75rem;
}

[dir="rtl"] .hiw-minimal-divider {
    order: 2;
}

/* Dot */
.hiw-minimal-dot {
    width: 14px;
    height: 14px;
    background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
    border-radius: 50%;
    position: relative;
    z-index: 1;
    transition: all 0.3s ease;
}

.hiw-minimal-step:hover .hiw-minimal-dot {
    transform: scale(1.3);
    box-shadow: 0 0 20px rgba(37, 99, 235, 0.4);
}

/* Line */
.hiw-minimal-line {
    width: 2px;
    height: 10px;
    background: var(--color-gray-200);
}

.dark .hiw-minimal-line {
    background: var(--color-gray-700);
}

/* Connector */
.hiw-minimal-connector {
    width: 2px;
    flex: 1;
    min-height: 80px;
    background: linear-gradient(180deg, var(--color-primary-300), var(--color-secondary-300));
}

.dark .hiw-minimal-connector {
    background: linear-gradient(180deg, var(--color-primary-700), var(--color-secondary-700));
}

/* Content Section */
.hiw-minimal-content {
    padding-top: 0.25rem;
    padding-left: 1.5rem;
    padding-bottom: 2rem;
}

[dir="rtl"] .hiw-minimal-content {
    padding-left: 0;
    padding-right: 1.5rem;
    order: 1;
}

/* Header Row */
.hiw-minimal-header {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 0.5rem;
}

[dir="rtl"] .hiw-minimal-header {
    flex-direction: row-reverse;
}

/* Icon */
.hiw-minimal-icon {
    width: 2.5rem;
    height: 2.5rem;
    background: var(--color-primary-50);
    border-radius: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
}

.dark .hiw-minimal-icon {
    background: var(--color-primary-900);
}

.hiw-minimal-step:hover .hiw-minimal-icon {
    background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-500));
}

/* Icon SVG colors */
.hiw-minimal-icon-svg,
.hiw-minimal-icon-svg svg {
    color: var(--color-primary-600);
    fill: var(--color-primary-600);
    transition: color 0.3s ease, fill 0.3s ease;
}

.dark .hiw-minimal-icon-svg,
.dark .hiw-minimal-icon-svg svg {
    color: var(--color-primary-400);
    fill: var(--color-primary-400);
}

.hiw-minimal-step:hover .hiw-minimal-icon-svg,
.hiw-minimal-step:hover .hiw-minimal-icon-svg svg,
.hiw-minimal-step:hover .hiw-minimal-icon svg {
    color: #fff !important;
    fill: #fff !important;
}

/* Title */
.hiw-minimal-title {
    font-size: 1.25rem;
    font-weight: 700;
    color: var(--color-gray-900);
    transition: color 0.3s ease;
}

.dark .hiw-minimal-title {
    color: var(--color-text-primary);
}

.hiw-minimal-step:hover .hiw-minimal-title {
    color: var(--color-primary-600);
}

.dark .hiw-minimal-step:hover .hiw-minimal-title {
    color: var(--color-primary-400);
}

/* Description */
.hiw-minimal-desc {
    font-size: 1rem;
    color: var(--color-gray-600);
    line-height: 1.6;
    max-width: 400px;
}

.dark .hiw-minimal-desc {
    color: var(--color-text-tertiary);
}

/* Mobile Responsive */
@media (max-width: 640px) {
    .hiw-minimal-step {
        grid-template-columns: 50px 40px 1fr;
    }

    [dir="rtl"] .hiw-minimal-step {
        grid-template-columns: 1fr 40px 50px;
    }

    .hiw-minimal-num-text {
        font-size: 1.75rem;
    }

    .hiw-minimal-number {
        padding-right: 0.5rem;
    }

    [dir="rtl"] .hiw-minimal-number {
        padding-left: 0.5rem;
    }

    .hiw-minimal-content {
        padding-left: 1rem;
    }

    [dir="rtl"] .hiw-minimal-content {
        padding-right: 1rem;
    }

    .hiw-minimal-title {
        font-size: 1.125rem;
    }

    .hiw-minimal-desc {
        font-size: 0.875rem;
    }
}
</style>
</div>
