<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(array $section = [], array $items = [], array $company = [], array $social = [], array $services = [], bool $isWhiteBackground = false, ?int $paddingTop = null, ?int $paddingBottom = null)
    {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Use items from standardized props
        $this->items = !empty($items) ? $items : [['step' => '01', 'title' => 'Schedule Pickup', 'description' => 'Book online or call us.', 'icon' => 'mdi:calendar-clock'], ['step' => '02', 'title' => 'We Collect', 'description' => 'Our driver arrives at your doorstep.', 'icon' => 'heroicons:truck'], ['step' => '03', 'title' => 'Expert Cleaning', 'description' => 'Your clothes are cleaned with care.', 'icon' => 'heroicons:sparkles'], ['step' => '04', 'title' => 'Delivery', 'description' => 'Fresh, clean laundry delivered back to you.', 'icon' => 'heroicons:home']];

        $this->section = array_merge(
            [
                'badge' => 'How It Works',
                'title' => 'Simple 4-Step Process',
                'subtitle' => 'Getting your laundry done has never been easier.',
            ],
            $section,
        );
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="how-it-works" class="section {{ $isWhiteBackground ? 'bg-white dark:bg-gray-800' : 'bg-gradient-to-b from-white to-gray-50 dark:from-gray-900 dark:to-gray-800' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {{-- Section Header --}}
        <div class="text-center mb-16">
            <span class="badge badge-primary mb-4">{{ $section['badge'] }}</span>
            <h2 class="section-title">{{ $section['title'] }}</h2>
            <p class="section-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Zigzag Steps --}}
        <div class="hiw-zigzag">
            @foreach ($items as $index => $step)
                <div class="scroll-animate hiw-zigzag-row {{ $index % 2 === 0 ? '' : 'hiw-zigzag-reverse' }}">
                    {{-- Content Side --}}
                    <div class="hiw-zigzag-content">
                        <div class="hiw-zigzag-box">
                            {{-- Step Badge --}}
                            <div class="hiw-zigzag-step">
                                Step {{ $step['step'] }}
                            </div>

                            <h3 class="text-2xl md:text-3xl font-bold text-gray-900 dark:text-white mb-4">
                                {{ $step['title'] }}
                            </h3>
                            <p class="text-gray-600 dark:text-gray-400 text-lg leading-relaxed">
                                {{ $step['description'] }}
                            </p>

                            {{-- Feature dots --}}
                            <div class="flex items-center gap-3 mt-6">
                                <div class="w-2 h-2 rounded-full bg-primary-500"></div>
                                <div class="w-2 h-2 rounded-full bg-secondary-500"></div>
                                <div class="w-2 h-2 rounded-full bg-primary-300"></div>
                            </div>
                        </div>
                    </div>

                    {{-- Icon Side --}}
                    <div class="hiw-zigzag-visual">
                        <div class="hiw-zigzag-icon-container">
                            {{-- Background Shape --}}
                            <div class="hiw-zigzag-shape"></div>

                            {{-- Icon Circle --}}
                            <div class="hiw-zigzag-icon">
                                <x-icon icon="{{ $step['icon'] ?? 'heroicons:home' }}" class="w-12 h-12 text-white" />
                            </div>

                            {{-- Step Number --}}
                            <div class="hiw-zigzag-number">
                                {{ $step['step'] }}
                            </div>
                        </div>
                    </div>

                    {{-- Connector Line --}}
                    @if (!$loop->last)
                        <div class="hiw-zigzag-connector {{ $index % 2 === 0 ? 'hiw-connector-right' : 'hiw-connector-left' }}">
                            <svg viewBox="0 0 100 50" class="hiw-connector-svg">
                                <path d="M0 25 Q50 {{ $index % 2 === 0 ? '0' : '50' }} 100 25" fill="none" stroke="url(#connector-gradient-{{ $index }})" stroke-width="2" stroke-dasharray="6 4"/>
                                <defs>
                                    <linearGradient id="connector-gradient-{{ $index }}" x1="0%" y1="0%" x2="100%" y2="0%">
                                        <stop offset="0%" style="stop-color: var(--color-primary-400)"/>
                                        <stop offset="100%" style="stop-color: var(--color-secondary-400)"/>
                                    </linearGradient>
                                </defs>
                            </svg>
                        </div>
                    @endif
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
/* Zigzag Container */
.hiw-zigzag {
    display: flex;
    flex-direction: column;
    gap: 4rem;
}

/* Zigzag Row */
.hiw-zigzag-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 4rem;
    align-items: center;
    position: relative;
}

.hiw-zigzag-reverse {
    direction: rtl;
}

.hiw-zigzag-reverse > * {
    direction: ltr;
}

/* RTL Support */
[dir="rtl"] .hiw-zigzag-row {
    direction: ltr;
}

[dir="rtl"] .hiw-zigzag-reverse {
    direction: rtl;
}

/* Content Side */
.hiw-zigzag-content {
    display: flex;
    justify-content: center;
}

.hiw-zigzag-box {
    max-width: 400px;
}

/* Step Badge */
.hiw-zigzag-step {
    display: inline-flex;
    padding: 0.5rem 1rem;
    background: linear-gradient(135deg, var(--color-primary-100), var(--color-secondary-100));
    color: var(--color-primary-700);
    font-size: 0.875rem;
    font-weight: 600;
    border-radius: 9999px;
    margin-bottom: 1rem;
}

.dark .hiw-zigzag-step {
    background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
    color: var(--color-primary-300);
}

/* Visual Side */
.hiw-zigzag-visual {
    display: flex;
    justify-content: center;
    align-items: center;
}

.hiw-zigzag-icon-container {
    position: relative;
    width: 12rem;
    height: 12rem;
}

/* Background Shape */
.hiw-zigzag-shape {
    position: absolute;
    inset: 0;
    background: linear-gradient(135deg, var(--color-primary-100), var(--color-secondary-100));
    border-radius: 2rem;
    transform: rotate(6deg);
    transition: all 0.4s ease;
}

.dark .hiw-zigzag-shape {
    background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
}

.hiw-zigzag-row:hover .hiw-zigzag-shape {
    transform: rotate(12deg) scale(1.05);
}

/* Icon Circle */
.hiw-zigzag-icon {
    position: absolute;
    inset: 1.5rem;
    background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-500));
    border-radius: 1.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 10px 40px -10px rgba(37, 99, 235, 0.5);
    transition: all 0.4s ease;
}

.hiw-zigzag-row:hover .hiw-zigzag-icon {
    transform: scale(1.05);
    box-shadow: 0 15px 50px -10px rgba(37, 99, 235, 0.6);
}

/* Icon SVG - always white on gradient background */
.hiw-zigzag-icon svg {
    color: #fff !important;
    fill: #fff !important;
}

/* Step Number */
.hiw-zigzag-number {
    position: absolute;
    bottom: -0.5rem;
    right: -0.5rem;
    width: 3rem;
    height: 3rem;
    background: #fff;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    font-weight: 800;
    color: var(--color-primary-600);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
}

.dark .hiw-zigzag-number {
    background: var(--color-bg-tertiary);
    color: var(--color-primary-400);
}

[dir="rtl"] .hiw-zigzag-number {
    right: auto;
    left: -0.5rem;
}

/* Connector Line */
.hiw-zigzag-connector {
    position: absolute;
    bottom: -3rem;
    width: 50%;
    height: 50px;
    opacity: 0.6;
}

.hiw-connector-right {
    right: 0;
}

.hiw-connector-left {
    left: 0;
}

[dir="rtl"] .hiw-connector-right {
    right: auto;
    left: 0;
}

[dir="rtl"] .hiw-connector-left {
    left: auto;
    right: 0;
}

.hiw-connector-svg {
    width: 100%;
    height: 100%;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .hiw-zigzag-row,
    .hiw-zigzag-reverse {
        grid-template-columns: 1fr;
        gap: 2rem;
        direction: ltr;
    }

    [dir="rtl"] .hiw-zigzag-row,
    [dir="rtl"] .hiw-zigzag-reverse {
        direction: rtl;
    }

    .hiw-zigzag-visual {
        order: -1;
    }

    .hiw-zigzag-icon-container {
        width: 10rem;
        height: 10rem;
    }

    .hiw-zigzag-box {
        text-align: center;
    }

    .hiw-zigzag-box .flex {
        justify-content: center;
    }

    .hiw-zigzag-connector {
        display: none;
    }
}
</style>
</div>
