<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $defaultItems = [
            [
                'name' => 'Basic',
                'price' => '29',
                'period' => 'per month',
                'description' => 'Perfect for individuals',
                'features' => [
                    ['icon' => 'lucide:scale', 'text' => '15 lbs of laundry', 'is_included' => true],
                    ['icon' => 'lucide:sparkles', 'text' => 'Wash & fold service', 'is_included' => true],
                    ['icon' => 'lucide:truck', 'text' => 'Free pickup & delivery', 'is_included' => true],
                    ['icon' => 'lucide:clock', 'text' => '48-hour turnaround', 'is_included' => true],
                    ['icon' => 'lucide:flask-conical', 'text' => 'Standard detergents', 'is_included' => true],
                    ['icon' => 'lucide:leaf', 'text' => 'Premium eco detergents', 'is_included' => false],
                    ['icon' => 'lucide:zap', 'text' => 'Same-day turnaround', 'is_included' => false],
                ],
                'popular' => false,
            ],
            [
                'name' => 'Family',
                'price' => '79',
                'period' => 'per month',
                'description' => 'Best for families',
                'features' => [
                    ['icon' => 'lucide:scale', 'text' => '50 lbs of laundry', 'is_included' => true],
                    ['icon' => 'lucide:sparkles', 'text' => 'Wash, fold & iron', 'is_included' => true],
                    ['icon' => 'lucide:truck', 'text' => 'Free pickup & delivery', 'is_included' => true],
                    ['icon' => 'lucide:clock', 'text' => '24-hour turnaround', 'is_included' => true],
                    ['icon' => 'lucide:leaf', 'text' => 'Premium eco detergents', 'is_included' => true],
                    ['icon' => 'lucide:scissors', 'text' => '2 dry clean items/month', 'is_included' => true],
                    ['icon' => 'lucide:zap', 'text' => 'Same-day turnaround', 'is_included' => false],
                ],
                'popular' => true,
            ],
            [
                'name' => 'Business',
                'price' => '199',
                'period' => 'per month',
                'description' => 'For businesses & hotels',
                'features' => [
                    ['icon' => 'lucide:infinity', 'text' => 'Unlimited laundry', 'is_included' => true],
                    ['icon' => 'lucide:badge-check', 'text' => 'All services included', 'is_included' => true],
                    ['icon' => 'lucide:calendar-days', 'text' => 'Daily pickup available', 'is_included' => true],
                    ['icon' => 'lucide:zap', 'text' => 'Same-day turnaround', 'is_included' => true],
                    ['icon' => 'lucide:user-circle', 'text' => 'Dedicated account manager', 'is_included' => true],
                    ['icon' => 'lucide:phone', 'text' => 'Priority support', 'is_included' => true],
                    ['icon' => 'lucide:paintbrush', 'text' => 'Custom branding', 'is_included' => true],
                ],
                'popular' => false,
            ],
        ];

        if (!empty($items)) {
            $this->items = array_map(function ($item) {
                if (isset($item['features']) && is_array($item['features'])) {
                    $item['features'] = array_map(function ($feature) {
                        if (is_string($feature)) {
                            return ['icon' => '', 'text' => $feature, 'is_included' => true];
                        }
                        return [
                            'icon' => $feature['icon'] ?? '',
                            'text' => $feature['text'] ?? '',
                            'is_included' => $feature['is_included'] ?? true,
                        ];
                    }, $item['features']);
                }
                return $item;
            }, $items);
        } else {
            $this->items = $defaultItems;
        }

        $this->section = array_merge([
            'badge' => 'Pricing Plans',
            'title' => 'Choose Your Perfect Plan',
            'subtitle' => 'Flexible plans designed to fit your lifestyle.',
            'currency_symbol' => '$',
            'popular_badge' => 'Most Popular',
            'cta_button' => 'Get Started',
            'cta_button_url' => '#contact',
        ], $section);
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section class="pricing-modern-section {{ $isWhiteBackground ? 'pricing-modern-white' : 'pricing-modern-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background Shapes --}}
    <div class="pricing-modern-bg">
        <div class="pricing-modern-shape pricing-modern-shape-1"></div>
        <div class="pricing-modern-shape pricing-modern-shape-2"></div>
        <div class="pricing-modern-circle"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="text-center mb-16">
            <span class="pricing-modern-badge">{{ $section['badge'] }}</span>
            <h2 class="pricing-modern-title">{{ $section['title'] }}</h2>
            <p class="pricing-modern-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Pricing Cards --}}
        <div class="pricing-modern-grid">
            @foreach ($items as $index => $plan)
                <div class="pricing-modern-card scroll-animate {{ $plan['popular'] ? 'pricing-modern-card-popular' : '' }}" style="animation-delay: {{ $index * 100 }}ms">
                    @if ($plan['popular'])
                        <div class="pricing-modern-popular-badge">
                            <x-icon icon="lucide:star" class="w-3 h-3" />
                            <span>{{ $section['popular_badge'] }}</span>
                        </div>
                    @endif

                    {{-- Card Header --}}
                    <div class="pricing-modern-header">
                        <div class="pricing-modern-icon">
                            <x-icon icon="{{ $index === 0 ? 'lucide:user' : ($index === 1 ? 'lucide:users' : 'lucide:building-2') }}" class="w-8 h-8" />
                        </div>
                        <h3 class="pricing-modern-plan-name">{{ $plan['name'] }}</h3>
                        <p class="pricing-modern-plan-desc">{{ $plan['description'] }}</p>
                    </div>

                    {{-- Price --}}
                    <div class="pricing-modern-price-container">
                        <div class="pricing-modern-price-wrap">
                            <span class="pricing-modern-currency">{{ $section['currency_symbol'] ?? '$' }}</span>
                            <span class="pricing-modern-price">{{ $plan['price'] }}</span>
                        </div>
                        <span class="pricing-modern-period">/{{ $plan['period'] }}</span>
                    </div>

                    {{-- Features --}}
                    <ul class="pricing-modern-features">
                        @foreach ($plan['features'] as $feature)
                            @php
                                $isIncluded = $feature['is_included'] ?? true;
                            @endphp
                            <li class="pricing-modern-feature {{ !$isIncluded ? 'pricing-modern-feature-excluded' : '' }}">
                                <span class="pricing-modern-feature-icon {{ !$isIncluded ? 'pricing-modern-feature-icon-excluded' : '' }}">
                                    @if (!empty($feature['icon']))
                                        <x-icon icon="{{ $feature['icon'] }}" class="w-4 h-4" />
                                    @else
                                        <x-icon icon="{{ $isIncluded ? 'lucide:check' : 'lucide:x' }}" class="w-4 h-4" />
                                    @endif
                                </span>
                                <span class="pricing-modern-feature-text">{{ $feature['text'] }}</span>
                            </li>
                        @endforeach
                    </ul>

                    {{-- CTA Button --}}
                    <a href="{{ $section['cta_button_url'] ?? '#contact' }}" class="pricing-modern-button">
                        <span>{{ $section['cta_button'] }}</span>
                        <x-icon icon="lucide:arrow-right" class="w-4 h-4" />
                    </a>

                    {{-- Decorative Element --}}
                    <div class="pricing-modern-decoration"></div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Pricing Modern Section ===== */
    .pricing-modern-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .pricing-modern-white {
        background: #ffffff;
    }

    .pricing-modern-gray {
        background: linear-gradient(to bottom, #f8fafc, #ffffff);
    }

    .dark .pricing-modern-white {
        background: var(--color-bg-secondary);
    }

    .dark .pricing-modern-gray {
        background: linear-gradient(to bottom, var(--color-bg-primary), var(--color-bg-secondary));
    }

    /* ===== Background ===== */
    .pricing-modern-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .pricing-modern-shape {
        position: absolute;
        width: 400px;
        height: 400px;
        background: linear-gradient(135deg, var(--color-primary-200), var(--color-secondary-200));
        opacity: 0.15;
    }

    .pricing-modern-shape-1 {
        top: -100px;
        left: -100px;
        border-radius: 50% 0 50% 0;
        transform: rotate(45deg);
    }

    .pricing-modern-shape-2 {
        bottom: -150px;
        right: -150px;
        border-radius: 0 50% 0 50%;
        transform: rotate(-45deg);
    }

    .pricing-modern-circle {
        position: absolute;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        width: 600px;
        height: 600px;
        border: 2px solid var(--color-primary-200);
        border-radius: 50%;
        opacity: 0.1;
    }

    .dark .pricing-modern-shape {
        background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
        opacity: 0.1;
    }

    .dark .pricing-modern-circle {
        border-color: var(--color-primary-800);
    }

    /* ===== Header ===== */
    .pricing-modern-badge {
        display: inline-block;
        padding: 0.5rem 1.25rem;
        background: var(--color-primary-50);
        color: var(--color-primary-600);
        font-size: 0.875rem;
        font-weight: 600;
        border-radius: 9999px;
        margin-bottom: 1rem;
    }

    .dark .pricing-modern-badge {
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
        color: var(--color-primary-400);
    }

    .pricing-modern-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .pricing-modern-title {
        color: var(--color-text-primary);
    }

    .pricing-modern-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
    }

    .dark .pricing-modern-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== Grid ===== */
    .pricing-modern-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
        gap: 2rem;
        max-width: 1200px;
        margin: 0 auto;
    }

    /* ===== Card ===== */
    .pricing-modern-card {
        position: relative;
        background: white;
        border: 1px solid var(--color-gray-200);
        border-radius: 1.5rem;
        padding: 2.5rem 2rem;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        overflow: visible;
    }

    .pricing-modern-card:hover {
        transform: translateY(-10px);
        border-color: var(--color-primary-300);
        box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.15);
    }

    .dark .pricing-modern-card {
        background: var(--color-card-bg);
        border-color: var(--color-border);
    }

    .dark .pricing-modern-card:hover {
        border-color: var(--color-primary-700);
        box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
    }

    /* ===== Popular Card ===== */
    .pricing-modern-card-popular {
        border-color: var(--color-primary-500);
        box-shadow: 0 20px 40px -15px rgba(var(--color-primary-rgb, 37, 99, 235), 0.3);
    }

    .pricing-modern-card-popular::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, var(--color-primary-500), var(--color-secondary-500));
    }

    .pricing-modern-popular-badge {
        position: absolute;
        top: -12px;
        right: 2rem;
        display: flex;
        align-items: center;
        gap: 0.375rem;
        padding: 0.375rem 1rem;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        color: white;
        font-size: 0.75rem;
        font-weight: 700;
        border-radius: 9999px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        z-index: 10;
    }

    /* ===== Header ===== */
    .pricing-modern-header {
        text-align: center;
        margin-bottom: 2rem;
    }

    .pricing-modern-icon {
        width: 4rem;
        height: 4rem;
        margin: 0 auto 1rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        color: white;
        border-radius: 1rem;
        transition: all 0.4s ease;
    }

    .pricing-modern-card:hover .pricing-modern-icon {
        transform: rotate(10deg) scale(1.1);
    }

    .pricing-modern-plan-name {
        font-family: var(--font-heading);
        font-size: 1.5rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.5rem;
    }

    .dark .pricing-modern-plan-name {
        color: var(--color-text-primary);
    }

    .pricing-modern-plan-desc {
        font-size: 0.875rem;
        color: var(--color-gray-600);
    }

    .dark .pricing-modern-plan-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Price ===== */
    .pricing-modern-price-container {
        text-align: center;
        margin-bottom: 2rem;
        padding: 1.5rem 0;
        border-top: 1px solid var(--color-gray-200);
        border-bottom: 1px solid var(--color-gray-200);
    }

    .dark .pricing-modern-price-container {
        border-color: var(--color-border);
    }

    .pricing-modern-price-wrap {
        display: flex;
        align-items: baseline;
        justify-content: center;
        margin-bottom: 0.25rem;
    }

    .pricing-modern-currency {
        font-size: 1.5rem;
        font-weight: 700;
        color: var(--color-gray-600);
    }

    .dark .pricing-modern-currency {
        color: var(--color-text-secondary);
    }

    .pricing-modern-price {
        font-family: var(--font-heading);
        font-size: 3rem;
        font-weight: 800;
        color: var(--color-gray-900);
        line-height: 1;
    }

    .dark .pricing-modern-price {
        color: var(--color-text-primary);
    }

    .pricing-modern-period {
        font-size: 0.875rem;
        color: var(--color-gray-600);
    }

    .dark .pricing-modern-period {
        color: var(--color-text-tertiary);
    }

    /* ===== Features ===== */
    .pricing-modern-features {
        list-style: none;
        padding: 0;
        margin: 0 0 2rem 0;
    }

    .pricing-modern-feature {
        display: flex;
        align-items: flex-start;
        gap: 0.75rem;
        padding: 0.75rem 0;
        border-bottom: 1px solid var(--color-gray-100);
    }

    .dark .pricing-modern-feature {
        border-color: var(--color-border);
    }

    .pricing-modern-feature:last-child {
        border-bottom: none;
    }

    .pricing-modern-feature-icon {
        flex-shrink: 0;
        width: 1.25rem;
        height: 1.25rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: var(--color-primary-50);
        color: var(--color-primary-600);
        border-radius: 0.375rem;
    }

    .dark .pricing-modern-feature-icon {
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
        color: var(--color-primary-400);
    }

    .pricing-modern-feature-icon-excluded {
        background: var(--color-gray-100);
        color: var(--color-gray-400);
    }

    .dark .pricing-modern-feature-icon-excluded {
        background: var(--color-gray-700);
        color: var(--color-gray-300);
    }

    .pricing-modern-feature-text {
        font-size: 0.9375rem;
        color: var(--color-gray-700);
    }

    .dark .pricing-modern-feature-text {
        color: var(--color-text-secondary);
    }

    .pricing-modern-feature-excluded .pricing-modern-feature-text {
        color: var(--color-gray-400);
        text-decoration: line-through;
    }

    .dark .pricing-modern-feature-excluded .pricing-modern-feature-text {
        color: var(--color-gray-600);
    }

    /* ===== Button ===== */
    .pricing-modern-button {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 0.5rem;
        width: 100%;
        padding: 1rem;
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600));
        color: white;
        font-weight: 600;
        border-radius: 0.75rem;
        transition: all 0.3s ease;
    }

    .pricing-modern-button:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px -5px rgba(var(--color-primary-rgb, 37, 99, 235), 0.4);
    }

    /* ===== Decoration ===== */
    .pricing-modern-decoration {
        position: absolute;
        bottom: 0;
        right: 0;
        width: 150px;
        height: 150px;
        background: linear-gradient(135deg, var(--color-primary-100), var(--color-secondary-100));
        border-radius: 100% 0 0 0;
        opacity: 0;
        transition: opacity 0.4s ease;
    }

    .pricing-modern-card:hover .pricing-modern-decoration {
        opacity: 0.3;
    }

    .dark .pricing-modern-decoration {
        background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .pricing-modern-section {
        direction: rtl;
    }

    [dir="rtl"] .pricing-modern-popular-badge {
        right: auto;
        left: 2rem;
    }

    [dir="rtl"] .pricing-modern-feature {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .pricing-modern-button svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .pricing-modern-decoration {
        right: auto;
        left: 0;
        border-radius: 0 100% 0 0;
    }

    [dir="rtl"] .pricing-modern-shape-1 {
        left: auto;
        right: -100px;
    }

    [dir="rtl"] .pricing-modern-shape-2 {
        right: auto;
        left: -150px;
    }

    [dir="rtl"] .pricing-modern-card:hover .pricing-modern-icon {
        transform: rotate(-10deg) scale(1.1);
    }

    /* ===== Responsive ===== */
    @media (max-width: 768px) {
        .pricing-modern-section {
            padding: 4rem 0;
        }

        .pricing-modern-title {
            font-size: 2rem;
        }

        .pricing-modern-grid {
            grid-template-columns: 1fr;
            gap: 1.5rem;
        }

        .pricing-modern-card {
            padding: 2rem 1.5rem;
        }

        .pricing-modern-price {
            font-size: 2.5rem;
        }
    }
</style>
</div>
